/*
 * Copyright (C) 2000-2025 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 * xine engine error handling/notification
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <stdarg.h>

#include "dump.h"

#include "common.h"
#include "config_wrapper.h"
#include "actions.h"
#include "errors.h"
#include "viewlog.h"
#include "videowin.h"

/*
 * Callback used to display the viewlog window.
 */
static void _errors_display_log_3 (void *data, int state) {
  uint8_t *land = (uint8_t *)data;
  uint32_t i = *land;
  gGui_t *gui = data;

  land -= i;
  xitk_container (gui, land, no_messages.helipad[0]);

  pthread_mutex_lock (&gui->no_messages.mutex);
  gui->no_messages.msg_windows[i] = 0;
  pthread_mutex_unlock (&gui->no_messages.mutex);

  if (state == 2)
    viewlog_main (XUI_W_ON, gui);
}

#define XUI_MSG_TYPE_MASK 7
void gui_msg (gGui_t *gui, unsigned int flags, const char *message, ...) {
  va_list   args;
  char     *buf;
  const char *text;
  unsigned int type = flags & XUI_MSG_TYPE_MASK;

  va_start (args, message);
  if (!strcmp (message, "%s")) {
    buf = NULL;
    text = va_arg (args, const char *);
    va_end (args);
    if (!text)
      return;
    pthread_mutex_lock (&gui->no_messages.mutex);
    if ((flags == gui->no_messages.flags[XUI_MAX_SHOWN_MSGS]) && !strcmp (text, gui->no_messages.msg[XUI_MAX_SHOWN_MSGS])) {
      pthread_mutex_unlock (&gui->no_messages.mutex);
      return;
    }
    gui->no_messages.flags[XUI_MAX_SHOWN_MSGS] = flags;
    strlcpy (gui->no_messages.msg[XUI_MAX_SHOWN_MSGS], text, sizeof (gui->no_messages.msg[XUI_MAX_SHOWN_MSGS]));
    pthread_mutex_unlock (&gui->no_messages.mutex);
  } else {
    text = buf = xitk_vasprintf (message, args);
    va_end (args);
    if (!text)
      return;
  }

  if (gui->stdctl_enable || !gui->xitk) {
    printf ("%s\n", text);
  } else {
    int is_error = (type == XUI_MSG_ERROR);
    if (gui->verbosity >= 2 - is_error)
      dump_msg (text, is_error);
    if (gui->nongui_error_msg) {
      gui->nongui_error_msg (gui, text);
    } else {
      static const xitk_msg_type_t types[8] = {
        [XUI_MSG_ERROR] = XITK_MSG_TYPE_ERROR,
        [XUI_MSG_WARN]  = XITK_MSG_TYPE_WARN
      };
      xitk_register_key_t key, prev;
      unsigned int index;

      pthread_mutex_lock (&gui->no_messages.mutex);
      gui->no_messages.msg_index = index = (gui->no_messages.msg_index + 1) & (XUI_MAX_SHOWN_MSGS - 1);
      prev = gui->no_messages.msg_windows[index];
      pthread_mutex_unlock (&gui->no_messages.mutex);
      xitk_unregister_event_handler (gui->xitk, &gui->no_messages.msg_windows[index]);
      key = xitk_window_dialog_3 (gui->xitk, NULL, gui_layer_above (gui, NULL), 400, XITK_MSG_TITLE (types[type]),
        _errors_display_log_3, &gui->no_messages.helipad[index],
        (flags & XUI_MSG_MORE) ? _("Done") : XITK_LABEL_OK,
        (flags & XUI_MSG_MORE) ? _("More...") : NULL,
        NULL, NULL, 0, ALIGN_CENTER, "%s", text);
      video_window_set_transient_for (gui->vwin, xitk_get_window (gui->xitk, key));
      pthread_mutex_lock (&gui->no_messages.mutex);
      gui->no_messages.msg_windows[index] = key;
      pthread_mutex_unlock (&gui->no_messages.mutex);
      if (prev && (gui->verbosity >= 2))
        printf ("gui.message: too many unanswered messages, dropping 1.\n");
    }
  }

  free(buf);
}

static ATTR_FORMAT_ARG (1) const char *_silence_nonliteral_fmt_warn (const char *test, const char *s) {
  (void)test;
  return s;
}

/*
 * Display an error window error from a xine engine error.
 */
void gui_handle_xine_error (gGui_t *gui, xine_stream_t *stream, const char *mrl) {
  const char *_mrl = mrl ? mrl
                   : (stream == gui->stream) ? gui->mmk.mrl
                   : gui->visual_anim.mrls [gui->visual_anim.current];
  int err = xine_get_error (stream), xerr = err;
  static const char * const msgs[7] = {
    "",
    N_("- xine engine error -\n\n"
      "There is no input plugin available to handle '%s'.\n"
      "Maybe MRL syntax is wrong or file/stream source doesn't exist."),
    N_("- xine engine error -\n\n"
      "There is no demuxer plugin available to handle '%s'.\n"
      "Usually this means that the file format was not recognized."),
    N_("- xine engine error -\n\n"
      "Demuxer failed. Maybe '%s' is a broken file?\n"),
    N_("- xine engine error -\n\n"
      "Malformed mrl. Mrl '%s' seems malformed/invalid.\n"),
    N_("- xine engine error -\n\n"
      "Input plugin failed to open mrl '%s'\n"),
    N_("- xine engine error -\n\n!! Unhandled error !!\n")
  };

  /* gcc, optimize most of this away :-) */
  switch (err) {
    case XINE_ERROR_NONE:            err = 0; break;
    case XINE_ERROR_NO_INPUT_PLUGIN: err = 1; break;
    case XINE_ERROR_NO_DEMUX_PLUGIN: err = 2; break;
    case XINE_ERROR_DEMUX_FAILED:    err = 3; break;
    case XINE_ERROR_MALFORMED_MRL:   err = 4; break;
    case XINE_ERROR_INPUT_FAILED:    err = 5; break;
    default:                         err = 6;
  }

  if (gui->verbosity >= 2 - !!err) {
    static const char types[7][16] = {
      "NONE", "NO_INPUT_PLUGIN", "NO_DEMUX_PLUGIN", "DEMUX_FAILED", "MALFORMED_MRL", "INPUT_FAILED", ""
    };
    const char *snum = types[err];
    char nbuf[16], sbuf[8192];
    if (!snum[0]) {
      sprintf (nbuf, "%d", xerr);
      snum = nbuf;
    }
    snprintf (sbuf, sizeof (sbuf), "got XINE_ERROR_%s on mrl \"%s\".", snum, _mrl);
    dump_msg (sbuf, err);
  }

  if (!err)
    return;
  /* need this while xitk event bridge is running (startup with wrong or unsupported mrl).
   * in particular, xitk text font stuff is known to loop infinitely when re-entering from 2 threads. */
  xitk_lock (gui->xitk, 1);
  gui_msg (gui, XUI_MSG_ERROR | XUI_MSG_MORE,
    _silence_nonliteral_fmt_warn ("_%s", gettext (msgs[err])), _mrl);
  xitk_lock (gui->xitk, 0);
  /* gui->new_pos = -1; */
}

static void _too_slow_done (void *data, int state) {
  gGui_t *gui = data;

  if (state & XITK_WINDOW_DIALOG_CHECKED)
    config_update_num (gui->xine, "gui.dropped_frames_warning", 0);

  if ((state & XITK_WINDOW_DIALOG_BUTTONS_MASK) == 2) {
    /* FIXME: how to properly open the system browser?
     * should we just make it configurable? */
    gui_msg (gui, XUI_MSG_INFO, _("Opening mozilla web browser, this might take a while..."));
    xitk_system (1, "mozilla http://www.xine-project.org/faq#SPEEDUP");
  }
}

/*
 * Create the real window.
 */
void too_slow_window (gGui_t *gui) {
  const char *message = _("The amount of dropped frame is too high, "
                          "your system might be slow, not properly optimized or just too loaded.\n\n"
                          "http://www.xine-project.org/faq#SPEEDUP");
  xitk_register_key_t key;

  if (gui->verbosity > 0)
    dump_msg (message, 1);

  if (!(gui->flags & XUI_FLAG_warn_too_slow))
    return;

  if (gui->nongui_error_msg || gui->stdctl_enable)
    return;

  key = xitk_window_dialog_3 (gui->xitk, NULL,
    gui_layer_above (gui, NULL), 500, XITK_MSG_TITLE (XITK_MSG_TYPE_WARN), _too_slow_done, gui,
    _("Done"), _("Learn More..."), NULL, _("Disable this warning."), 0, ALIGN_CENTER, "%s", message);
  video_window_set_transient_for (gui->vwin, xitk_get_window (gui->xitk, key));
}
