// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.

using System.Collections.Generic;
using System.Linq;
using System.Security.Cryptography.SLHDsa.Tests;
using System.Security.Cryptography.Tests;
using Test.Cryptography;

namespace System.Security.Cryptography.Pkcs.Tests
{
    internal static class Certificates
    {
        public static readonly CertLoader RSAKeyTransfer1 = new CertLoaderFromRawData(RawData.s_RSAKeyTransfer1Cer, RawData.s_RSAKeyTransfer1Pfx, "1111");
        public static readonly CertLoader RSAKeyTransfer2 = new CertLoaderFromRawData(RawData.s_RSAKeyTransfer2Cer, RawData.s_RSAKeyTransfer2Pfx, "1111");
        public static readonly CertLoader RSAKeyTransfer3 = new CertLoaderFromRawData(RawData.s_RSAKeyTransfer3Cer, RawData.s_RSAKeyTransfer3Pfx, "1111");
        public static readonly CertLoader RSAKeyTransfer_ExplicitSki = new CertLoaderFromRawData(RawData.s_RSAKeyTransferCer_ExplicitSki, RawData.s_RSAKeyTransferPfx_ExplicitSki, "1111");
        public static readonly CertLoader RSAKeyTransferCapi1 = new CertLoaderFromRawData(RawData.s_RSAKeyTransferCapi1Cer, RawData.s_RSAKeyTransferCapi1Pfx, "1111");
        public static readonly CertLoader RSASha256KeyTransfer1 = new CertLoaderFromRawData(RawData.s_RSASha256KeyTransfer1Cer, RawData.s_RSASha256KeyTransfer1Pfx, "1111");
        public static readonly CertLoader RSASha384KeyTransfer1 = new CertLoaderFromRawData(RawData.s_RSASha384KeyTransfer1Cer, RawData.s_RSASha384KeyTransfer1Pfx, "1111");
        public static readonly CertLoader RSASha512KeyTransfer1 = new CertLoaderFromRawData(RawData.s_RSASha512KeyTransfer1Cer, RawData.s_RSASha512KeyTransfer1Pfx, "1111");
        public static readonly CertLoader RSA2048Sha256KeyTransfer1 = new CertLoaderFromRawData(RawData.s_RSA2048Sha256KeyTransfer1Cer, RawData.s_RSA2048Sha256KeyTransfer1Pfx, "1111");
        public static readonly CertLoader DHKeyAgree1 = new CertLoaderFromRawData(RawData.s_DHKeyAgree1Cer);
        public static readonly CertLoader RSA2048SignatureOnly = new CertLoaderFromRawData(RawData.s_Rsa2048SignatureOnlyCer, RawData.s_Rsa2048SignatureOnlyPfx, "12345");
        public static readonly CertLoader Dsa1024 = new CertLoaderFromRawData(RawData.s_dsa1024Cert, RawData.s_dsa1024Pfx, "1234");
        public static readonly CertLoader ECDsaP256Win = new CertLoaderFromRawData(RawData.ECDsaP256_DigitalSignature_Cert, RawData.ECDsaP256_DigitalSignature_Pfx_Windows, "Test");
        public static readonly CertLoader ECDsaP521Win = new CertLoaderFromRawData(RawData.ECDsaP521_DigitalSignature_Cert, RawData.ECDsaP521_DigitalSignature_Pfx_Windows, "Test");
        public static readonly CertLoader ValidLookingTsaCert = new CertLoaderFromRawData(RawData.ValidLookingTsaCert_Cer, RawData.ValidLookingTsaCert_Pfx, "export");
        public static readonly CertLoader TwoEkuTsaCert = new CertLoaderFromRawData(RawData.TwoEkuTsaCert, RawData.TwoEkuTsaPfx, "export");
        public static readonly CertLoader NonCriticalTsaEku = new CertLoaderFromRawData(RawData.NonCriticalTsaEkuCert, RawData.NonCriticalTsaEkuPfx, "export");
        public static readonly CertLoader TlsClientServerCert = new CertLoaderFromRawData(RawData.TlsClientServerEkuCert, RawData.TlsClientServerEkuPfx, "export");
        public static readonly CertLoader RSAKeyTransfer4_ExplicitSki = new CertLoaderFromRawData(RawData.s_RSAKeyTransfer4_ExplicitSkiCer, RawData.s_RSAKeyTransfer4_ExplicitSkiPfx, "1111");
        public static readonly CertLoader RSAKeyTransfer5_ExplicitSkiOfRSAKeyTransfer4 = new CertLoaderFromRawData(RawData.s_RSAKeyTransfer5_ExplicitSkiOfRSAKeyTransfer4Cer, RawData.s_RSAKeyTransfer5_ExplicitSkiOfRSAKeyTransfer4Pfx, "1111");
        public static readonly CertLoader NegativeSerialNumber = new CertLoaderFromRawData(RawData.NegativeSerialNumberCert, RawData.NegativeSerialNumberPfx, "1234");
        public static readonly CertLoader RsaOaep2048_NullParameters = new CertLoaderFromRawData(RawData.RsaOaep2048_NullParametersCert, RawData.RsaOaep2048_NullParametersPfx, "1111");
        public static readonly CertLoader RsaOaep2048_Sha1Parameters = new CertLoaderFromRawData(RawData.RsaOaep2048_Sha1ParametersCert, RawData.RsaOaep2048_Sha1ParametersPfx, "1111");
        public static readonly CertLoader RsaOaep2048_Sha256Parameters = new CertLoaderFromRawData(RawData.RsaOaep2048_Sha256ParametersCert, RawData.RsaOaep2048_Sha256ParametersPfx, "1111");
        public static readonly CertLoader RsaOaep2048_NoParameters = new CertLoaderFromRawData(RawData.RsaOaep2048_NoParametersCert, RawData.RsaOaep2048_NoParametersPfx, "1111");
        public static readonly CertLoader SlhDsaSha2_128s_Ietf = new CertLoaderFromRawData(SlhDsaTestData.IetfSlhDsaSha2_128sCertificate, SlhDsaTestData.IetfSlhDsaSha2_128sCertificatePfx, "PLACEHOLDER");
        public static readonly CertLoader[] SlhDsaGeneratedCerts = [..SlhDsaTestData.GeneratedKeyInfosRaw.Select(info => new CertLoaderFromRawData(info.Certificate, info.SelfSignedCertificatePfx, info.EncryptionPassword))];

        public static readonly Dictionary<MLDsaAlgorithm, CertLoader> MLDsaIetf = new()
        {
            { MLDsaAlgorithm.MLDsa44, new CertLoaderFromRawData(MLDsaTestsData.IetfMLDsa44.Certificate, MLDsaTestsData.IetfMLDsa44.Pfx_Seed, "PLACEHOLDER") },
            { MLDsaAlgorithm.MLDsa65, new CertLoaderFromRawData(MLDsaTestsData.IetfMLDsa65.Certificate, MLDsaTestsData.IetfMLDsa65.Pfx_Seed, "PLACEHOLDER") },
            { MLDsaAlgorithm.MLDsa87, new CertLoaderFromRawData(MLDsaTestsData.IetfMLDsa87.Certificate, MLDsaTestsData.IetfMLDsa87.Pfx_Seed, "PLACEHOLDER") },
        };

        // Note: the raw data is its own (nested) class to avoid problems with static field initialization ordering.
        private static class RawData
        {
            public static byte[] s_RSAKeyTransfer1Cer =
                 ("308201c830820131a003020102021031d935fb63e8cfab48a0bf7b397b67c0300d06092a864886f70d0101050500301a3118"
                + "30160603550403130f5253414b65795472616e7366657231301e170d3136303431323136323534375a170d31373034313232"
                + "32323534375a301a311830160603550403130f5253414b65795472616e736665723130819f300d06092a864886f70d010101"
                + "050003818d00308189028181009eaab63f5629db5ac0bd74300b43ba61f49189ccc30c001fa96bd3b139f45732cd3c37e422"
                + "ccbb2c598a4c6b3977a516a36ff850a5e914331f7445e86973f5a6cbb590105e933306e240eab6db72d08430cd7316e99481"
                + "a272adef0f2479d0b7c58e89e072364d660fdad1b51a603ff4549a82e8dc914df82bcc6c6c232985450203010001a30f300d"
                + "300b0603551d0f040403020520300d06092a864886f70d01010505000381810048c83e6f45d73a111c67e8f9f9c2d646292b"
                + "75cec52ef0f9ae3e1639504aa1759512c46527fcf5476897d3fb6fc515ff1646f8f8bc09f84ea6e2ad04242d3fb9b190b816"
                + "86b73d334e8b3afa7fb8eb31483efc0c7ccb0f8c1ca94d8be4f0daade4498501d02e6f92dd7b2f4401550896eb511ef14417"
                + "cbb5a1b360d67998d334").HexToByteArray();

            // password = "1111"
            public static byte[] s_RSAKeyTransfer1Pfx =
                 ("308205d20201033082058e06092a864886f70d010701a082057f0482057b308205773082034806092a864886f70d010701a0"
                + "82033904820335308203313082032d060b2a864886f70d010c0a0102a08202a6308202a2301c060a2a864886f70d010c0103"
                + "300e040818fdedadbb31b101020207d0048202806aa390fa9a4cb071a0daf25765ed69efe039896036c0f0edfc03ebe35d2a"
                + "f2f6a5bc9efd907f3b64ae15ac7f61d830e48810aa096ee37fe442b7bfbceeb92e22c25bd5484baf91460be29e06648485db"
                + "7b10ea92d17983c4d22067396c12e4598541ab989d7beb38bf8a0213fd7c9d49ecd46d319bbb58b1423504cd4145e1b33978"
                + "41306c5ace9eab42d408e05101911adc684e63a8c8c9579ce929e48ce2393af1a63c3180c52bd87475e3edb9763dff731ede"
                + "38fc8043dee375001a59e7d6eec5d686d509efee38ef0e7bddcd7ba0477f6f38ff7172ceaeef94ff56ad4b9533241f404d58"
                + "c2b5d54f1ab8250c56b1a70f57b7fffc640b7037408b8f830263befc031ffe7dbc6bef23f02c1e6e2b541be12009bfb11297"
                + "02fc0559e54d264df9b0d046c73ad1b25056231e5d3c4015bdc4f0a9af70ac28b7241233ecc845ce14484779102a45da2560"
                + "c354ec3e01f26d0e0b9a8b650f811d2ffeba95ec1e5cf6be2d060788c1b18ea4ec8f41e46da734c1216044a10a3e171620ed"
                + "79f7e9dd36972c89d91111c68fd60a94d2aa2a3dbbde0383c7c367f77b70a218ddf9fb4ed7abf94c233ffb2797d9ca3802ed"
                + "77868d3ab5651abb90e4de9ea74854b13603859b308689d770a62b5821e5a5650ecb23ca2894ad7901c7e1d2f22ef97e9092"
                + "f0791e886487a59d380d98c0368d3f2f261e0139714b02010e61aa073ee782b1fe5b6f79d070ef1412a13270138330a2e308"
                + "599e1e7829be9f983202ac0dc1c38d38587defe2741903af35227e4f979a68adef86a8459be4a2d74e5de7f94e114a8ea7e4"
                + "0ea2af6b8a93a747377bdd8ddd83c086bb20ca49854efb931ee689b319f984e5377f5a0f20d0a613326d749af00675c6bc06"
                + "0be528ef90ec6a9b2f9b3174301306092a864886f70d0109153106040401000000305d06092b060104018237110131501e4e"
                + "004d006900630072006f0073006f0066007400200053006f0066007400770061007200650020004b00650079002000530074"
                + "006f0072006100670065002000500072006f007600690064006500723082022706092a864886f70d010701a0820218048202"
                + "14308202103082020c060b2a864886f70d010c0a0103a08201e4308201e0060a2a864886f70d01091601a08201d0048201cc"
                + "308201c830820131a003020102021031d935fb63e8cfab48a0bf7b397b67c0300d06092a864886f70d0101050500301a3118"
                + "30160603550403130f5253414b65795472616e7366657231301e170d3136303431323136323534375a170d31373034313232"
                + "32323534375a301a311830160603550403130f5253414b65795472616e736665723130819f300d06092a864886f70d010101"
                + "050003818d00308189028181009eaab63f5629db5ac0bd74300b43ba61f49189ccc30c001fa96bd3b139f45732cd3c37e422"
                + "ccbb2c598a4c6b3977a516a36ff850a5e914331f7445e86973f5a6cbb590105e933306e240eab6db72d08430cd7316e99481"
                + "a272adef0f2479d0b7c58e89e072364d660fdad1b51a603ff4549a82e8dc914df82bcc6c6c232985450203010001a30f300d"
                + "300b0603551d0f040403020520300d06092a864886f70d01010505000381810048c83e6f45d73a111c67e8f9f9c2d646292b"
                + "75cec52ef0f9ae3e1639504aa1759512c46527fcf5476897d3fb6fc515ff1646f8f8bc09f84ea6e2ad04242d3fb9b190b816"
                + "86b73d334e8b3afa7fb8eb31483efc0c7ccb0f8c1ca94d8be4f0daade4498501d02e6f92dd7b2f4401550896eb511ef14417"
                + "cbb5a1b360d67998d3343115301306092a864886f70d0109153106040401000000303b301f300706052b0e03021a0414c4c0"
                + "4e0c0b0a20e50d58cb5ce565ba7c192d5d3f041479b53fc5f1f1f493a02cf113d563a247462e8726020207d0").HexToByteArray();

            public static byte[] s_RSAKeyTransfer2Cer =
                 ("308201c830820131a00302010202102bce9f9ece39f98044f0cd2faa9a14e7300d06092a864886f70d0101050500301a3118"
                + "30160603550403130f5253414b65795472616e7366657232301e170d3136303332353231323334325a170d31373033323630"
                + "33323334325a301a311830160603550403130f5253414b65795472616e736665723230819f300d06092a864886f70d010101"
                + "050003818d0030818902818100ea5a3834bfb863ae481b696ea7010ba4492557a160a102b3b4d11c120a7128f20b656ebbd2"
                + "4b426f1a6d40be0a55ca1b53ebdca202d258eebb20d5c662819182e64539360461dd3b5dda4085f10250fc5249cf023976b8"
                + "db2bc5f5e628fdb0f26e1b11e83202cbcfc9750efd6bb4511e6211372b60a97adb984779fdae21ce070203010001a30f300d"
                + "300b0603551d0f040403020520300d06092a864886f70d0101050500038181004dc6f9fd6054ae0361d28d2d781be590fa8f"
                + "5685fedfc947e315db12a4c47e220601e8c810e84a39b05b7a89f87425a06c0202ad48b3f2713109f5815e6b5d61732dac45"
                + "41da152963e700a6f37faf7678f084a9fb4fe88f7b2cbc6cdeb0b9fdcc6a8a16843e7bc281a71dc6eb8bbc4092d299bf7599"
                + "a3492c99c9a3acf41b29").HexToByteArray();

            // password = "1111"
            public static byte[] s_RSAKeyTransfer2Pfx =
                 ("308205d20201033082058e06092a864886f70d010701a082057f0482057b308205773082034806092a864886f70d010701a0"
                + "82033904820335308203313082032d060b2a864886f70d010c0a0102a08202a6308202a2301c060a2a864886f70d010c0103"
                + "300e04080338620310d29656020207d0048202804a94d3b1a1bf43efe3726aa9f0abc90c44585d2f0aee0864b4d574cd2cc1"
                + "dca4a353b102779e072ed6072d3c083b83974e74069b353ba8ac8be113228e0225993f5ecb7293ab1a6941bef75f7bcb0e3b"
                + "e6902832be46b976e94c6a0bc6865822ff07371551d206e300558da67cf972d89c3d181beb86d02f5523baa8351b88992654"
                + "a4c507e136dd32120530585a25424fe40f9962b910e08fb55f582c3764946ba7f6d92520decfc9faa2d5e180f9824e5ed4c8"
                + "c57e549a27950e7a875f2ed450035a69de6d95ec7bd9e30b65b8563fdd52809a4a1fc960f75c817c72f98afb000e8a8a33be"
                + "f62e458c2db97b464121489bf3c54de45e05f9c3e06c21892735e3f2d9353a71febcd6a73a0af3c3fc0922ea71bdc483ed7e"
                + "5653740c107cfd5e101e1609c20061f864671ccb45c8b5b5b7b48436797afe19de99b5027faf4cead0fd69d1987bbda5a0a4"
                + "0141495998d368d3a4747fc370205eed9fc28e530d2975ca4084c297a544441cf46c39fb1f0f42c65b99a6c9c970746012ad"
                + "c2be15fbbc803d5243f73fdec50bdee0b74297bd30ca3ea3a1dc623db6a199e93e02053bd1a6ca1a00a5c6090de1fa10cdd5"
                + "b5541bd5f5f92ff60a139c50deff8768e7b242018611efd2cce0d9441f3c8b207906345a985617ba5e98e7883c9b925ba17d"
                + "c4fadddbbe025cecd24bb9b95cae573a8a24ceb635eb9f663e74b0084a88f4e8e0d2baf767be3abe5b873695989a0edac7bd"
                + "092de79c3b6427dcbedee0512918fc3f7a45cd6898701673c9ed9f2f873abb8aa64cec7b8d350e8c780c645e50ce607a1afd"
                + "bcefba6cf5cebbc766d1e61d78fbef7680b38dd0f32133ceb39c6c9cabd0b33af9f7ef73c94854b57cf68e61997b61393a0b"
                + "6fc37f8834157e0c9fba3174301306092a864886f70d0109153106040401000000305d06092b060104018237110131501e4e"
                + "004d006900630072006f0073006f0066007400200053006f0066007400770061007200650020004b00650079002000530074"
                + "006f0072006100670065002000500072006f007600690064006500723082022706092a864886f70d010701a0820218048202"
                + "14308202103082020c060b2a864886f70d010c0a0103a08201e4308201e0060a2a864886f70d01091601a08201d0048201cc"
                + "308201c830820131a00302010202102bce9f9ece39f98044f0cd2faa9a14e7300d06092a864886f70d0101050500301a3118"
                + "30160603550403130f5253414b65795472616e7366657232301e170d3136303332353231323334325a170d31373033323630"
                + "33323334325a301a311830160603550403130f5253414b65795472616e736665723230819f300d06092a864886f70d010101"
                + "050003818d0030818902818100ea5a3834bfb863ae481b696ea7010ba4492557a160a102b3b4d11c120a7128f20b656ebbd2"
                + "4b426f1a6d40be0a55ca1b53ebdca202d258eebb20d5c662819182e64539360461dd3b5dda4085f10250fc5249cf023976b8"
                + "db2bc5f5e628fdb0f26e1b11e83202cbcfc9750efd6bb4511e6211372b60a97adb984779fdae21ce070203010001a30f300d"
                + "300b0603551d0f040403020520300d06092a864886f70d0101050500038181004dc6f9fd6054ae0361d28d2d781be590fa8f"
                + "5685fedfc947e315db12a4c47e220601e8c810e84a39b05b7a89f87425a06c0202ad48b3f2713109f5815e6b5d61732dac45"
                + "41da152963e700a6f37faf7678f084a9fb4fe88f7b2cbc6cdeb0b9fdcc6a8a16843e7bc281a71dc6eb8bbc4092d299bf7599"
                + "a3492c99c9a3acf41b293115301306092a864886f70d0109153106040401000000303b301f300706052b0e03021a04143cdb"
                + "6a36dfd2288ba4e3771766d7a5289c04419704146c84193dc4f3778f21197d11ff994d8bf4822049020207d0").HexToByteArray();

            public static byte[] s_RSAKeyTransfer3Cer =
                 ("308201c830820131a00302010202104497d870785a23aa4432ed0106ef72a6300d06092a864886f70d0101050500301a3118"
                + "30160603550403130f5253414b65795472616e7366657233301e170d3136303332353231323335355a170d31373033323630"
                + "33323335355a301a311830160603550403130f5253414b65795472616e736665723330819f300d06092a864886f70d010101"
                + "050003818d0030818902818100bbc6fe8702a4e92eadb9b0f41577c0fffc731411c6f87c27c9ef7c2e2113d4269574f44f2e"
                + "90382bd193eb2f57564cf00092172d91a003e7252a544958b30aab6402e6fba7e442e973d1902e383f6bc4a4d8a00e60b3f3"
                + "3a032bdf6bedb56acb0d08669b71dd7b35f5d39d9914f5e111e1cd1559eb741a3075d673c39e7850a50203010001a30f300d"
                + "300b0603551d0f040403020520300d06092a864886f70d01010505000381810058abccbf69346360351d55817a61a6091b0b"
                + "022607caeb44edb6f05a91f169903608d7391b245ac0dcbe052e16a91ac1f8d9533f19f6793f15cb6681b2cbaa0d8e83d77b"
                + "5207e7c70d843deda8754af8ef1029e0b68c35d88c30d7da2f85d1a20dd4099facf373341b50a8a213f735421062e1477459"
                + "6e27a32e23b3f3fcfec3").HexToByteArray();

            // password = "1111"
            public static byte[] s_RSAKeyTransfer3Pfx =
                 ("308205d20201033082058e06092a864886f70d010701a082057f0482057b308205773082034806092a864886f70d010701a0"
                + "82033904820335308203313082032d060b2a864886f70d010c0a0102a08202a6308202a2301c060a2a864886f70d010c0103"
                + "300e0408a9197ad512c316b5020207d004820280b1c213fa87f3906cde3502249830a01d1d636d0058bd8d6172222544c35a"
                + "9676f390a5ef1d52f13fae2f04fe2ca1bcb9914296f97fdf729a52e0c3472c9f7ae72bd746f0a66b0c9363fae0328ad063fa"
                + "45d35cc2679c85e970c7420ad036012ce553ef47ed8fe594917739aab1123be435a0ca88ac4b85cf3d341d4aeb2c6816d8fc"
                + "a2e9611224b42f0ca00bde4f25db460200f25fe99ed4fd0236e4d00c48085aec4734f0bce7e6c8fea08b11a2a7214f4a18c0"
                + "fa4b732c8dae5c5857f2edec27fa94eb17ac05d1d05b321b01c1368231ff89c46c6378abf67cb751156370bbcc35591e0028"
                + "d4ace5158048d9d25b00e028b7766f1c74ade9603a211aad241fc3b7599a2b15f86846dfdc106f49cf56491b3f6ff451d641"
                + "400f38fabcdb74a4423828b041901fa5d8c528ebf1cc6169b08eb14b2d457acb6970a11ccaa8fbc3b37b6454803b07b1916e"
                + "2ad3533f2b72721625c11f39a457033744fde3745c3d107a3f1e14118e04db41ca8970a383e8706bcf8ba5439a4cb360b250"
                + "4fcae3dbfb54af0154f9b813ad552f2bdbc2a9eb61d38ae5e6917990cbeb1c5292845637c5fed477dabbed4198a2978640ba"
                + "7db22c85322115fa9027ad418a61e2e31263da3776398faaaab818aae6423c873bd393f558fa2fc05115b4983d35ecfeae13"
                + "601519a53c7a77b5688aeddc6f210a65303eeb0dbd7e3a5ec94d7552cf4cbe7acebf5e4e10abaccd2e990f1cf217b98ad9b5"
                + "06820f7769a7c5e61d95462918681c2b111faf29f13e3615c4c5e75426dbcd903c483590434e8ab1965dc620e7d8bebea36f"
                + "53f1bc0807933b0ef9d8cc1b36b96aff8288e9a8d1bba24af562dfeb497b9a58083b71d76dacd6f2ce67cb2593c6f06472ef"
                + "e508012c34f40d87e0be3174301306092a864886f70d0109153106040401000000305d06092b060104018237110131501e4e"
                + "004d006900630072006f0073006f0066007400200053006f0066007400770061007200650020004b00650079002000530074"
                + "006f0072006100670065002000500072006f007600690064006500723082022706092a864886f70d010701a0820218048202"
                + "14308202103082020c060b2a864886f70d010c0a0103a08201e4308201e0060a2a864886f70d01091601a08201d0048201cc"
                + "308201c830820131a00302010202104497d870785a23aa4432ed0106ef72a6300d06092a864886f70d0101050500301a3118"
                + "30160603550403130f5253414b65795472616e7366657233301e170d3136303332353231323335355a170d31373033323630"
                + "33323335355a301a311830160603550403130f5253414b65795472616e736665723330819f300d06092a864886f70d010101"
                + "050003818d0030818902818100bbc6fe8702a4e92eadb9b0f41577c0fffc731411c6f87c27c9ef7c2e2113d4269574f44f2e"
                + "90382bd193eb2f57564cf00092172d91a003e7252a544958b30aab6402e6fba7e442e973d1902e383f6bc4a4d8a00e60b3f3"
                + "3a032bdf6bedb56acb0d08669b71dd7b35f5d39d9914f5e111e1cd1559eb741a3075d673c39e7850a50203010001a30f300d"
                + "300b0603551d0f040403020520300d06092a864886f70d01010505000381810058abccbf69346360351d55817a61a6091b0b"
                + "022607caeb44edb6f05a91f169903608d7391b245ac0dcbe052e16a91ac1f8d9533f19f6793f15cb6681b2cbaa0d8e83d77b"
                + "5207e7c70d843deda8754af8ef1029e0b68c35d88c30d7da2f85d1a20dd4099facf373341b50a8a213f735421062e1477459"
                + "6e27a32e23b3f3fcfec33115301306092a864886f70d0109153106040401000000303b301f300706052b0e03021a0414cd11"
                + "0833d653f2e18d2afb2de74689ff0446ec7d0414f2ca1c390db19317697044b9012ef6864e0f05cc020207d0").HexToByteArray();

            public static byte[] s_RSAKeyTransferCapi1Cer =
                 ("3082020c30820179a00302010202105d2ffff863babc9b4d3c80ab178a4cca300906052b0e03021d0500301e311c301a0603"
                + "55040313135253414b65795472616e736665724361706931301e170d3135303431353037303030305a170d32353034313530"
                + "37303030305a301e311c301a060355040313135253414b65795472616e73666572436170693130819f300d06092a864886f7"
                + "0d010101050003818d0030818902818100aa272700586c0cc41b05c65c7d846f5a2bc27b03e301c37d9bff6d75b6eb6671ba"
                + "9596c5c63ba2b1af5c318d9ca39e7400d10c238ac72630579211b86570d1a1d44ec86aa8f6c9d2b4e283ea3535923f398a31"
                + "2a23eaeacd8d34faaca965cd910b37da4093ef76c13b337c1afab7d1d07e317b41a336baa4111299f99424408d0203010001"
                + "a3533051304f0603551d0104483046801015432db116b35d07e4ba89edb2469d7aa120301e311c301a060355040313135253"
                + "414b65795472616e73666572436170693182105d2ffff863babc9b4d3c80ab178a4cca300906052b0e03021d050003818100"
                + "81e5535d8eceef265acbc82f6c5f8bc9d84319265f3ccf23369fa533c8dc1938952c5931662d9ecd8b1e7b81749e48468167"
                + "e2fce3d019fa70d54646975b6dc2a3ba72d5a5274c1866da6d7a5df47938e034a075d11957d653b5c78e5291e4401045576f"
                + "6d4eda81bef3c369af56121e49a083c8d1adb09f291822e99a429646").HexToByteArray();

            // Password = "1111"
            //
            // Built by:
            //
            //   makecert -r -len 1024 -n "CN=RSAKeyTransferCapi1" -b 04/15/2015 -e 04/15/2025 RSAKeyTransferCapi1.cer -sv RSAKeyTransferCapi1.pvk -sky exchange
            //   pvk2pfx.exe -pvk RSAKeyTransferCapi1.pvk -spc RSAKeyTransferCapi1.cer -pfx RSAKeyTransferCapi1.pfx -po 1111
            //
            public static byte[] s_RSAKeyTransferCapi1Pfx =
                 ("30820626020103308205e206092a864886f70d010701a08205d3048205cf308205cb3082035806092a864886f70d010701a0"
                + "82034904820345308203413082033d060b2a864886f70d010c0a0102a08202b6308202b2301c060a2a864886f70d010c0103"
                + "300e0408dbd82a9abd7c1a2b020207d004820290768873985e74c2ece506531d348d8b43f2ae8524a2bcc737eeb778fac1ee"
                + "b21f82deb7cf1ba54bc9a865be8294de23e6648ffb881ae2f0132265c6dacd60ae55df1497abc3eb9181f47cb126261ea66f"
                + "d22107bbcdb8825251c60c5179ef873cb7e047782a4a255e3e9d2e0dd33f04cde92f9d268e8e4daf8ba74e54d8b279a0e811"
                + "9a3d0152608c51331bbdd23ff65da492f85809e1d7f37af9ae00dca796030a19e517e7fe2572d4502d4738fd5394ee369216"
                + "fb64cf84beab33860855e23204156dcf774fac18588f1c1ca1a576f276e9bfbf249449842f193020940a35f163378a2ce7da"
                + "37352d5b0c7c3ac5eb5f21ed1921a0076523b2e66a101655bb78d4ecc22472ac0151b7e8051633747d50377258ab19dcb22e"
                + "e09820876607d3291b55bba73d713d6689486b310507316b4f227383e4869628ad31f0b431145d45f4f38f325772c866a20e"
                + "0b442088cbf663e92e8ee82dd495fba8d40345474a384bb3b80b49ca1d66eef5321235135dcc0a5425e4bf3b8ce5c2469e2a"
                + "c0f8d53aab276361d9a2ff5c974c6e6b66126158676331fe7f74643fd1e215b22d7799846651350ed0f1f21a67ac6b3bfd62"
                + "7defb235ef8732d772d1c4bea2ae80c165f0182f547ea7a3f3366288f74c030689988a9838c27b10a48737a620d8220f68b4"
                + "ea8d8eb26298d5359d54a59c6be6716cefc12c929e17bb71c57c560659a7757ba8ac08ae90794474e50f0e87a22e2b7c3ebd"
                + "061390928bf48c6c6200c225f7025eab20f5f6fee5dc41682b2d4a607c8c81964b7d52651e5a62a41f4e8ea3982c294a4aee"
                + "8a67dc36a8b34b29509a4868c259dc205d1e8a3b6259a76a147f002f3bfbc8378e8edd230a34f9cd5f13ce6651b10394709d"
                + "5092bb6a70d8c2816f1c0e44cd45dfa7c2d94aa32112d79cb44a3174301306092a864886f70d010915310604040100000030"
                + "5d06092b060104018237110131501e4e004d006900630072006f0073006f006600740020005300740072006f006e00670020"
                + "00430072007900700074006f0067007200610070006800690063002000500072006f007600690064006500723082026b0609"
                + "2a864886f70d010701a082025c048202583082025430820250060b2a864886f70d010c0a0103a082022830820224060a2a86"
                + "4886f70d01091601a0820214048202103082020c30820179a00302010202105d2ffff863babc9b4d3c80ab178a4cca300906"
                + "052b0e03021d0500301e311c301a060355040313135253414b65795472616e736665724361706931301e170d313530343135"
                + "3037303030305a170d3235303431353037303030305a301e311c301a060355040313135253414b65795472616e7366657243"
                + "6170693130819f300d06092a864886f70d010101050003818d0030818902818100aa272700586c0cc41b05c65c7d846f5a2b"
                + "c27b03e301c37d9bff6d75b6eb6671ba9596c5c63ba2b1af5c318d9ca39e7400d10c238ac72630579211b86570d1a1d44ec8"
                + "6aa8f6c9d2b4e283ea3535923f398a312a23eaeacd8d34faaca965cd910b37da4093ef76c13b337c1afab7d1d07e317b41a3"
                + "36baa4111299f99424408d0203010001a3533051304f0603551d0104483046801015432db116b35d07e4ba89edb2469d7aa1"
                + "20301e311c301a060355040313135253414b65795472616e73666572436170693182105d2ffff863babc9b4d3c80ab178a4c"
                + "ca300906052b0e03021d05000381810081e5535d8eceef265acbc82f6c5f8bc9d84319265f3ccf23369fa533c8dc1938952c"
                + "5931662d9ecd8b1e7b81749e48468167e2fce3d019fa70d54646975b6dc2a3ba72d5a5274c1866da6d7a5df47938e034a075"
                + "d11957d653b5c78e5291e4401045576f6d4eda81bef3c369af56121e49a083c8d1adb09f291822e99a429646311530130609"
                + "2a864886f70d0109153106040401000000303b301f300706052b0e03021a041463c18f4fec17cf06262e8acd744e18b8ab7b"
                + "8f280414134ec4a25653b142c3d3f9999830f2ac66ef513b020207d0").HexToByteArray();

            public static byte[] s_RSASha256KeyTransfer1Cer =
                 ("308201d43082013da003020102021072c6c7734916468c4d608253da017676300d06092a864886f70d01010b05003020311e"
                + "301c060355040313155253415368613235364b65795472616e7366657231301e170d3136303431383130353934365a170d31"
                + "37303431383136353934365a3020311e301c060355040313155253415368613235364b65795472616e736665723130819f30"
                + "0d06092a864886f70d010101050003818d0030818902818100cad046de3a7f6dc78fc5a4e01d1f7d90db596f586334d5708a"
                + "ecb8e52d6bb912c0b5ec9633a82b4abac4c2860c766f2fdf1c905c4a72a54adfd041adabe5f2afd1e2ad88615970e818dc3d"
                + "4d00bb6c4ce94c5eb4e3efedd80d14c3d295ea471ae430cbb20b071582f1396369fbe90c14aa5f85b8e3b14011d81fbd41ec"
                + "b1495d0203010001a30f300d300b0603551d0f040403020520300d06092a864886f70d01010b050003818100baed2a5ae2d1"
                + "1ee4209c0694c790e72e3e8ad310b2506b277d7c001b09f660d48dba846ac5bbef97653613adf53d7624fc9b2b337f25cb33"
                + "74227900cfefbe2fdac92b4f769cf2bf3befb485f282a85bfb09454b797ce5286de560c219fb0dd6fce0442adbfef4f767e9"
                + "ac81cf3e9701baf81efc73a0ed88576adff12413b827").HexToByteArray();

            // password = "1111"
            public static byte[] s_RSASha256KeyTransfer1Pfx =
                 ("308205de0201033082059a06092a864886f70d010701a082058b04820587308205833082034806092a864886f70d010701a0"
                + "82033904820335308203313082032d060b2a864886f70d010c0a0102a08202a6308202a2301c060a2a864886f70d010c0103"
                + "300e040829e4911057aa5fb6020207d00482028052e016e1e339ca6a648ab1e152813899bd2ec0de1e34804c33e109cf2136"
                + "d42edc0d5ff8a005939ec38d4284aa0cfda295e801b701855c3c129e9311dc80b3538ba76d3164d48d83a73949d695f42294"
                + "75469f262c807767bc5c12bb83b2c4857fa9f8c7c519143136ba93ab93e17ad4b0b63cf6449708e6128425b00eaeae6bc5b6"
                + "7ff092673c9aabbbb63e90424295f0ae828bcd00f5ad85fe8384711ca5fffd4cbfe57ddbc3e5bb1df19e6fd7640fbd8d4516"
                + "f8d2d5ec84baca72ac42b50e77be0055dfdbbbe9c6de42c06fc86de8fbfc6231db89b30065d534e76aa851833b6c9c651288"
                + "c12f87ba12ae429e9bec0b22297c666046355ebd5a54dc7f13a55e0ebd53c768f69eee57d6041263f5bdf1c4c5b2b55dfb9b"
                + "38171aaed0d21fd5a41e0ef760db42f373c9007e1df47fd79ba9b41528c9c02dffdd04472265763ae94f4e05b86976a2c459"
                + "093d8e6bb0d0c5da5994fe3edbdf843b67e8e4c4daf59351788bf8b96da116aecbb95d52bf727ff10ca41340112f0bcb41e0"
                + "b8373a6e55727c745b77cf1944b74fa447ed0a6d93b8e43fd6e4b4b3e0d49d03ee2ee12d15519406c49a4c1be70de5171c93"
                + "d056e9f47b8a96d50f01873be4c596590f1247a2f2822dea9339fa87dd49545b559e0225ab738ecc0b054155749670d412be"
                + "472d13dfb0a8c8f56b3c0be1aa0d9195ba937b0c2119c702a0be1f83e1b4a77375ed1654e3dcf6b8ce119db3ac7cd440369a"
                + "b0b964e0b526b865680015cc3046a20badeaca4543ce65042ff5eb691e93232754a7b34fd8b6833c2625fdfdc59d80b3dcb4"
                + "ce70d1833ecf6344bb7331e46b71bb1592b6d814370548ee2b2f4df207696be87d2e1e0c5dc0ca528e5a231802cbb7853968"
                + "beb6ceb1b3a2998ecd313174301306092a864886f70d0109153106040401000000305d06092b060104018237110131501e4e"
                + "004d006900630072006f0073006f0066007400200053006f0066007400770061007200650020004b00650079002000530074"
                + "006f0072006100670065002000500072006f007600690064006500723082023306092a864886f70d010701a0820224048202"
                + "203082021c30820218060b2a864886f70d010c0a0103a08201f0308201ec060a2a864886f70d01091601a08201dc048201d8"
                + "308201d43082013da003020102021072c6c7734916468c4d608253da017676300d06092a864886f70d01010b05003020311e"
                + "301c060355040313155253415368613235364b65795472616e7366657231301e170d3136303431383130353934365a170d31"
                + "37303431383136353934365a3020311e301c060355040313155253415368613235364b65795472616e736665723130819f30"
                + "0d06092a864886f70d010101050003818d0030818902818100cad046de3a7f6dc78fc5a4e01d1f7d90db596f586334d5708a"
                + "ecb8e52d6bb912c0b5ec9633a82b4abac4c2860c766f2fdf1c905c4a72a54adfd041adabe5f2afd1e2ad88615970e818dc3d"
                + "4d00bb6c4ce94c5eb4e3efedd80d14c3d295ea471ae430cbb20b071582f1396369fbe90c14aa5f85b8e3b14011d81fbd41ec"
                + "b1495d0203010001a30f300d300b0603551d0f040403020520300d06092a864886f70d01010b050003818100baed2a5ae2d1"
                + "1ee4209c0694c790e72e3e8ad310b2506b277d7c001b09f660d48dba846ac5bbef97653613adf53d7624fc9b2b337f25cb33"
                + "74227900cfefbe2fdac92b4f769cf2bf3befb485f282a85bfb09454b797ce5286de560c219fb0dd6fce0442adbfef4f767e9"
                + "ac81cf3e9701baf81efc73a0ed88576adff12413b8273115301306092a864886f70d0109153106040401000000303b301f30"
                + "0706052b0e03021a0414282ee1780ac2a08b2783b1f8f7c855fb1a53ce9e04143fad59471323dc979f3bf29b927e54eca677"
                + "7576020207d0").HexToByteArray();

            public static byte[] s_RSASha384KeyTransfer1Cer =
                 ("308201d43082013da00302010202103c724fb7a0159a9345caac9e3df5f136300d06092a864886f70d01010c05003020311e"
                + "301c060355040313155253415368613338344b65795472616e7366657231301e170d3136303431383131303530365a170d31"
                + "37303431383137303530365a3020311e301c060355040313155253415368613338344b65795472616e736665723130819f30"
                + "0d06092a864886f70d010101050003818d0030818902818100e6b46b0e6f4f6df724081e11f201b9fbb07f2b6db2b868f607"
                + "68e2b5b843f690ca5e8d48f439d8b181ace2fb27dfa07eff0324642d6c9129e2d95e136702f6c31fe3ccf3aa87ba9f1b6f7b"
                + "acd07156ff3dd2a7f4c70356fb94b0adbde6819383c19bbefb4a6d1d6491a770d5f9feb11bcb3e5ac99cb153984dee0910e4"
                + "b57f8f0203010001a30f300d300b0603551d0f040403020520300d06092a864886f70d01010c0500038181003842cc95a680"
                + "c8a31534a461d061a4706a0aba52b7a1c709c2f1e3b94acf6dc0930b74e63e3babf3c5b11c8f8a888722d9f23c7e0a8c9b09"
                + "90ebcdbce563b8d4209efc1b04750f46c8c6117ccb96b26b5f02b0b5f961ab01b0c3b4cdb2530cbc5dcf37786712a3476ce7"
                + "32c5c544c328db5ebc3a338b18fe32aedaffedd973ef").HexToByteArray();

            // password = "1111"
            public static byte[] s_RSASha384KeyTransfer1Pfx =
                 ("308205de0201033082059a06092a864886f70d010701a082058b04820587308205833082034806092a864886f70d010701a0"
                + "82033904820335308203313082032d060b2a864886f70d010c0a0102a08202a6308202a2301c060a2a864886f70d010c0103"
                + "300e040856d7d59810ce8b17020207d00482028082012797edb5f74429bb6b91dd1e24aa32a19b89d92fd486e826773a7a11"
                + "03a9b49d98c6b7e97d411d19b44cd79559964f31cb6f0443c70d687c390d31c656ee3887391ae1735c142d891ec8337c5dc4"
                + "d6b5a4f09400a4cc35dd8dbde831f7625b7afedf4990294988b0b32b2889c97cd85c2568ffef332be83232449dd4083a43d4"
                + "89e654520eb922239379b5e9f5dfc1e64972339dee27dfdd874e2ee2b85f941f3b313ab881571c3a5a9b292d8c82d79d74a0"
                + "2d78dd5cfce366b3a914b61b861b35948757d137e5d53589a0fa2f1b4d06ee6b4aa4b8d3f526b059637b236ceb2de128d7bd"
                + "f91c12612d09e1cb4bed1b5e336fb56424b68dcc6d6cd5d90f666047c8b181526a60622027d322db0172046c23e84a3c725e"
                + "45ce774df037cafb74b359c3ec6874dce98673d9f7581f54dcb6e3c40583de2de6aaf6739bba878362e9bfab331cab2eb22d"
                + "3b130dec4eedf55a7ed8d5960e9f037209f9c1ef584c6dd5de17245d0da62c54420dc862b6648418d2aa9797f86a2cd0ecf6"
                + "abcbeb16907d8f44021690682a4e1286cd3f9aea4866108b3c968cf4b80a39c60436079617346861662e01a5419d8cebe2c6"
                + "e186141e42baf7cfc596270dbab8db03da9bd501daa426e24aa2d8ccf4d4512a8dce3ae8954be69b5c3a70fac587ac91ad97"
                + "fb427c8118659b710b57183c4fd16ffd276834e2fe45d74e175f3f5077783cdd7668b4e87217512ceb7f3e64715ba22bbab7"
                + "0d1b3485820c16304758cf1dd0b806d801f1185bb14d12f2c147ec65b95088077dec23498ebe40a952727c559c7af5cf20f1"
                + "f491f4123db093dc1a67014c3db46c11c7d5833b15167c91138eba6b4badf869aefba5fbea523a5ad02bb676db6039e7aabd"
                + "44f0702d59cf3d1ad9bb3174301306092a864886f70d0109153106040401000000305d06092b060104018237110131501e4e"
                + "004d006900630072006f0073006f0066007400200053006f0066007400770061007200650020004b00650079002000530074"
                + "006f0072006100670065002000500072006f007600690064006500723082023306092a864886f70d010701a0820224048202"
                + "203082021c30820218060b2a864886f70d010c0a0103a08201f0308201ec060a2a864886f70d01091601a08201dc048201d8"
                + "308201d43082013da00302010202103c724fb7a0159a9345caac9e3df5f136300d06092a864886f70d01010c05003020311e"
                + "301c060355040313155253415368613338344b65795472616e7366657231301e170d3136303431383131303530365a170d31"
                + "37303431383137303530365a3020311e301c060355040313155253415368613338344b65795472616e736665723130819f30"
                + "0d06092a864886f70d010101050003818d0030818902818100e6b46b0e6f4f6df724081e11f201b9fbb07f2b6db2b868f607"
                + "68e2b5b843f690ca5e8d48f439d8b181ace2fb27dfa07eff0324642d6c9129e2d95e136702f6c31fe3ccf3aa87ba9f1b6f7b"
                + "acd07156ff3dd2a7f4c70356fb94b0adbde6819383c19bbefb4a6d1d6491a770d5f9feb11bcb3e5ac99cb153984dee0910e4"
                + "b57f8f0203010001a30f300d300b0603551d0f040403020520300d06092a864886f70d01010c0500038181003842cc95a680"
                + "c8a31534a461d061a4706a0aba52b7a1c709c2f1e3b94acf6dc0930b74e63e3babf3c5b11c8f8a888722d9f23c7e0a8c9b09"
                + "90ebcdbce563b8d4209efc1b04750f46c8c6117ccb96b26b5f02b0b5f961ab01b0c3b4cdb2530cbc5dcf37786712a3476ce7"
                + "32c5c544c328db5ebc3a338b18fe32aedaffedd973ef3115301306092a864886f70d0109153106040401000000303b301f30"
                + "0706052b0e03021a041429bd86de50f91b8f804b2097b1d9167ca56577f40414b8714b8172fa1baa384bed57e3ddb6d1851a"
                + "f5e9020207d0").HexToByteArray();

            public static byte[] s_RSASha512KeyTransfer1Cer =
                 ("308201d43082013da00302010202102f5d9d58a5f41b844650aa233e68f105300d06092a864886f70d01010d05003020311e"
                + "301c060355040313155253415368613531324b65795472616e7366657231301e170d3136303431383131303532355a170d31"
                + "37303431383137303532355a3020311e301c060355040313155253415368613531324b65795472616e736665723130819f30"
                + "0d06092a864886f70d010101050003818d0030818902818100b2eca20240da8486b1a933ade62ad8781ef30d4434ebbc9b3f"
                + "c9c550d0f9a75f4345b5520f3d0bafa63b8037785d1e8cbd3efe9a22513dc8b82bcd1d44bf26bd2c292205ca3e793ff1cb09"
                + "e0df4afefb542362bc148ea2b76053d06754b4a37a535afe63b048282f8fb6bd8cf5dc5b47b7502760587f84d9995acbf1f3"
                + "4a3ca10203010001a30f300d300b0603551d0f040403020520300d06092a864886f70d01010d050003818100493d857684d2"
                + "7468dd09926d20933254c7c79645f7b466e7b4a90a583cedba1c3b3dbf4ccf1c2506eb392dcf15f53f964f3c3b519132a38e"
                + "b966d3ea397fe25457b8a703fb43ddab1c52272d6a12476df1df1826c90fb679cebc4c04efc764fd8ce3277305c3bcdf1637"
                + "91784d778663194097180584e5e8ab69039908bf6f86").HexToByteArray();

            // password = "1111"
            public static byte[] s_RSASha512KeyTransfer1Pfx =
                 ("308205de0201033082059a06092a864886f70d010701a082058b04820587308205833082034806092a864886f70d010701a0"
                + "82033904820335308203313082032d060b2a864886f70d010c0a0102a08202a6308202a2301c060a2a864886f70d010c0103"
                + "300e04083a0e344b65dd4e27020207d00482028014464df9f07d2cb37a28607570130de5877e829e759040976866afc831db"
                + "4d2741734ae53ea5eb80c1080dae7b0a2acddabd3d47b1ed5f3051455429308f3b7b0b48c5a4dbc5d718534472c746ce62f1"
                + "bbb8c5d178c1d3e91efdbd4f56569517bcadf3c81dbe4c34746194e47bcf46b74cd1880d7bd12d9b819b462fbcf6f51f3972"
                + "2858c9b9af8975bfefd7f007928b39e11d50b612761d03e566b992f92e9c9873d138c937fc43fe971db4c8e57b51aeef4ed0"
                + "022ec76c3bb4bd9f2395b99585449303a6d68183edf6e5dda1885531bee10b7cf6509390f4ee6a37ed2931d658548bd6390f"
                + "a7094fdf017166309074c00581d2b7dcaaee657f9c48e08edf636004dc5e60486dd022c45058700fe682472b371380948792"
                + "74c2a20dd9e07e149e7ab52157db748160ad81f91019297baa58ce68656b0b2f7c9ac88b3da6920c2a5eab7bcc2629974f8a"
                + "6c8bf33629af05e4e34d5d24393448e9751b7708f5915b0fd97a5af4dd5a37d71b18b6526316cbc65b1c6af8a6779acbc470"
                + "2381f027bdb118cb84e9005b02a8bd2d02365d280cffb04831f877de7bd3d3287f11beed8978a5389e2b28317eb90569781f"
                + "94f66f672736a09b4a7caeaaefd1909f2d20255df51512dbd08ec6125455d932b626bdfd3c4f669148fa783671f90b59ceff"
                + "560c338f92cbe8bf7fbab4db3e9b943effac747eb34f06bd72aee961ed31742caa2a9934a5fe4685677ecbca6fb1b1c0b642"
                + "b4f71d55d0e2cb1dc10ce845514090cc117a875c4d10c0ce367e31091144eacd7e600792d61d036bde020e3bb9a004a7dd1a"
                + "cf03541b6fff3bcef4c30df05d98b75688320685261b2b34813407b20a7c92a04eeb46cb7e618a6ee32154728ba6735668f4"
                + "11abece4ba07426a394b3174301306092a864886f70d0109153106040401000000305d06092b060104018237110131501e4e"
                + "004d006900630072006f0073006f0066007400200053006f0066007400770061007200650020004b00650079002000530074"
                + "006f0072006100670065002000500072006f007600690064006500723082023306092a864886f70d010701a0820224048202"
                + "203082021c30820218060b2a864886f70d010c0a0103a08201f0308201ec060a2a864886f70d01091601a08201dc048201d8"
                + "308201d43082013da00302010202102f5d9d58a5f41b844650aa233e68f105300d06092a864886f70d01010d05003020311e"
                + "301c060355040313155253415368613531324b65795472616e7366657231301e170d3136303431383131303532355a170d31"
                + "37303431383137303532355a3020311e301c060355040313155253415368613531324b65795472616e736665723130819f30"
                + "0d06092a864886f70d010101050003818d0030818902818100b2eca20240da8486b1a933ade62ad8781ef30d4434ebbc9b3f"
                + "c9c550d0f9a75f4345b5520f3d0bafa63b8037785d1e8cbd3efe9a22513dc8b82bcd1d44bf26bd2c292205ca3e793ff1cb09"
                + "e0df4afefb542362bc148ea2b76053d06754b4a37a535afe63b048282f8fb6bd8cf5dc5b47b7502760587f84d9995acbf1f3"
                + "4a3ca10203010001a30f300d300b0603551d0f040403020520300d06092a864886f70d01010d050003818100493d857684d2"
                + "7468dd09926d20933254c7c79645f7b466e7b4a90a583cedba1c3b3dbf4ccf1c2506eb392dcf15f53f964f3c3b519132a38e"
                + "b966d3ea397fe25457b8a703fb43ddab1c52272d6a12476df1df1826c90fb679cebc4c04efc764fd8ce3277305c3bcdf1637"
                + "91784d778663194097180584e5e8ab69039908bf6f863115301306092a864886f70d0109153106040401000000303b301f30"
                + "0706052b0e03021a041401844058f6e177051a87eedcc55cc4fa8d567ff10414669cb82c9cc3ceb4d3ca9f65bd57ba829616"
                + "60d9020207d0").HexToByteArray();

            public static byte[] s_RSA2048Sha256KeyTransfer1Cer =
                 ("3082031e30820206a003020102020900dce4d72a7f3ca58e300d06092a86"
                + "4886f70d01010b050030243122302006035504030c195253413230343853"
                + "68613235364b65795472616e7366657231301e170d313930313033323034"
                + "3133345a170d3238313233313230343133345a3024312230200603550403"
                + "0c19525341323034385368613235364b65795472616e7366657231308201"
                + "22300d06092a864886f70d01010105000382010f003082010a0282010100"
                + "bf8495ef84c91eb42ae9196c52aa2c6e22a258c4d76e09e93980d81fafe6"
                + "977fe1a3c8b4e9ca414405a1a4db6d549a4849189d568aa639f10fa556f5"
                + "5377c8fde8ec33ccdc833c4c5b8b36275e8f1899443871b7363f204a2826"
                + "8bd9e890fdd488b17bc49890f78fab5d8a5927b69655760d400a30e0ee87"
                + "dd38b4168df6814596e0efaafb2f9cc134664bb8629e65aa107847b4416b"
                + "d57f0b701c30e9a160aa589646e981815d9823201ae0ee601208c51c08d3"
                + "e50b1fab1560802916e678310c03e3cff3bcc039f98d5b254e2baf027ae9"
                + "f78faef72c2d93075eca64b5ad3e92348068c4dcb8cad0bad954a7d04fce"
                + "3aff94ef8cf282095db75a25a4114f8f0203010001a3533051301d060355"
                + "1d0e04160414fedfd2558b95e705584a50008ed0d1683aee840e301f0603"
                + "551d23041830168014fedfd2558b95e705584a50008ed0d1683aee840e30"
                + "0f0603551d130101ff040530030101ff300d06092a864886f70d01010b05"
                + "00038201010031c46689db724a1a8d268eaa7139efcf8a31496f15f325a1"
                + "9349547114d7992ead5fa0bd6ffd5539b1639f0cf8f975c5e9ebed8a0482"
                + "89c782ec7e0342f339f53e811cc6a6864173502388875964f0c50d84aba9"
                + "228dd2e3ddd3339374b76a6826f4f848a403adda2e095974232b084c8672"
                + "b12c18d675d81ed8fa4b15a1c7abc73f9ccb5bc25716eeb9d2c97f8ebcb9"
                + "d4824fb23c7fbc78e1d3b9b1b3749f6c0981b2b6cc81dc723aa5cc589346"
                + "dceca222c78f68fe80563e12282f7df936a161163f86a9eadba2e180a906"
                + "de73aebeea0775f6ab9ae97c71f26e187e3d8cb50c8137663c99d440877d"
                + "5a5e17df9785976e9031dd96134a07899281b6b64e3f").HexToByteArray();

            public static byte[] s_RSA2048Sha256KeyTransfer1Pfx =
                 ("308209710201033082093706092a864886f70d010701a08209280482092430820920308203c706092a864886f70d010706a0"
                + "8203b8308203b4020100308203ad06092a864886f70d010701301c060a2a864886f70d010c0106300e04081681e2854fdf32"
                + "7702020800808203807561ee3de7d54036ba61f41f76155bd406cd909358a12db40e07698863cc1143c77becf6dcb87a4a8a"
                + "701c8ff59021431a98ff6193c6d36e3bd7d11806c788042bc6f5dbceede60a1d6a314300ceca09b571e1a7842db92e19185b"
                + "e2408db8458d3e9127096b1f8a8c493aebc22bdfeb48bcd6d592b9db1d075fc7468cae8bf5bc01a0fa93bb2047981c5a4e23"
                + "f8c8e7795b1f94f5fbe2dbd03578ae2cdf13560bb73c219d258919f002217cb15a8b6e037da01c1de49b4c699c74c5396cf8"
                + "67bb308026c3a6b85da079de0aedecea0770a6b76128d185fc94a955789f3f26ffb99689b7311c7619ad24a33c6be16e022b"
                + "7216de49e1f11c9b4ded79aa438f535dbdb8bcc9f58ed2b3cd62515c5742f5e2ae0e9f791b83d4ebba5b3766e10194ae27f0"
                + "68960d039777c5ce71b11da9a4c791f612e06cddd2e474cfb8bda95faf88f6a854c26f6c787d20db07aa777f6ec2c208760c"
                + "67cb51cc5c79e13ba2760013d4ff54ab33a0aff355fea7aabd607a4fd8449d16173249be6785be5fa2e415932b6be65b483c"
                + "718e151e820bb4930f7df183cfa8c4a88b989a133ddb1717ea5b73f0950bf485de09d335347a16b96f02a4417e3671856c19"
                + "8b45462a21ce0df8f8e4c5178aae1999dc936aff91512ef9b7667de37e924118a6e958b111bf33088938987136143fa28ec7"
                + "216f77f4fecf87cfe3cd6af4d73a2e544d2f24a915de8d619faf0a4b891f09a9365bdd6dc48bc1eed864354e84c1bf0bc9dd"
                + "99a9e1280f6c81501424700960209ce0da9fd9e1b462d06673bb041439c8726f2b8890b90e859f717478637594675299c29c"
                + "da3f2dc6b6b9f1b72befc542407f180c89204774965c277fec78905b14b0ba54d19a9e8c3eb9c6ec6d056773a248908a6483"
                + "de20539b75de610c4ebdc77dc3a903a7e00fb1f6f46bfc97fe08a75b9d82bc4c536e4c71cf7c88d2eb57c50a97278ea5a254"
                + "1c5ee418f2aeccf800dd1dee31aacb4e6c679c58400757817a08ad7ec12e128f2e177bf2870d68c6d9fb4b2987abee13bfc1"
                + "7b529dd00edc8eea55fc8268e02dcbd84e7cb2cfe1ae8d4c1fb0a486871e18ddde79759ad0127e5dac938a105aaee9368901"
                + "186c149424ec992b57fe927b1f6c35142229e3d427a0d86d454d302c3cf4fa4f1b75c42813e8029a1428570aa1903acde079"
                + "6d0d2259780bf4752a70788fc642c372009f57cd430ee2edc01954e5a615744e7cd3bd8784e3c3b80b0b63847464207f92ec"
                + "821dd564f43082055106092a864886f70d010701a08205420482053e3082053a30820536060b2a864886f70d010c0a0102a0"
                + "8204fe308204fa301c060a2a864886f70d010c0103300e04084dcda62e7b3f05f602020800048204d87345553aefe9a5c6d8"
                + "488c069190343bd43adcb7a3058066b12b01f58d238b517627b440b704a1944955c4f97b59eb0bafb4d6ea35f6d931773013"
                + "e0202a1229659ab32469edd459a7712757457a5a967af3a4328d772a87ca46076c76dcbcb3671be555969826e1f94d2879be"
                + "9f464bca6d970cc31ac485ff50d2f055a1cf1cfe1aa0a14e166a85bf27c9d5f8d2ba876e54d737936a6f5f61d6e2dd7cf4e1"
                + "3bd34e397809cf62de6d793944d37c82b180bf98d1ffd7b345174883864fd31b80fb598ba4eb63c1f7ca9c676ec0c38f2306"
                + "7329f63b28ca0dd17f0f78caa00d1953d818bf198f769095ed84a8fa50888ee3327ffcd2eb508545dc961435ced61d5af9d9"
                + "7cdae7fa44ab658ecbf7bc4656deb32cdb8fbf45538c21d653f89cdeed08f84a732cd1a9f3f5d27231daf06fb80e8521dec5"
                + "51331466c851d133648e9ed978aca042d66a8dd224b6bd3ebd861302c1c096247a193beb6fc37a50c6b79e3751bf68b4bf45"
                + "2ff71bfa07ae20d22d3003c92bd1f7907850c0a689ed2ed15466f23e164d5356cf8c0fdf22c109e2de5178a3e58216402f6a"
                + "ffffd911924dfebe0d771b4056399753972ed7006f0e89ca0a6e256c8d439c0199bc2407e2ddadf820edb464cee1cf059d9a"
                + "eae490493dcbc3c59c1bdcc70059a636a6d6092fe554efb616bcf653b4d673e025569fc95218ce99aa48bd01832f76739f30"
                + "05ddb6616e0f42cf236f30aacd261afd567765b5b2e8363d94e871f32755b8b2a9170d40ecfe497f88c4287550df80a9b802"
                + "2560d6137f6f4641c8479535e4f2c2f66f60843623f30a8201422c797b7ba8df24440249571203eacc6d362c2c87c417f854"
                + "dc64194e8e8566a414bb32d2eb8a316296286ed36767bdc34c97ee9d891c6d28453b1e83a525b8dcddcb20510c913acea8e3"
                + "3053784a36097f8cd2984c8ee6623c1e0875046769cda607cba996417ee97b788819e476425a9b5369bd8affcc867c5974ba"
                + "639a5c2d1ad33d63776a06b040e364233e9407f7004200fe1716a55ca6871f5e68033a8c0b5d1392df55f39ee2c751bcacc5"
                + "3c8347870e453c2fa4e0abbf3ed9a668bdb2839ca9529292a9545db18af6d0213d67fca687d164a7fce7a7c1482f3788bf8e"
                + "77e5d820f75a975e3b4b00a4ee9d2a21cb4eb81c5e31a48d82f7c09b92eff40a83dc0e5528c2f90442ef85e47e3af680ac74"
                + "4117add8582e293b78d078f0f07307453be0e7a64ff9cae27eb51fb14eb5b8760643ad1204b352a9a9772f419f92a37a0ce9"
                + "2f002b5a34b97bbd9df209a090df38fe67244fd01e19b92fa881d00b70c6083480c12483a2fe1103c1c12f3148819728ce30"
                + "11678d6711452bda6163911d2c29168f3e915f3b7e9afc6fa8b7d1e21c32db0252f784f33341b87107af7c783a05270ef2de"
                + "48175e147683e515fa97f4d3fe8dd77b1d4e24e6049fbf1ed9d08d0dc6b2be65554ba116b25e3cb1462f3aaf1fbd60624909"
                + "a579a726f6176552000e84ab323a88f23a630985685e3aeb0b08c9f027f717257c595d053f257d6e5f2ecfd30903bdc35868"
                + "f4b20f86938102d22c782713e3ea01d67bfa0ecea9668b6b15de3198adb4e87e438634cb3fa3ea7e2c195d9616dd2b42e107"
                + "3e29e94dc9067f2794f25f8a78108db5c99a51298353cb7c1c0352d76f19ea6222e03836cc7bcf9f8873b07d4e0ba845b3f2"
                + "f19eb36e9cc8215f28f2b911f772d4330340fbb6f8cc1840241720074c0c743125302306092a864886f70d01091531160414"
                + "c6fe1078ea0dbc7c8d7f96ad3b7157625c8aab8c30313021300906052b0e03021a05000414bd17b7ccb644846ca45544381e"
                + "b780abf0ccc2650408e4b108ac25f3820e02020800").HexToByteArray();

            public static byte[] s_DHKeyAgree1Cer =
                 ("3082041930820305a00302010202100ae59b0cb8119f8942eda74163413a02300906052b0e03021d0500304f314d304b0603"
                + "5504031e44004d0061006e006100670065006400200050004b00430053002300370020005400650073007400200052006f00"
                + "6f007400200041007500740068006f0072006900740079301e170d3136303431333132313630315a170d3339313233313233"
                + "353935395a301f311d301b06035504031314446648656c6c654b657941677265656d656e7431308201b63082012b06072a86"
                + "48ce3e02013082011e02818100b2f221e2b4649401f817557771e4f2ca1c1309caab3fa4d85b03dc1ea13c8395665eb4d05a"
                + "212b33e1d727403fec46d30ef3c3fd58cd5b621d7d30912f2360676f16b206aa419dba39b95267b42f14f6500b1729de2d94"
                + "ef182ed0f3042fd3850a7398808c48f3501fca0e929cec7a9594e98bccb093c21ca9b7dbdfcdd733110281805e0bed02dd17"
                + "342f9f96d186d2cc9e6ff57f5345b44bfeeb0da936b37bca62e2e508d9635a216616abe777c3fa64021728e7aa42cfdae521"
                + "01c6a390c3eb618226d8060ceacdbc59fa43330ad41e34a604b1c740959b534f00bd6cf0f35b62d1f8de68d8f37389cd435d"
                + "764b4abec5fc39a1e936cdf52a8b73e0f4f37dda536902150093ced62909a4ac3aeca9982f68d1eed34bf055b30381840002"
                + "81804f7e72a0e0ed4aae8e498131b0f23425537b9a28b15810a3c1ff6f1439647f4e55dcf73e72a7573ce609a5fb5c5dc3dc"
                + "daa883b334780c232ea12b3af2f88226775db48f4b800c9ab1b54e7a26c4c0697bbd5e09355e3b4ac8005a89c65027e1d0d7"
                + "091b6aec8ede5dc72e9bb0d3597915d50da58221673ad8a74e76b2a79f25a38194308191300c0603551d130101ff04023000"
                + "3081800603551d010479307780109713ac709a6e2cc6aa54b098e5557cd8a151304f314d304b06035504031e44004d006100"
                + "6e006100670065006400200050004b00430053002300370020005400650073007400200052006f006f007400200041007500"
                + "740068006f00720069007400798210d581eafe596cd7a34d453011f4a4b6f0300906052b0e03021d05000382010100357fbe"
                + "079401e111bf80db152752766983c756eca044610f8baab67427dc9b5f37df736da806e91a562939cf876a0998c1232f31b9"
                + "9cf38f0e34d39c7e8a2cc04ed897bfdc91f7f292426063ec3ec5490e35c52a7f98ba86a4114976c45881373dacc95ad3e684"
                + "7e1e28bb58e4f7cfc7138a56ce75f01a8050194159e1878bd90f9f580f63c6dd41e2d15cd80dc0a8db61101df9009d891ec2"
                + "28f70f3a0a37358e7917fc94dfeb6e7cb176e8f5dbfa1ace2af6c0a4306e22eb3051e7705306152ce87328b24f7f153d565b"
                + "73aef677d25ae8657f81ca1cd5dd50404b70b9373eadcd2d276e263105c00607a86f0c10ab26d1aafd986313a36c70389a4d"
                + "1a8e88").HexToByteArray();

            public static byte[] s_RSAKeyTransferCer_ExplicitSki =
                ("3082033E30820226A003020102020900B5EFA7E1E80518B4300D06092A864886F70D01010B0500304D310B3009060355"
                + "04061302515A310D300B060355040813044C616E643111300F060355040713084D7974686963616C311C301A06035504"
                + "03131353656C662D5369676E6564204578616D706C65301E170D3136303632383030323034355A170D31363037323830"
                + "30323034355A304D310B300906035504061302515A310D300B060355040813044C616E643111300F060355040713084D"
                + "7974686963616C311C301A0603550403131353656C662D5369676E6564204578616D706C6530820122300D06092A8648"
                + "86F70D01010105000382010F003082010A0282010100D95D63618741AD85354BA58242835CD69D7BC2A41173221E899E"
                + "109F1354A87F5DC99EF898881293880D55F86E779E353C226CEA0D1FFCC2EE7227216DDC9116B7DF290A81EC9434CDA4"
                + "408B7C06517B3AFF2C9D0FD458F9FCCDE414849C421402B39D97E197CA0C4F874D65A86EAD20E3041A0701F6ABA063AC"
                + "B418186F9BF657C604776A6358C0F031608673278EFD702A07EE50B6DC1E090EEE5BB873284E6547F612017A26DEC5C2"
                + "7533558F10C1894E899E9F8676D8C0E547B6B5C6EEEF23D06AA4A1532144CF104EB199C324F8E7998DB63B251C7E35A0"
                + "4B7B5AFFD652F5AD228B099863C668772BEEEFF4F60EA753C8F5D0780AAED4CFA7860F1D3D490203010001A321301F30"
                + "1D0603551D0E0416041401952851C55DB594B0C6167F5863C5B6B67AEFE6300D06092A864886F70D01010B0500038201"
                + "0100BB7DDDC4EEC4E31163B84C5C3F4EA1B9314B75D4CCCA8F5339503DC6318E279E73826AD77AC6EA4344F2905D8792"
                + "83D867A66319E1EA383B70CE997A98D545FE258E38C577554E3BBDF42349B98BED9C94B7CE55D51F740EF60ACC75D017"
                + "F3D0D1438F8F86B2453F2308279521EE4AC09046498F60ECEC8867E7BF011C882A514BF85F6C915A70E0383AB5320034"
                + "19A107A9FFEDBF34681AEEE57DF6A3DB3759D605A9269FB694D8EA56A90740529159D725BFD70C9141A38B98D4E88CDC"
                + "31124ABBB4C3D3D49C220CCB6F2F94176B8225A0E2ADDB0F4A72E6B021601CD297AC45A0CAB95EBAC4001C8167899868"
                + "3188DB9364AAD52D4E28169CC898B621FF84").HexToByteArray();

            // password = "1111"
            private static byte[] s_RSAKeyTransferPfx_ExplicitSki_RC2ContentEncryption =
                ("308209810201033082094706092A864886F70D010701A08209380482093430820930308203E706092A864886F70D0107"
                + "06A08203D8308203D4020100308203CD06092A864886F70D010701301C060A2A864886F70D010C0106300E0408101C5A"
                + "3E2DBE2A9102020800808203A0F58476F5E4741F8834F50ED49D1A3A5B2FC8C345B54255C30556B1426C1BA1D9EE4440"
                + "CD63CD48557B7BDC55D877D656183E2815DEDE92236E036E0D7FD93022174EFA179A85EF76DEE10950EE3BEB004FB118"
                + "C58D4372A319575DB129F5912B385E63E1E83DC420A8FC8C23A283977480281EDDD745D97EC768328875D19FE414D7D9"
                + "9D3B0AAA2FBA77346F82E4E1357C54E142B2F5E929CBD6057801F49ED08A8BD2456918CCEDAD6DAD9A7281C4EFD2FCF5"
                + "6F04EDC5E62E79741024AF8BE401141AA9A9CE08F5D51D4636D8B25F9B3C59B4BC2DD7E60FBABA0A7E8FE15EAECB7221"
                + "3BC22D8CE56987427B41A79333FB4B9BC5DB6E79C2AE4E954F975C343D155C5587BD7206414B9C0529D00C6DB1470C33"
                + "A51D2A9BBDE5CC2352C61B0FB9945487FDB0E26981426BE7CCF44CF494E695D4760060468B7D23BA3C6F9B1762AC4B3A"
                + "428B23A36F275F3FDFD7BAB236197C7C7FB6466A11B05DB39F947FB19EFE9BFED2B18308E2BBD0AB00AA399508194CB2"
                + "1073B1B278BE389A8AA843B610B439AFA056A0EC81EBDF4061D2AB87C9CB840C3E6B92BB2FC30815D5744593862CC34A"
                + "EF1C4B7BBCF640CBA2D1E15E13D3B571FD3C403BC927812B291E53EAE6721C994E343148C10A16053AE560A55DFA5695"
                + "33CA35D83D81643CC7788E7F94C6592F99C09AFB770E9FE1380A1212A646A936BE531BF85F89D19EF57C180E8E3F1F4F"
                + "BD266032095862E3A0F8394E93CEFF2B8ADAD374DFCB8A041DB498618D1D71765EFD1CD5B024AC13B9FF0F96F975B443"
                + "08C14AC60965082CC409AE43D033512CF1B83458D475D2E06A49131894F1D4BFAF5FC4CBADA8566B6312E8DA31D8A397"
                + "273BE77B8395F4CAB4428B22DFE18FD4365C134B7724220D2DCE31938EFCF8E4DFC321E02CF15476BF5EB675F2055205"
                + "9662166A4549904BC6A5E4B8353C43DAC225317F4B4FA963C900F0B0D0E7FC854BE91A1CFF330FE77B03776EABD0326B"
                + "0FB37AC5176CF82530960F423B13299E037285C9324E0A872414ECF35735F58463506EBFB2CC91D790FC0D67E2714287"
                + "960C68FB43A7EE42A14F5397F07E055E75EE4F7D047634907702EEC8ABB08D82C82CEBE2B042B1F20367DFDB839B82AF"
                + "88F72272AE91DA94CD9B334343196889381FE307A76BE0B627EE32D827582A7CD68BF467D954805030753FA8DABFCC21"
                + "E68A77E2A76F9E95E61A2FBCA1C8FFC2CE272E9D125E5C65759529BF3FDD2E28722EC9B7B57BD9819BAAC01556002D88"
                + "3B8BD842C3EB3BCC4A54B4D0B1DB32ECEBA8DD668D67C859A6EB0BAE293082054106092A864886F70D010701A0820532"
                + "0482052E3082052A30820526060B2A864886F70D010C0A0102A08204EE308204EA301C060A2A864886F70D010C010330"
                + "0E0408482E476C7712FD7202020800048204C8AF5357D64578320D963917F44B2B7714334AAE6571554F03A599913275"
                + "4BA03892316159385225C4EEA7C958091BC8A32A9433AECA3A07F791ACE431217F0DFBD53DC15D257069B99DA04EF719"
                + "892004FD307E151EBB359C1D69AE8FF78A1CC1654470B0862CDAC1AED521608C29AA8D30E81A4015D54F75421B9BDB29"
                + "5036D79E4535F28D4A2ABF4F203BC67065A709AEF6EAF4D6A3DC7387CB3459D82399F4797CE53FD2BD0A19B1A9566F74"
                + "246D6B6C50BD2777B6F6DE1A8C469F69D7EBF230018D56DF4C1764862CD982E81F56A543DA4ADB63EF8612A1BB561471"
                + "56035541B0B41F06BBE2CD47DC402A75023558205870060438CF99D8BFC7CAADDE0583311FE4B854051C83638420BC5E"
                + "93F999E67EDBBC266F519609E2BE9FC1BC3C7FEE54DBAB04DAC8A94BB347F3DC28DDAB7D28DD3BBFFB44C84E6F23A8E0"
                + "1CAB36382723DB94CD8973A303D8D4C7A22B9F811C07ED9A78E06135E0426FC93BB408F1DC915DF4ADBF048D22C201D8"
                + "0FDC0EF942D1E2AC0F39F8A95BA849C07BB0DA165B3F0317478870F704B8A34B7D5816BC4F8CA0C6BDB5346F2211416C"
                + "79D7117AD1B86E44E0BC0C3793F9895638E5B7A2A5B957E0E691819AC7FA1F05E8D05ED99E4E286C96E3E31DF99633E8"
                + "CB73CA135109AE727CB047641726A1599492B6F3E8E62195A79239279B2A9FBF47B31FEFF3C20DEC2DFBDB0CE98B183D"
                + "BA773561DEE404BA1A5BEF5AB9729DBE22FB1C17EFD4D3AC81F04F49F9855CEACECB202090A1290C10E9D676F0658F3D"
                + "E4C43DCD5A17B88881893DA87060C5F56D5CC9A92E6B1A47A6D16FB32C09925606F6D5C7CAFBC7A82D8E441A05DFBEE0"
                + "BEC92D89264B62D5DECC342D29D9A7727BBDE4E63EEB7CAED7C76953F6AC8CB570619C7607B753FD46889C76D29C9AC6"
                + "6F56CB3848323FA9CD16578EA5C6D876AE63D95F65E2CDEF68A1CF3D2FC3DF91D0055B0CDBD1510E291C0E7AC6EAA0D2"
                + "AB5E8FAD44108C94A592251447926DB7139BC2A433D61711C6DA5EF82A8E18CEBF99AF753E33FFF65126B7D3D3D09FF0"
                + "C50EFF7822FA8797BAC52383B94E8FE602E62577994ACA6A2150F60A31CA0E79FE6DF3405D6918EADC2231558FB29045"
                + "034EB9DA9FB87BD71996C6AB6EA71A70EBFBC99BC292338A363176516C14EC92E174C59C6BE82F5BC0296525109C9A7F"
                + "C9D9E654955992A5C9EDFD39ED9889BEAF105B2EF62B041789F20A6AB26563FCFA1A1482EE2A20E8C1A2F0931ACBA7F8"
                + "756EE4C9119D29817ACA7D2B81FE736FD7A33D20EC333AC5123D29345647B734DB24B5C56B4576ABBF9B02F782DDE0B4"
                + "BA277080F28F3D86DEC35F0F19B2B5DB0BD7A59B7C4B2BAE08E8584449BD3685F371F6A24F5F91EA6843DC6ABA89976E"
                + "589511EB23A733D23F6CE076C952E9E7190ED78D5A34387F93418A87CB02270941F19DD35E1DB5836E67296A7F28A5EB"
                + "8F32DA73EA7A47D5CEB292E767468CDF938A44B3CEEE6276A34705606A8F7496D7310DA1A0468A604B8A9E7AB50450A6"
                + "DFE0C4540CEA058CD7919E823D8A32FB811C6BF8754C65D7A9723018ADE95AED5C30978A8DBA185CF0BA36346456CD3E"
                + "15C511FAD71B445DDFA7C5455A3597FE536E3BB87518C0725D6BE673D05DC5E74B4FF442711D242A37D0CCB88E6D19BD"
                + "6B7299207D7036EB87D5E86189768CB14AE4F8886BB5AB7864BDA9757D0C26BFFF3FAA4001258557D394313125302306"
                + "092A864886F70D01091531160414080FB9AAB81BD67FD85C2186B359054CEB13D2D730313021300906052B0E03021A05"
                + "0004142C205F0B1E9B99B0ED14E83F13D84BC683F66D3B04080D22E45D6A657CC602020800").HexToByteArray();

            public static byte[] s_RSAKeyTransferPfx_ExplicitSki_TripleDESContentEncryption =
                ( "308209930201033082094F06092A864886F70D010701A08209400482093C30820938308203EF06092A864886F70D0107"
                + "06A08203E0308203DC020100308203D506092A864886F70D0107013024060A2A864886F70D010C0103301604103C01F0"
                + "817F2D7818D335FD48397D829C020207D0808203A004485AD45419862A564E9C8161C9A6BCB94EC72C3D18CB01C499D4"
                + "F157C055EF92E915B85A15AD0F175DF99D8B3B0516AEB95D774FABFBAA84EB8B2331FE920D139BA5AB4CE23CE7F04EB3"
                + "493F390B977EBF502AE81703C28CF9EEDD7247A8B930AFCB866FEFD08683270F75905D6FD456B4C6F87544F89329A5C6"
                + "16991E60CBA5EC8287FB69A2BC85564CC5739D9416D20BEAE1238FA9FDBF1DC552AADD74DE777AA6D1C096D61F3354CC"
                + "B920825812A4F4E886C4CDAA86DC4C3E77797240D66ACB4F4705743CB05E56FEEA1911A896AD1AD4A9A451F217CF2634"
                + "1950C165577FB6E0200DB77A84679266D957B60E9B77F5690B44988C0D3087C945FE1A2FC0A8E5E7C186E72F670E98F6"
                + "911F76F6FBF076CFC1149562AF67D132FDEE8F207DF89BDCFF79C5D3EA0D2B710071C47C36F6FFED6874BB2013B78207"
                + "1F331F411D894BD2E140087AD293514C272145F9E51D0E8B390920BF89984C5567D5A2BAB726CE60DDB2F1E73F2276DB"
                + "3ED142612C76717C3A97DD63B160A6560AF4EC65D6F8FDBCCF9ED03CD006570354C19CBC70BBA9896A15A02B6FBCB215"
                + "FE771DCDA5BB4CDC4E86732776527C3A1AA01EE86D0C2B7BC525D19BE430270DD1AE7E0ED02174FF27974AB9BF9EBA91"
                + "3437B743B7B6E7BAF80CE9D8048136F564B03813ABDF69291DFC8D215E85C0123A6E0DE0CA6202038ABB0F650ED7FE48"
                + "CA366408DEB344FD81A6E1ED1DB6D4AAC97498D957734632C6F317076703EB2EDBB94B1F9D1D3049ED95BFE4AEF89CB8"
                + "C51D73B5833DA7E2A9CA365B5D3F8BD2F8014633BB1B112FF42D96D0B9F5C002BBAA499509086E37A54F0875BC32CC61"
                + "EA61650E5A251214BE6E9249DD1DA4B64EAAD9B389D8D7B5A3D9887A2A30F147AA931D0DEDA93D7D5F19B84AD2259D00"
                + "7692FDCB6963FBD8F5897844A5C154B81C410EEA5B2DE770AF4335966B5B33354AA5C4C7C124C002B134EFFAC308B92F"
                + "8034618213D3DEB805E6297A6EEB1AE16A27C55B01AAA68AB25823484B41C75B607F9F056484E8FA1C2D5AADB0BB4BEF"
                + "3732E447CD510FA4813395D1147BDBFFBE33A1B9AA3A466A825DE6A8D0D0DC1B869553766039EC6FE366711B2BFF407B"
                + "9C72A0939CB6D7CDE61CE7AC50CCE24A2B1796517FA4F0B0CE43B226F12BEC1837A7F0F4C467393A057BC9A2AE530349"
                + "54DC10CE645D7EBC6E747F5DBC6695C49AFDFDC3C41B7236A8017BA16027E46A1EF3BC93C232DC8C19E3A8BCD71754AF"
                + "A56AF755C0CFE8BC50ABCECD3F7ADDDE9B741614DF86A4C4BE9F297663CAE52C1922D41C153082054106092A864886F7"
                + "0D010701A08205320482052E3082052A30820526060B2A864886F70D010C0A0102A08204EE308204EA301C060A2A8648"
                + "86F70D010C0103300E0408482E476C7712FD7202020800048204C8AF5357D64578320D963917F44B2B7714334AAE6571"
                + "554F03A5999132754BA03892316159385225C4EEA7C958091BC8A32A9433AECA3A07F791ACE431217F0DFBD53DC15D25"
                + "7069B99DA04EF719892004FD307E151EBB359C1D69AE8FF78A1CC1654470B0862CDAC1AED521608C29AA8D30E81A4015"
                + "D54F75421B9BDB295036D79E4535F28D4A2ABF4F203BC67065A709AEF6EAF4D6A3DC7387CB3459D82399F4797CE53FD2"
                + "BD0A19B1A9566F74246D6B6C50BD2777B6F6DE1A8C469F69D7EBF230018D56DF4C1764862CD982E81F56A543DA4ADB63"
                + "EF8612A1BB56147156035541B0B41F06BBE2CD47DC402A75023558205870060438CF99D8BFC7CAADDE0583311FE4B854"
                + "051C83638420BC5E93F999E67EDBBC266F519609E2BE9FC1BC3C7FEE54DBAB04DAC8A94BB347F3DC28DDAB7D28DD3BBF"
                + "FB44C84E6F23A8E01CAB36382723DB94CD8973A303D8D4C7A22B9F811C07ED9A78E06135E0426FC93BB408F1DC915DF4"
                + "ADBF048D22C201D80FDC0EF942D1E2AC0F39F8A95BA849C07BB0DA165B3F0317478870F704B8A34B7D5816BC4F8CA0C6"
                + "BDB5346F2211416C79D7117AD1B86E44E0BC0C3793F9895638E5B7A2A5B957E0E691819AC7FA1F05E8D05ED99E4E286C"
                + "96E3E31DF99633E8CB73CA135109AE727CB047641726A1599492B6F3E8E62195A79239279B2A9FBF47B31FEFF3C20DEC"
                + "2DFBDB0CE98B183DBA773561DEE404BA1A5BEF5AB9729DBE22FB1C17EFD4D3AC81F04F49F9855CEACECB202090A1290C"
                + "10E9D676F0658F3DE4C43DCD5A17B88881893DA87060C5F56D5CC9A92E6B1A47A6D16FB32C09925606F6D5C7CAFBC7A8"
                + "2D8E441A05DFBEE0BEC92D89264B62D5DECC342D29D9A7727BBDE4E63EEB7CAED7C76953F6AC8CB570619C7607B753FD"
                + "46889C76D29C9AC66F56CB3848323FA9CD16578EA5C6D876AE63D95F65E2CDEF68A1CF3D2FC3DF91D0055B0CDBD1510E"
                + "291C0E7AC6EAA0D2AB5E8FAD44108C94A592251447926DB7139BC2A433D61711C6DA5EF82A8E18CEBF99AF753E33FFF6"
                + "5126B7D3D3D09FF0C50EFF7822FA8797BAC52383B94E8FE602E62577994ACA6A2150F60A31CA0E79FE6DF3405D6918EA"
                + "DC2231558FB29045034EB9DA9FB87BD71996C6AB6EA71A70EBFBC99BC292338A363176516C14EC92E174C59C6BE82F5B"
                + "C0296525109C9A7FC9D9E654955992A5C9EDFD39ED9889BEAF105B2EF62B041789F20A6AB26563FCFA1A1482EE2A20E8"
                + "C1A2F0931ACBA7F8756EE4C9119D29817ACA7D2B81FE736FD7A33D20EC333AC5123D29345647B734DB24B5C56B4576AB"
                + "BF9B02F782DDE0B4BA277080F28F3D86DEC35F0F19B2B5DB0BD7A59B7C4B2BAE08E8584449BD3685F371F6A24F5F91EA"
                + "6843DC6ABA89976E589511EB23A733D23F6CE076C952E9E7190ED78D5A34387F93418A87CB02270941F19DD35E1DB583"
                + "6E67296A7F28A5EB8F32DA73EA7A47D5CEB292E767468CDF938A44B3CEEE6276A34705606A8F7496D7310DA1A0468A60"
                + "4B8A9E7AB50450A6DFE0C4540CEA058CD7919E823D8A32FB811C6BF8754C65D7A9723018ADE95AED5C30978A8DBA185C"
                + "F0BA36346456CD3E15C511FAD71B445DDFA7C5455A3597FE536E3BB87518C0725D6BE673D05DC5E74B4FF442711D242A"
                + "37D0CCB88E6D19BD6B7299207D7036EB87D5E86189768CB14AE4F8886BB5AB7864BDA9757D0C26BFFF3FAA4001258557"
                + "D394313125302306092A864886F70D01091531160414080FB9AAB81BD67FD85C2186B359054CEB13D2D7303B301F3007"
                + "06052B0E03021A04145418AC4FCE987C52004DA05A858BFEA94B21D4860414A26BFC44B061FE0F7E01ED25D963641D1C"
                + "057847020207D0").HexToByteArray();

            public static readonly byte[] s_RSAKeyTransferPfx_ExplicitSki =
                PlatformSupport.IsRC2Supported
                    ? s_RSAKeyTransferPfx_ExplicitSki_RC2ContentEncryption
                    : s_RSAKeyTransferPfx_ExplicitSki_TripleDESContentEncryption;

            public static byte[] s_Rsa2048SignatureOnlyCer = (
                "3082032C30820214A003020102020900E0D8AB6819D7306E300D06092A864886" +
                "F70D01010B05003038313630340603550403132D54776F2074686F7573616E64" +
                "20666F7274792065696768742062697473206F662052534120676F6F646E6573" +
                "73301E170D3137313130333233353131355A170D313831313033323335313135" +
                "5A3038313630340603550403132D54776F2074686F7573616E6420666F727479" +
                "2065696768742062697473206F662052534120676F6F646E6573733082012230" +
                "0D06092A864886F70D01010105000382010F003082010A028201010096C114A5" +
                "898D09133EF859F89C1D848BA8CB5258793E05B92D499C55EEFACE274BBBC268" +
                "03FB813B9C11C6898153CC1745DED2C4D2672F807F0B2D957BC4B65EBC9DDE26" +
                "E2EA7B2A6FE9A7C4D8BD1EF6032B8F0BB6AA33C8B57248B3D5E3901D8A38A283" +
                "D7E25FF8E6F522381EE5484234CFF7B30C174635418FA89E14C468AD89DCFCBB" +
                "B535E5AF53510F9EA7F9DA8C1B53375B6DAB95A291439A5648726EE1012E4138" +
                "8E100691642CF6917F5569D8351F2782F435A579014E8448EEA0C4AECAFF2F47" +
                "6799D88457E2C8BCB56E5E128782B4FE26AFF0720D91D52CCAFE344255808F52" +
                "71D09F784F787E8323182080915BE0AE15A71D66476D0F264DD084F302030100" +
                "01A3393037301D0603551D0E04160414745B5F12EF962E84B897E246D399A2BA" +
                "DEA9C5AC30090603551D1304023000300B0603551D0F040403020780300D0609" +
                "2A864886F70D01010B0500038201010087A15DF37FBD6E9DED7A8FFF25E60B73" +
                "1F635469BA01DD14BC03B2A24D99EFD8B894E9493D63EC88C496CB04B33DF252" +
                "22544F23D43F4023612C4D97B719C1F9431E4DB7A580CDF66A3E5F0DAF89A267" +
                "DD187ABFFB08361B1F79232376AA5FC5AD384CC2F98FE36C1CEA0B943E1E3961" +
                "190648889C8ABE8397A5A338843CBFB1D8B212BE46685ACE7B80475CC7C97FC0" +
                "377936ABD5F664E9C09C463897726650711A1110FA9866BC1C278D95E5636AB9" +
                "6FAE95CCD67FD572A8C727E2C03E7B242457318BEC1BE52CA5BD9454A0A41140" +
                "AE96ED1C56D220D1FD5DD3B1B4FB2AA0E04FC94F7E3C7D476F29896224556395" +
                "3AD7225EDCEAC8B8509E49292E62D8BF").HexToByteArray();

            private static byte[] s_Rsa2048SignatureOnlyPfx_RC2ContentEncryption = (
                "308209E3020103308209A306092A864886F70D010701A0820994048209903082" +
                "098C308205BD06092A864886F70D010701A08205AE048205AA308205A6308205" +
                "A2060B2A864886F70D010C0A0102A08204FE308204FA301C060A2A864886F70D" +
                "010C0103300E04083EF905F0EA26FBF7020207D0048204D82297B5546DA6CC49" +
                "BD8C1444E3FE1845A2C9E9BDB8B83E78235DF4ADF7A97496A62D31D4EEEB76B0" +
                "71C0B183ACC3663272F88CF4F31E2E00D76357C0A051B8D6E0BB0BCF4CCDD064" +
                "CCBAF546EABA80DA56CD11FE952C61154792559D65F26B0476CF7A5FDB8CC794" +
                "B89F6ACD50003459054FE82C48D8791B226A0EEEC01F048AC3CE716C9F3BB313" +
                "D64BEBBF0037D83133DD9C15D04F15BB11652D793B613A68AFE580245724E5D1" +
                "110040B332B5C39BE04086BA4DFC58E905BC2FE8B3C696181E2879AF197EE24D" +
                "91D8AD67013F14C4864C8D0FB19C134B766CF3E48B8C9E363A11EB19F1E82E74" +
                "D25EDD96517D64A94314B40C11651030D561E742E63856E8D1A3EE9FDFD6CF64" +
                "7140CFC354AE7EA1D14C157C2985F82D54296F7D3DE456AF7513F5F30A0421E4" +
                "3A9DAD6DF8A2BF69005B35CA8066F80755D848DA73EF03BC0CC129C5799911D9" +
                "3A1ED43F8E76732AF56FD62DC6D0B0DBA6AAC6DCDE77D0E8AC9F6A5EB5A02B61" +
                "BC477706D4F1873240AB45E0291EF21E6034D48F1AE8EB139DE7ACD8B8A821E6" +
                "70B395C3EC4B0E75C34BF0067F052FCED835CAC1F17C3FBEA2FC9FD281FCDE21" +
                "D5B27CF31A07E90164A979ACEF0E1C67DBA6C33082E9B189D4BDA2D2D504776F" +
                "455843437BDF10D4AF48639EC03344BCC36EFEA7CDE08D7F94DDEAF98BCB5D65" +
                "207AE0C349EECE3032DE19F3B337F29A3457AA0AAF4306FA8301619AB01B7235" +
                "BE16CB93728E142DAA6C1CBBCC5BD82D913596994DA40FB916CF2DB5FBCC20CF" +
                "E893DC62BBC5FC59E00BC0A704A9DB25BBF9291971F2377FC1A20F2C954869DB" +
                "6FFCC90C625AABE97ED4CF7C0209D39AD780003C437152D636ACB3B484C46885" +
                "DC1584733D2153A3F9B968F12CDD5937CDF9DD2581D72EE67C83AE2530197AA7" +
                "C6110613BEFF0B75E586C28394EA8EBCF7F9DB133295B33DC86C8DBA92BF8BD1" +
                "ADCAF8D2CD2E018B08D59FF1C30A13484AB11468F7DCEB1FE53A6DAF309B0510" +
                "7772CB735314A5B2F053E60A653F0496BCB9CADF5E50339A4D2EF2382056B768" +
                "558EB9230D996C636E6D29664F92F70A088DE3EE4EC4BBD8A9C4C98C7892D122" +
                "28806622B87E581A321AD835B8F4B964A17B5BE6D9DA50133D494732A41884E2" +
                "9E891FE2D40ACCFD585C8BF626C1E8A412D2EE7CDE060E2CCDA826BF79D80F1B" +
                "F6B8400473BCE0C19D03ACF55D1FAA994C04A8CD11D49743B1F45F48DFFDD701" +
                "18B5FA82ECDF67714F5DE5D3D3DDDCB76ED0EA6A6E151665A4AA351DB1A99F8C" +
                "7502D3795C2C358CCA589C390C1F4810615130B91BA42A85E77FA37197E1B083" +
                "FE1246B067C6444D49B369D45B130A6D7B463C3F0459EB41D68009CABD2F5C60" +
                "49DB706FA742C9773FB5791AF123FBE485E05F87759ADD25281BE337B6095EC3" +
                "4EFF9FC692798FB4217EF4B2B59902D930F28181933FAA278C041123CAE3CA63" +
                "6DFD3AD4E04EB751A30D50C26288EA4D01C7B323E4FD6387F88E020BC433BF60" +
                "C4406398C44EA5C7A6EB24134B0811E4F94DFAF5553172306FA5543C254E7E04" +
                "DEEC84DBF9FAF7BFEA8D61E094CBB18DD45C5BAB9199DD719F9A305E205605CC" +
                "671DCD566FEBA2C8F4C1A445625C4F42D1CFE32087F095591798D1D48DA46DE9" +
                "230F5102B56A1EF879D48936D5331D6B3D9F1B564CF08FD3C641CFF3B02CB4FC" +
                "8995E5EC5DD1D183704940C02DEA7430FD594E54800DCC74B7732731C63FBBA2" +
                "A2F6DC031174390A74781D352B09FB4F318190301306092A864886F70D010915" +
                "3106040401000000307906092B0601040182371101316C1E6A004D0069006300" +
                "72006F0073006F0066007400200045006E00680061006E006300650064002000" +
                "520053004100200061006E006400200041004500530020004300720079007000" +
                "74006F0067007200610070006800690063002000500072006F00760069006400" +
                "650072308203C706092A864886F70D010706A08203B8308203B4020100308203" +
                "AD06092A864886F70D010701301C060A2A864886F70D010C0106300E04087CB7" +
                "E0256AD1AD80020207D0808203800C21CEBEF9F3765F11A188B56702050E3DCA" +
                "78AA27123654D066399DD56E62187C89A30940B5B63493950EEFA06C04B5CAF0" +
                "329143AF30EE0B47406E49D4E6241817986F864780B743B58F03DF13523F5C01" +
                "C889046356623AFA816B163E57A36672FAC9CA72294B2A17F75F5ADB1A4CBDB7" +
                "B3F5C33C643DA0CC00CB79E54FAB25D1881B81C03BA5762BAA551A7E8BA38144" +
                "353B07285B288BC2747F75B7AF249040C338CFC585D0B1CECFED46BCAE7FAF09" +
                "60BB3EE996E30E626CB544A38393BC7DFDB7A27A21A6CF09332B544F448DF5B3" +
                "31E000F7CCD5CE5C8E8765A2339919C713352FCD30FA52B994C25EA95E548C4B" +
                "5EC23B3BDEC7342D0676B9227D3405758DBA5BD09F9253791FAA03F158F04848" +
                "D5073DD240F466F57770353528B3AE83A626F33D05BD1BBB4E28CB067FFAA97D" +
                "B4C79EEAAFB4B30BE738C1AA5DB1830B3968CDF6BAF778494AE40EF003DCDA54" +
                "486E9952EB44628385E149C348E0E431928B85608622B994CF43433DA9C19482" +
                "360121560E53E85FE7CBB7C31E27AD335BC247F284EAC3CA94C30DBB4DF2AB02" +
                "DF1154626838240213D910D5B7476A025CA7565CECBA0051320FC7EECD6C74FF" +
                "505566F75804D1E2BD2B0181B235CE911EAD9260C0799C817F956AE290E00EF0" +
                "997F7B6BD059B315915D580CF0C019A23A6D4993F6E8B8106A1AB6CE1991B609" +
                "1B42B6D33EE01EC96CB475430365E9C710C5EB4C6010260D12108022449F7E6D" +
                "1A2F28838304DB2A60B9FF714FC887579A4CDC139DAF30A18D3910D82313CCB1" +
                "FA43A8930E0F10DE24652AC1E5B797084BEBDB8AB5FA6DCE03E44ABF35EDEB1A" +
                "FFEAD3F7C9CB342CCA2882D945EB52C20DC595FA10161866EB9426281CF13341" +
                "311B59FDE8E69F9B853117740D92F4AC1B2E4597D41B8A097E1DAA688FFF1C5C" +
                "846DF96CA75224EC26F4FF328164F5D1EC06B697211BDB42E6C97EB294A5798C" +
                "0FCE6104C950A5207F74EC0DED8AEE10463EF2D9ACD7473D2BE48EBBF0A550B9" +
                "AA19A465147B378B078229E8804918136633D7FCE5340AC61A1418D7D9BB18D1" +
                "98B7B7866C4D7DC562B1F93F3F322484BDDCEB23680B8EB9904EC783D5CD7177" +
                "CFE9CA9D1893104E97760E871DE0907D4BDF6263E7BB0F47414AF31E377C7447" +
                "B881E68AE3E80D06597F12D5EF5ED861D055D494D89F04A70800DA3FD4E53877" +
                "87FBEED7B772E3A24E7F4832A956FEC0C81847C68373ED4760ABF542F77DC794" +
                "249519BDDF5F846EB8C5078BCC053037301F300706052B0E03021A0414461F5B" +
                "19C6933240012EFEB95F734C648CCD13460414FA1743400686D25BA1CB28D736" +
                "F2B1ED97699EA4").HexToByteArray();

            private static readonly byte[] s_Rsa2048SignatureOnlyPfx_TripleDESContentEncryption = (
                "308209EF020103308209AB06092A864886F70D010701A082099C048209983082" +
                "0994308205BD06092A864886F70D010701A08205AE048205AA308205A6308205" +
                "A2060B2A864886F70D010C0A0102A08204FE308204FA301C060A2A864886F70D" +
                "010C0103300E04083EF905F0EA26FBF7020207D0048204D82297B5546DA6CC49" +
                "BD8C1444E3FE1845A2C9E9BDB8B83E78235DF4ADF7A97496A62D31D4EEEB76B0" +
                "71C0B183ACC3663272F88CF4F31E2E00D76357C0A051B8D6E0BB0BCF4CCDD064" +
                "CCBAF546EABA80DA56CD11FE952C61154792559D65F26B0476CF7A5FDB8CC794" +
                "B89F6ACD50003459054FE82C48D8791B226A0EEEC01F048AC3CE716C9F3BB313" +
                "D64BEBBF0037D83133DD9C15D04F15BB11652D793B613A68AFE580245724E5D1" +
                "110040B332B5C39BE04086BA4DFC58E905BC2FE8B3C696181E2879AF197EE24D" +
                "91D8AD67013F14C4864C8D0FB19C134B766CF3E48B8C9E363A11EB19F1E82E74" +
                "D25EDD96517D64A94314B40C11651030D561E742E63856E8D1A3EE9FDFD6CF64" +
                "7140CFC354AE7EA1D14C157C2985F82D54296F7D3DE456AF7513F5F30A0421E4" +
                "3A9DAD6DF8A2BF69005B35CA8066F80755D848DA73EF03BC0CC129C5799911D9" +
                "3A1ED43F8E76732AF56FD62DC6D0B0DBA6AAC6DCDE77D0E8AC9F6A5EB5A02B61" +
                "BC477706D4F1873240AB45E0291EF21E6034D48F1AE8EB139DE7ACD8B8A821E6" +
                "70B395C3EC4B0E75C34BF0067F052FCED835CAC1F17C3FBEA2FC9FD281FCDE21" +
                "D5B27CF31A07E90164A979ACEF0E1C67DBA6C33082E9B189D4BDA2D2D504776F" +
                "455843437BDF10D4AF48639EC03344BCC36EFEA7CDE08D7F94DDEAF98BCB5D65" +
                "207AE0C349EECE3032DE19F3B337F29A3457AA0AAF4306FA8301619AB01B7235" +
                "BE16CB93728E142DAA6C1CBBCC5BD82D913596994DA40FB916CF2DB5FBCC20CF" +
                "E893DC62BBC5FC59E00BC0A704A9DB25BBF9291971F2377FC1A20F2C954869DB" +
                "6FFCC90C625AABE97ED4CF7C0209D39AD780003C437152D636ACB3B484C46885" +
                "DC1584733D2153A3F9B968F12CDD5937CDF9DD2581D72EE67C83AE2530197AA7" +
                "C6110613BEFF0B75E586C28394EA8EBCF7F9DB133295B33DC86C8DBA92BF8BD1" +
                "ADCAF8D2CD2E018B08D59FF1C30A13484AB11468F7DCEB1FE53A6DAF309B0510" +
                "7772CB735314A5B2F053E60A653F0496BCB9CADF5E50339A4D2EF2382056B768" +
                "558EB9230D996C636E6D29664F92F70A088DE3EE4EC4BBD8A9C4C98C7892D122" +
                "28806622B87E581A321AD835B8F4B964A17B5BE6D9DA50133D494732A41884E2" +
                "9E891FE2D40ACCFD585C8BF626C1E8A412D2EE7CDE060E2CCDA826BF79D80F1B" +
                "F6B8400473BCE0C19D03ACF55D1FAA994C04A8CD11D49743B1F45F48DFFDD701" +
                "18B5FA82ECDF67714F5DE5D3D3DDDCB76ED0EA6A6E151665A4AA351DB1A99F8C" +
                "7502D3795C2C358CCA589C390C1F4810615130B91BA42A85E77FA37197E1B083" +
                "FE1246B067C6444D49B369D45B130A6D7B463C3F0459EB41D68009CABD2F5C60" +
                "49DB706FA742C9773FB5791AF123FBE485E05F87759ADD25281BE337B6095EC3" +
                "4EFF9FC692798FB4217EF4B2B59902D930F28181933FAA278C041123CAE3CA63" +
                "6DFD3AD4E04EB751A30D50C26288EA4D01C7B323E4FD6387F88E020BC433BF60" +
                "C4406398C44EA5C7A6EB24134B0811E4F94DFAF5553172306FA5543C254E7E04" +
                "DEEC84DBF9FAF7BFEA8D61E094CBB18DD45C5BAB9199DD719F9A305E205605CC" +
                "671DCD566FEBA2C8F4C1A445625C4F42D1CFE32087F095591798D1D48DA46DE9" +
                "230F5102B56A1EF879D48936D5331D6B3D9F1B564CF08FD3C641CFF3B02CB4FC" +
                "8995E5EC5DD1D183704940C02DEA7430FD594E54800DCC74B7732731C63FBBA2" +
                "A2F6DC031174390A74781D352B09FB4F318190301306092A864886F70D010915" +
                "3106040401000000307906092B0601040182371101316C1E6A004D0069006300" +
                "72006F0073006F0066007400200045006E00680061006E006300650064002000" +
                "520053004100200061006E006400200041004500530020004300720079007000" +
                "74006F0067007200610070006800690063002000500072006F00760069006400" +
                "650072308203CF06092A864886F70D010706A08203C0308203BC020100308203" +
                "B506092A864886F70D0107013024060A2A864886F70D010C010330160410FA37" +
                "DF4A395E316613255C1FCC424EE7020207D080820380B6A6EF2B8047F4541047" +
                "384B943C35C28E4097BF204745FF0A112E9B00DB2AF79938B994483B308412D9" +
                "F69E82027A3B35AB71540597565C4C5D20D62891AD495B49B4CD4C9A13D5ADFB" +
                "77870EB32893C7F48758F7C7580BD02C658949874B6EF41A18C419A9D31FFDAC" +
                "2FADE69BB6D17EBADAFF84AE420DFCE67124D7DEA9576711D5C340D2B1C122F2" +
                "0031F46D57F8D3A7A44438429BFCFCBBA36A135DB1FA42B40491EF3E63168F7E" +
                "992F4EBA4AA687AC5ECDC6AC4E8C1B8976E1868394AA9D703EB6D4227BA603EE" +
                "C7CAE369BBD26934AF9F31D15DB04EACA13EC71D103F44669A965AE74060B33C" +
                "67CBEAF974BEA1B74EBB4123AB1BA961CCE5B744FE1C97897BDB0D724317C7A5" +
                "A1C8C41BCF6E10DD65C118D1EB0504EA17D0A9937854CC3C27295A33DA035B7C" +
                "BA5F92156BD62347F7BB33CA9631E869A3987F60E9BE4231E7D069FE0406DA70" +
                "C54ED86199C67097F30D44E36DD5CD5C379368456F71BC45BC5E75DCADAC3164" +
                "068F005ED7E6E83F9F13532F555A1F5131DA6BA1BCE375B3FBEF681C4B60A9F1" +
                "C79FEFB1742103117F5ABADF4FF93FDB697E13C5DB93288FE9D7DF45BB7E7BC9" +
                "7EC85931791A6E889A4DC34599C985F39572785C4C8C31EBB2A9A32B093D7BE7" +
                "0AC1B1C1F04B099865CB54E01F0D6F990EF598333E9D19BBEB42F0AFAA650D1B" +
                "9E00902819BB68203BC5768315757F3417E7C4E744498FBBD2376D07D6BF06F7" +
                "9E1F2AC7013F609A68987EBCE1CA891B8D8E9A5C4F391F441FA2CED912CA2D2B" +
                "A5478E885F1AD7056962F078BCAF1FEEAF663B06192A0F185541DA4EF7887E34" +
                "5A847A857102F84C45845FEDB5288B06D9C4DDC78FE94C691A206EB65CDB357C" +
                "A680EE919CED5A472A5BACB48D13C158A542B4B64376A13E44EA99629DC7B6CB" +
                "5F4293BFA9FEB3101E28E11E56FD50FC380CB29F79E5753793B75B4E0B582148" +
                "C0E1B7A31175DE703E4AE6AD1D7B35AAA075FA30CC1909EE443D1BA610AB7FC3" +
                "8A0810AA5EDB9DFAC3972A0B0A4D07D2562DB4C48B9D04AE93A43AFF8D94C9BE" +
                "FBD5FDE36E4A3F0C643EDEB2C0F3C2ADFAAEEC627718BB6022C99D9C6FAF7345" +
                "CEA3E01D032125CA0A14E7B8101E4061692CCC7650F08DCEB9440EB0E4E9293D" +
                "386843A4F4CD761C82835658E688C9CDFB46A71629E58CFF7639B3F537BFC588" +
                "B38ABE5EE0124A74FE4FB166B9ABDF013181CCD696FF190440055A2EFE98E368" +
                "14F8E01DBBC97DDC00D5A2C79DFD4ACC9FA206C035FA303B301F300706052B0E" +
                "03021A0414EA08415D254A79B627FFF47D0D844454A24BBC2B041476D3D168D6" +
                "31F22156DA4116AD098C466C4680D1020207D0").HexToByteArray();

            public static readonly byte[] s_Rsa2048SignatureOnlyPfx =
                PlatformSupport.IsRC2Supported
                    ? s_Rsa2048SignatureOnlyPfx_RC2ContentEncryption
                    : s_Rsa2048SignatureOnlyPfx_TripleDESContentEncryption;

            private static byte[] s_dsa1024Pfx_RC2ContentEncryption = (
                "308206EE020103308206B406092A864886F70D010701A08206A5048206A13082" +
                "069D3082043706092A864886F70D010706A0820428308204240201003082041D" +
                "06092A864886F70D010701301C060A2A864886F70D010C0106300E04084AF212" +
                "89D5D7E2E702020800808203F0DECCF218AC91F26BAB026998AB77C7629D20DB" +
                "E2FB7022A3C4A1CECD743C0F932E944AE229DAFB61AD76C4DEB6995DF4F4BA01" +
                "2DBAD5C63A4C846E0807FCA0BC4A162CDFBAB4B3C4D304F473B3ACC1D268436E" +
                "F537DAE97ECC3C634C8DF2A294CC23E904A169F369021A0C024A03DE98A65B0F" +
                "3F14D6910525D76AD98B91E67BB7398E245CF48A4D2A5603CFCCF4E547D7EDAB" +
                "669D9A8597C6839119EB9FD932D1E4BA8B45D3317186CDA2EFF247BCFD64A5CA" +
                "ED604BF7033E423CC21CEC6454FE3B74E03A26C51A1C3519CE339FBE9F10B81D" +
                "DF6A0AAB4F8166D90B6F52B3439AB4B5273D0A506E3E01869F8FEBD1521EF8E5" +
                "BFB357FA630E3C988926EF3ACC0A0F4176FE8A93337C1A5C6DEAB5758EC2F07C" +
                "11E8B2495ECDE58D12312CCCA2E8B2EE8564B533D18C7A26A9290394C2A9942C" +
                "295EBB0317F5695103627519567960908323FFE6560AD054C97800218A52F37A" +
                "DDE4E7F18EF3BF3718A9D7BF57B700DBEB5AB86598C9604A4546995E34DBABBB" +
                "6A9FB483A3C2DFE6046DFD54F2D7AC61C062AF04B7FBAC395C5DD19408D6926A" +
                "93B896BFB92DA6F7F5A4E54EDBE2CFBB56576878150676ADB0D37E0177B91E0D" +
                "F09D7B37769E66842DD40C7B1422127F152A165BC9669168885BA0243C9641B4" +
                "48F68575AA6AB9247A49A61AC3C683EE057B7676B9610CF9100096FC46BDC8B9" +
                "BAA03535815D5E98BA3ABC1E18E39B50A8AF8D81E30F2DFD6AF5D0F9FC3636AB" +
                "69E128C793571723A79E42FC7C1BD7F39BD45FBE9C39EEB010005435BEC19844" +
                "2058033D2601B83124BD369DADB831317E0B2C28CE7535A2E89D8A0E5E34E252" +
                "3B0FCEC34FF26A2B80566F4D86F958F70106BF3322FA70A3312E48EAA130246A" +
                "07412E93FDE91F633F758BC49311F6CBBAEC5D2F22AFCD696F72BC22E7DE6C00" +
                "3275DFEC47E3848226FE9DBA184EA711E051B267C584749F897EFE7EAFD02F1D" +
                "BF3FD8E882474CA1F45509EF2E7B82F35B677CB88ED42AF729848EE2B424B0CE" +
                "2E9AAC945BABA550C20D5B25075A30FE70D8CAA5A527A35F1DF17BCCB91930C1" +
                "7120C625667120E0806C2B51EDFF540F928BD555FB48DBCB83CCCE0C385E78C8" +
                "65BE715AE6F8BE472E5FC187EBE3FEFD8D7FE62D4DB2EE61F42D24D81FAA9179" +
                "0FB17E8EBC8E219B6F9E039F5AB3BC4870821D474B36C8F8D0583D9DC06E4383" +
                "D03424420B8C8B26276877166A0F51E22F0D8FA60A070CFBD47EAFBC717C879C" +
                "B5A1EA69C4C2A38F26A1EEF96A0C32BFCECCE4EA97E90A425066B1DD0891353F" +
                "766EB9F2BFA2563A815DAF3639EBB147E1E8757A6BFAB902C4A8F037AD47E03F" +
                "AF2E019FCF6CA7430BDFEA4B45B28ED746BB90E09BEF7B370A75E7924BBA0920" +
                "25FE654A9A197A5B8BBBE43DC7C892FF14E75A37EB97FC489AB121A43E308202" +
                "5E06092A864886F70D010701A082024F0482024B3082024730820243060B2A86" +
                "4886F70D010C0A0102A082017630820172301C060A2A864886F70D010C010330" +
                "0E0408ECB4D1550DA52C6302020800048201509322DC0193DD9E79ADAFD38827" +
                "AD6DE9299327DDDF6E9DF4FB70D53A64951E4B814E90D2A19B3F4B8E39A2F851" +
                "A3E5E9B9EB947DD248A3E5F5EB458F3323D4656709E97C6BD59238C4D1F26AB6" +
                "7D73235FAE7780D98705957B6650AC0DE3E2D46E22455D0A105D138F16A84839" +
                "14EDDF5C518B748558704ED3AE4A8C4914F667BBDE07978E4A4FC66194F6B86B" +
                "AB9F558EDE890C25DFB97C59653906CC573B5DEB62165CFF8A5F4F8059A478EB" +
                "F6FED75F1DACDC612C2E271E25A7083E15D33697270FD442D79FFCB25DB135F9" +
                "8E580DC9CE14F73C3B847931AF821C77718455F595CA15B86386F3FCC5962262" +
                "5FC916DDB4A08479DCB49FF7444333FA99FBB22F1AEC1876CF1E099F7A4ECA85" +
                "A325A8623E071EEA9359194EEE712F73076C5EB72AA243D0C0978B934BC8596F" +
                "8353FD3CA859EEA457C6175E82AE5854CC7B6598A1E980332F56AB1EE1208277" +
                "4A91A63181B9302306092A864886F70D01091531160414E6335FA7097AB6DE4A" +
                "1CDB0C678D7A929883FB6430819106092B06010401823711013181831E818000" +
                "4D006900630072006F0073006F0066007400200045006E00680061006E006300" +
                "650064002000440053005300200061006E006400200044006900660066006900" +
                "65002D00480065006C006C006D0061006E002000430072007900700074006F00" +
                "67007200610070006800690063002000500072006F0076006900640065007230" +
                "313021300906052B0E03021A0500041466FD3518CEBBD69877BA663C9E8D7092" +
                "8E8A98F30408DFB5AE610308BCF802020800").HexToByteArray();

            private static byte[] s_dsa1024Pfx_TripleDESContentEncryption = (
                "30820700020103308206BC06092A864886F70D010701A08206AD048206A93082" +
                "06A53082043F06092A864886F70D010706A08204303082042C02010030820425" +
                "06092A864886F70D0107013024060A2A864886F70D010C010330160410B7C75A" +
                "92EFD0F0CFA7B422AC87BBA3B2020207D0808203F079F1EACA1E60CBB110E12A" +
                "88FFCC2A888F4DEA4737692DA84C1040DD10F3FAC5EAC6B626633C2A690DBBFA" +
                "57BC01171653584EE6E9B47F575476C45BA73C2530E0104933E7FD1C3B6E9115" +
                "E2749B3A8A167D0F61BFDF2D4D405AE86AAB9693707583BFCCAD6DA60768EA4F" +
                "34B5220E5A78D01ACAA99351D1D5025BBAE623A0AEA002032FF701BE35830CB1" +
                "362D68CF7E470CF281697582ADBA32FFBAD5DAD3671105C2754ACCCF2A9839C9" +
                "08E6D6A1D3DF39510517E0C6C89C709E0F77DA1228BC4E9DF10535A62CC0B5DD" +
                "93E17D56533C2B5CD8EBD04EDC0DC7EAB89F5B1AD0056AAE9F7D0DE5C0F4A613" +
                "9BEB7C2CC2E5051BF6F0E68B8F347A1221E6D6C0DABE50164FE9608FB41C0EA1" +
                "01BB441DF3C9421869F4F8218ED93FFABC88FDD2E67922A0DF04CEDA1BBC286A" +
                "82DCA5C12A494384F0C8900E06CC6931397ED6BF4C8A763759B7C9D50F42835E" +
                "B61C6C1E029B8CDB59CFF27CB95D4850C6F4C89F5A57E716BA8723FB9D4302A3" +
                "F9CBA8C4E995971D70ED7F5598F1006695DCBC6F0E259845BFD0BD5C480FD9F9" +
                "2144E8EDCE881CBBE553B568928D934BCB0EFFAC5619398CE924BB52E1AF3D83" +
                "249E345BEE05AABCD66F7C28D64BF32AFE7A1F028AFE9B256BBAA3BF67BDC91F" +
                "2C5DE56790B2949155FCC5A4BFEB670549D182F06E1A328B56BAF23C85CF0F16" +
                "A23F6B596F86F5415B20E6E14B5A532585C78D9F23ED08343A7C9D21F495ADD9" +
                "C82DF7F22B1D9081F552010B2D1AEB9F690723F38BF53031C89BA4D4691437F7" +
                "D9B0CC002A551B69F37B086B4EEC3330B19563FE0832DF5A8751646D208C1ABA" +
                "7102E8161D6B473EA0C8A8BE5119FF46F6B3F78CF35A9DFE325AC27169C1BFAE" +
                "AD6D8E21442C0EA61EC559A28B3B113550E8D84F12E398CC343050C5A6B99A0E" +
                "BB801BBD226EA132DBEC3EB5525A4D1555C367443A1DCF96FB4D01AFAC5F1C02" +
                "DE7394A2C80630614E81AFEEB62051DB79AA1FFAA773C58800A90A46F5784E3D" +
                "D85C77BD74606E2FC2229EB671E09A88879E69488C630D8D510405CA189D6180" +
                "DA5B29EBE3B289546433C8A64192685C55510A78F02EBAB861406BC966B7107D" +
                "4F13A8EEFBDAA050AC1887781663A7B59354D52E61AAD77EE314E2DA62B9319F" +
                "6F02FFA32DF62A8AE63681B85F516AA82614DD8F8D6A378A2AB8505B85E96E51" +
                "6408140A20F2E517A386BD335504A239DA86524E44B7F1C08FEAB8F254CBC038" +
                "E8551ECA5491BD39058A992EF55A0E64F47305F3C8781A3EF5A1CDD7C6B30BFE" +
                "011C98167FDE1810A3FDD862DE14207C6A0D282431D6EA7B3A5103A29E32233C" +
                "B677FC62C68EF14EC0B57A35376B5F399183D350970848D09D5489F156AC2066" +
                "102037FB92DE0A6FA84A970270AFCC00D29DF43F970A01D7A67CDBFE0C9DD742" +
                "876450411A3082025E06092A864886F70D010701A082024F0482024B30820247" +
                "30820243060B2A864886F70D010C0A0102A082017630820172301C060A2A8648" +
                "86F70D010C0103300E0408ECB4D1550DA52C6302020800048201509322DC0193" +
                "DD9E79ADAFD38827AD6DE9299327DDDF6E9DF4FB70D53A64951E4B814E90D2A1" +
                "9B3F4B8E39A2F851A3E5E9B9EB947DD248A3E5F5EB458F3323D4656709E97C6B" +
                "D59238C4D1F26AB67D73235FAE7780D98705957B6650AC0DE3E2D46E22455D0A" +
                "105D138F16A8483914EDDF5C518B748558704ED3AE4A8C4914F667BBDE07978E" +
                "4A4FC66194F6B86BAB9F558EDE890C25DFB97C59653906CC573B5DEB62165CFF" +
                "8A5F4F8059A478EBF6FED75F1DACDC612C2E271E25A7083E15D33697270FD442" +
                "D79FFCB25DB135F98E580DC9CE14F73C3B847931AF821C77718455F595CA15B8" +
                "6386F3FCC59622625FC916DDB4A08479DCB49FF7444333FA99FBB22F1AEC1876" +
                "CF1E099F7A4ECA85A325A8623E071EEA9359194EEE712F73076C5EB72AA243D0" +
                "C0978B934BC8596F8353FD3CA859EEA457C6175E82AE5854CC7B6598A1E98033" +
                "2F56AB1EE12082774A91A63181B9302306092A864886F70D01091531160414E6" +
                "335FA7097AB6DE4A1CDB0C678D7A929883FB6430819106092B06010401823711" +
                "013181831E8180004D006900630072006F0073006F0066007400200045006E00" +
                "680061006E006300650064002000440053005300200061006E00640020004400" +
                "690066006600690065002D00480065006C006C006D0061006E00200043007200" +
                "7900700074006F0067007200610070006800690063002000500072006F007600" +
                "69006400650072303B301F300706052B0E03021A04142C936C6E3D06E14A72F8" +
                "C08B2FD50E40D2DDE70A0414D58EED14B5437EFB11F19215656B22E05D119346" +
                "020207D0").HexToByteArray();

            public static readonly byte[] s_dsa1024Pfx =
                PlatformSupport.IsRC2Supported
                    ? s_dsa1024Pfx_RC2ContentEncryption
                    : s_dsa1024Pfx_TripleDESContentEncryption;

            public static byte[] s_dsa1024Cert = (
                "3082038D3082034AA003020102020900AB740A714AA83C92300B060960864801" +
                "650304030230818D310B3009060355040613025553311330110603550408130A" +
                "57617368696E67746F6E3110300E060355040713075265646D6F6E64311E301C" +
                "060355040A13154D6963726F736F667420436F72706F726174696F6E3120301E" +
                "060355040B13172E4E4554204672616D65776F726B2028436F72654658293115" +
                "30130603550403130C313032342D62697420445341301E170D31353131323531" +
                "34343030335A170D3135313232353134343030335A30818D310B300906035504" +
                "0613025553311330110603550408130A57617368696E67746F6E3110300E0603" +
                "55040713075265646D6F6E64311E301C060355040A13154D6963726F736F6674" +
                "20436F72706F726174696F6E3120301E060355040B13172E4E4554204672616D" +
                "65776F726B2028436F7265465829311530130603550403130C313032342D6269" +
                "7420445341308201B73082012C06072A8648CE3804013082011F02818100AEE3" +
                "309FC7C9DB750D4C3797D333B3B9B234B462868DB6FFBDED790B7FC8DDD574C2" +
                "BD6F5E749622507AB2C09DF5EAAD84859FC0706A70BB8C9C8BE22B4890EF2325" +
                "280E3A7F9A3CE341DBABEF6058D063EA6783478FF8B3B7A45E0CA3F7BAC9995D" +
                "CFDDD56DF168E91349130F719A4E717351FAAD1A77EAC043611DC5CC5A7F0215" +
                "00D23428A76743EA3B49C62EF0AA17314A85415F0902818100853F830BDAA738" +
                "465300CFEE02418E6B07965658EAFDA7E338A2EB1531C0E0CA5EF1A12D9DDC7B" +
                "550A5A205D1FF87F69500A4E4AF5759F3F6E7F0C48C55396B738164D9E35FB50" +
                "6BD50E090F6A497C70E7E868C61BD4477C1D62922B3DBB40B688DE7C175447E2" +
                "E826901A109FAD624F1481B276BF63A665D99C87CEE9FD063303818400028180" +
                "25B8E7078E149BAC352667623620029F5E4A5D4126E336D56F1189F9FF71EA67" +
                "1B844EBD351514F27B69685DDF716B32F102D60EA520D56F544D19B2F08F5D9B" +
                "DDA3CBA3A73287E21E559E6A07586194AFAC4F6E721EDCE49DE0029627626D7B" +
                "D30EEB337311DB4FF62D7608997B6CC32E9C42859820CA7EF399590D5A388C48" +
                "A330302E302C0603551D110425302387047F0000018710000000000000000000" +
                "0000000000000182096C6F63616C686F7374300B060960864801650304030203" +
                "3000302D021500B9316CC7E05C9F79197E0B41F6FD4E3FCEB72A8A0214075505" +
                "CCAECB18B7EF4C00F9C069FA3BC78014DE").HexToByteArray();

            // Password: "Test"
            private static readonly byte[] ECDsaP256_DigitalSignature_Pfx_Windows_RC2ContentEncryption = (
                "308204470201033082040306092A864886F70D010701A08203F4048203F03082" +
                "03EC3082016D06092A864886F70D010701A082015E0482015A30820156308201" +
                "52060B2A864886F70D010C0A0102A081CC3081C9301C060A2A864886F70D010C" +
                "0103300E0408EC154269C5878209020207D00481A80BAA4AF8660E6FAB7B050B" +
                "8EF604CFC378652B54FE005DC3C7E2F12E5EFC7FE2BB0E1B3828CAFE752FD64C" +
                "7CA04AF9FBC5A1F36E30D7D299C52BF6AE65B54B9240CC37C04E7E06330C24E9" +
                "6D19A67B7015A6BF52C172FFEA719B930DBE310EEBC756BDFF2DF2846EE973A6" +
                "6C63F4E9130083D64487B35C1941E98B02B6D5A92972293742383C62CCAFB996" +
                "EAD71A1DF5D0380EFFF25BA60B233A39210FD7D55A9B95CD8A440DF666317430" +
                "1306092A864886F70D0109153106040401000000305D06092B06010401823711" +
                "0131501E4E004D006900630072006F0073006F0066007400200053006F006600" +
                "7400770061007200650020004B00650079002000530074006F00720061006700" +
                "65002000500072006F007600690064006500723082027706092A864886F70D01" +
                "0706A0820268308202640201003082025D06092A864886F70D010701301C060A" +
                "2A864886F70D010C0106300E0408175CCB1790C48584020207D080820230E956" +
                "E38768A035D8EA911283A63F2E5B6E5B73231CFC4FFD386481DE24B7BB1B0995" +
                "D614A0D1BD086215CE0054E01EF9CF91B7D80A4ACB6B596F1DFD6CBCA71476F6" +
                "10C0D6DD24A301E4B79BA6993F15D34A8ADB7115A8605E797A2C6826A4379B65" +
                "90B56CA29F7C36997119257A827C3CA0EC7F8F819536208C650E324C8F884794" +
                "78705F833155463A4EFC02B5D5E2608B83F3CAF6C9BB97C1BBBFC6C5584BDCD3" +
                "9C46A3944915B3845C41429C7792EB4FA3A7EDECCD801F31A4B6EF57D808AEEA" +
                "AF3D1F55F378EF8EF9632CED16EDA3EFBE4A9D5C5F608CA90A9AC8D3F86462AC" +
                "219BFFD0B8A87DDD22CF029230369B33FC2B488B5F82702EFC3F270F912EAD2E" +
                "2402D99F8324164C5CD5959F22DEC0D1D212345B4B3F62848E7D9CFCE2224B61" +
                "976C107E1B218B4B7614FF65BCCA388F85D6920270D4C588DEED323C416D014F" +
                "5F648CC2EE941855EB3C889DCB9A345ED11CAE94041A86ED23E5789137A3DE22" +
                "5F4023D260BB686901F2149B5D7E37102FFF5282995892BDC2EAB48BD5DA155F" +
                "72B1BD05EE3EDD32160AC852E5B47CA9AEACE24946062E9D7DCDA642F945C9E7" +
                "C98640DFAC7A2B88E76A560A0B4156611F9BE8B3613C71870F035062BD4E3D9F" +
                "D896CF373CBFBFD31410972CDE50739FFB8EC9180A52D7F5415EBC997E5A4221" +
                "349B4BB7D53614630EEEA729A74E0C0D20726FDE5814321D6C265A7DC6BA24CA" +
                "F2FCE8C8C162733D58E02E08921E70EF838B95C96A5818489782563AE8A2A85F" +
                "64A95EB350FF8EF6D625AD031BCD303B301F300706052B0E03021A0414C8D96C" +
                "ED140F5CA3CB92BEFCA32C690804576ABF0414B59D4FECA9944D40EEFDE7FB96" +
                "196D167B0FA511020207D0").HexToByteArray();

            private static readonly byte[] ECDsaP256_DigitalSignature_Pfx_Windows_TripleDESContentEncryption = (
                "3082041D020103308203D906092A864886F70D010701A08203CA048203C63082" +
                "03C2308201CB06092A864886F70D010701A08201BC048201B8308201B4308201" +
                "B0060B2A864886F70D010C0A0102A081CC3081C9301C060A2A864886F70D010C" +
                "0103300E040867EFC6F9352D2AA6020207D00481A8FB4F97F1C05439604C1210" +
                "31F2FE34BE90B3FCC24554D931236C1139FA658210EB58AC8FB22FA9970A29B8" +
                "F9B2825D23EC03024CB1EDAD3E47C5D285AAC5C33F0B4DFDC83B9B209F4AA859" +
                "BCDD17A9A3A9BCC7A26F3C6FB224E5532B98FE8F14D0BC5217F72BF602D55309" +
                "CC58CCFA98E68A882D7A6031A8C2F947E3DCDEC32318CDA62C04D7BD8D85A5F1" +
                "B5BAFC2BD05243D8FA00906F4E3006D4E05E4BE5A8A7EE8F86726898AE3181D1" +
                "301306092A864886F70D0109153106040401000000305B06092A864886F70D01" +
                "0914314E1E4C007B00430038003100320043003900300043002D004200340030" +
                "0041002D0034004500390039002D0041003000300043002D0035003300300038" +
                "00440042004200390043003400430043007D305D06092B060104018237110131" +
                "501E4E004D006900630072006F0073006F0066007400200053006F0066007400" +
                "770061007200650020004B00650079002000530074006F007200610067006500" +
                "2000500072006F00760069006400650072308201EF06092A864886F70D010706" +
                "A08201E0308201DC020100308201D506092A864886F70D010701301C060A2A86" +
                "4886F70D010C0103300E04083B4E94A29A55072F020207D0808201A8AB9F6DD1" +
                "EB034A689913986D91865C4F4C17595039EBC2F145FEFB656381F17F7222B628" +
                "B10EACECC6C0658184A4913DFE74870284BA3EE5C83F88A09E0CD72D54D3304F" +
                "DE93F274E2FA8E77F2F8C4F4A5F1C996004B12063B83DADDBA8388F9C407C801" +
                "3D24D4BE2AA8D1C5F9BB250D42F7D2A7A93C4706C7A878410AA2D0971EFB551A" +
                "80179AB3AB6597EE19D2025E572EEB36EE98AEE989F44AD03A6F38C262E7C43F" +
                "C40E89157925E9E03A8B7B80B99DCAA92E3F50DD63EA059ECA49B456A5136DDB" +
                "930F32DC38BE6F041856461E73BD8B0E9FCDBF8930B497C4914858B054A15D57" +
                "0732B932A0A123400E9DBE50E867A1F98E582E87B29E670F9A2F44566B4BCBF5" +
                "385C90DCD458975126E5534E1C51E58FBA4E1C26C34457E64D7BEC7D4A92AF56" +
                "A090B09516A185D281B1F36EC14A58472546A77D7DD6960B3CC491740B267734" +
                "1A884181E05C00ACFA0BDD8E0A13FA154E0A39FBA8B4E70E2EBEF47416D6E869" +
                "C5FB84EF125903EEAB905746AF8E60EA3126719683E609AD6BEAE01B1D0BF700" +
                "C8C61F2BC4C44348A5FC32B1E8F4908C369BC90731EC30D71FFEE237C87F2308" +
                "6593D7A9303B301F300706052B0E03021A041497892AC27A24A2CC4F8841A8B3" +
                "D4C17ACCAD1DAC04149135233EFE8BFB1A6C494E5E4F2621BF97B50E1F020207" +
                "D0").HexToByteArray();

            internal static readonly byte[] ECDsaP256_DigitalSignature_Pfx_Windows =
                PlatformSupport.IsRC2Supported
                    ? ECDsaP256_DigitalSignature_Pfx_Windows_RC2ContentEncryption
                    : ECDsaP256_DigitalSignature_Pfx_Windows_TripleDESContentEncryption;

            internal static readonly byte[] ECDsaP256_DigitalSignature_Cert = (
                "308201583081FFA003020102021035428F3B3C5107AD49E776D6E74C4DC8300A" +
                "06082A8648CE3D04030230153113301106035504030C0A454344534120546573" +
                "74301E170D3135303530313030333730335A170D313630353031303035373033" +
                "5A30153113301106035504030C0A454344534120546573743059301306072A86" +
                "48CE3D020106082A8648CE3D030107034200047590F69CA114E92927E034C997" +
                "B7C882A8C992AC00CEFB4EB831901536F291E1B515263BCD20E1EA32496FDAC8" +
                "4E2D8D1B703266A9088F6EAF652549D9BB63D5A331302F300E0603551D0F0101" +
                "FF040403020388301D0603551D0E0416041411218A92C5EB12273B3C5CCFB822" +
                "0CCCFDF387DB300A06082A8648CE3D040302034800304502201AFE595E19F1AE" +
                "4B6A4B231E8851926438C55B5DDE632E6ADF13C1023A65898E022100CBDF434F" +
                "DD197D8B594E8026E44263BADE773C2BEBD060CC4109484A498E7C7E").HexToByteArray();

            internal static readonly byte[] ECDsaP521_DigitalSignature_Pfx_Windows = (
                "308205C10201033082057D06092A864886F70D010701A082056E0482056A3082" +
                "05663082024706092A864886F70D010701A08202380482023430820230308202" +
                "2C060B2A864886F70D010C0A0102A082013630820132301C060A2A864886F70D" +
                "010C0103300E04089C795C574944FD6F020207D004820110C7F41C3C3314CCFC" +
                "8A0CF90698179B7B6F1618C7BE905B09718023C302A98AFCD92C74CEFDBE9568" +
                "6031510BEB8765918E07007F3C882B49BFBBDEFA4B9414B4A76E011A793DA489" +
                "F5B21F9129CB81A4718A2690BE65BCBE3714DE62DEF4C792FFA52CCDE59FC48E" +
                "5ABE03B9A34D342CE5B148FBA66CE699B9F2DDCF0475E31A1EE71543922EF65A" +
                "3EACB69553ABE4316590D640F6DB58D7B8BF33B57EF2A35445CA6A553BF59B48" +
                "F4D9A5B7E37AF448DCBFED3A2BD258A4BA3180A66D7508CA2037061517675135" +
                "DB609B7DF7CB5F39874E66512C57F65DA93E3A43A12929B63FCACC82C5B8D638" +
                "00713A83B04A6CEB47A3C79D9852AFF5DB58180B6CF8193E7194CF7F0B6EED2E" +
                "A6697C42AC556C8C3181E2300D06092B06010401823711023100301306092A86" +
                "4886F70D0109153106040401000000305D06092A864886F70D01091431501E4E" +
                "00740065002D00650037003900350037003300640030002D0037003000390036" +
                "002D0034006200300035002D0062003400330038002D00640036003000650032" +
                "0030006600360030003100300034305D06092B060104018237110131501E4E00" +
                "4D006900630072006F0073006F0066007400200053006F006600740077006100" +
                "7200650020004B00650079002000530074006F00720061006700650020005000" +
                "72006F007600690064006500723082031706092A864886F70D010706A0820308" +
                "30820304020100308202FD06092A864886F70D010701301C060A2A864886F70D" +
                "010C0103300E0408812C7B1E5FBB35DF020207D0808202D00AFB0F5D92F40AD4" +
                "CCABAA4805775198F5988210F0601C39EA4A5B025A0FADB6A95C3ED3CB86E65C" +
                "B13BA11216244BE2773705202CF5895D9E31E5FC208A9DD2D90B47495475A078" +
                "B1B531AE496E4E534E4A23D828D2DC3488D982CB05FF9A32E7C60FCADEFA8EAB" +
                "F01F1D29E0650DAC688F434C4D5D8A26C4D7AD339FD0A2C4E22785E07DEC2FB6" +
                "7D041FA03BAE4BD6F3175EBB65EE79B276FECE8A8155A925792DA2F8AF2FAAF8" +
                "75AC2207078643C6E3C3AFEF37FED0AA60BDB06C8C1908ACF8FA2BCD28BCC8D4" +
                "47D998449108BA7A03E7AE6A439D7310E1A1A7296DBDFF48F7401E259ACE7B0A" +
                "D7B77B06001B0D526278B109217A7FF14CBA143DBCA99604EB2067D1DE3FA94D" +
                "D306D0937D9EF825E3B5B425F1F01A2153ECE2DFFF81779B7678626965F586DB" +
                "B519633EE3CEBD9FB14F5CF88DF6E907176D79CC6A7B9754C5EA02AFB549C65B" +
                "612866D48D294E44EC8A74CB36592F1C40C4B53640F5AE13F0A7A7AE731942CC" +
                "5091C57EA07743373404698E347A10BF0433F495B69FE8077EC8971B8B322DA0" +
                "82746883DE62FF08D3BE6D0575BCDAC392E79CC9B286953AE3B17D27C477B040" +
                "63AC2BEA6AA74BE63456DA2DE4685E8195EF620F19FDA8943515A4BC2ECBAA2D" +
                "16DF0237A9C272D6418F948715238EC5CA386E74E4AA67248A56A285F1FD2E17" +
                "A5F0F09DD3448BA570A056DDC7FA275A42CC0D524911BFA8DB42BFA04588AE5B" +
                "5044CABD353B9090F9F8CF02514C9AC0F347A9BE2A03EC351BAE2D1AC42CAC2B" +
                "89C9ABF3600F41EDD4447CDD14192F85094FEFB95DBF260A6739276A279FD104" +
                "E346A3FD238F7497474B1B4F8B7C02E4EECA34284C19D0AA169C178207BB1F19" +
                "62CD5E0C8A2C7C55249628F0BFC575DF2ECB25D36E1B29631A612945B9A99070" +
                "5FF55769B50D0B77725F61FE55284301A604C63BC7FC58F79CDF89F7E57ED060" +
                "CDD855DF59E9A8C9A06EA0DFECA32DF2A5263D0BF72B9D485519AB87EA963D01" +
                "9BF9F6B1D77ED4AA303B301F300706052B0E03021A0414AF6FEF9E53E1144A81" +
                "53A9459AADF886BA2979990414AC82239C24D3BB89B37A6A6109D7B43ABC433D" +
                "12020207D0").HexToByteArray();

            internal static readonly byte[] ECDsaP521_DigitalSignature_Cert = (
                "3082024D308201AEA00302010202101549950E8AA087A34C179BE49774C8AF30" +
                "0A06082A8648CE3D040302301A3118301606035504030C0F4543445341205035" +
                "32312054657374301E170D3138303930353130333230395A170D313930393035" +
                "3130353230395A301A3118301606035504030C0F454344534120503532312054" +
                "65737430819B301006072A8648CE3D020106052B810400230381860004019D99" +
                "4545E2E70D4CD0901FFBADC7B6CCEED2DC4EE69624B7F0B5C3E81F7A8DEEAFFF" +
                "BC317A3768D03F1582877D1D84FC69554B76DFFAA439929D94B6BE5CBA8CBA01" +
                "ECB0CDCD730492414D10A00DCA812CEC46D6D92B9142297500C543652FE54A81" +
                "427E18EC155EB05D3426A28F24819F5293F3FBC95A9CD7646D5D4D046753ECE0" +
                "B9A3819230818F300E0603551D0F0101FF040403020388301D0603551D0E0416" +
                "0414FFA852184A01C69680A052AF5BE279E949718DF6302E060A2B0601040182" +
                "370A0B0B04204500430044005300410020005000350032003100200054006500" +
                "730074000000302E060A2B0601040182370A0B0D042045004300440053004100" +
                "20005000350032003100200054006500730074000000300A06082A8648CE3D04" +
                "030203818C00308188024201D8EB229D9D73C8C6634E305836E938349672D12D" +
                "73BFC5A87E2CD2985FF64EE44BB2800214E4839A2DBCEAA3F2342C269D74126A" +
                "FE248C0C0F7C700B4680CA8F36024200F6625A58C219C389F2B4127BFDC228D8" +
                "2765E2F9399DB66ED71EDF4D64F85998DE15ED82A75F363E42432BCE108CE55A" +
                "41A9899160F95848826A9CE39498AEC2EF").HexToByteArray();

            internal static readonly byte[] ValidLookingTsaCert_Cer = (
                "308204243082020CA003020102020401020304300D06092A864886F70D01010B" +
                "05003029312730250603550403131E4578706572696D656E74616C2049737375" +
                "696E6720417574686F72697479301E170D3138303130393137313532355A170D" +
                "3138303431303137313532355A302C312A30280603550403132156616C69642D" +
                "4C6F6F6B696E672054696D657374616D7020417574686F726974793082012230" +
                "0D06092A864886F70D01010105000382010F003082010A0282010100D32985C3" +
                "0E46ADE50E0D7D984CC072291E723DC4AA12DF9F0212414C5A6E56CBB9F8F977" +
                "73E50C2F578F432EDADFCA3A6E5D6A0A3ECFE129F98EAA9A25D1AB6895B90475" +
                "AD569BF8355A1C4E668A48D7EAD73206CCA97D762EB46DA75CF189E2EC97A8DE" +
                "BA8A4AF9CFAB6D3FD37B6EB8BBED5ADA192655831CFDAA8C72778A314992AB29" +
                "65F3860B74D96DEB2E425216D927FCF556B241D43AAF5FA47E2BE68592D2F964" +
                "F5E0DE784D0FAD663C3E61BD4A4CF55B690291B052EC79CEA9B7F128E6B8B40C" +
                "5BADCDB8E8A2B3A15C7F0BD982A1F0C1A59C8E1A9C6FC91EE9B82794BA9E79A8" +
                "C89C88BF8261822813E7465B68FFE3008524707FEA6760AD52339FFF02030100" +
                "01A351304F30090603551D1304023000300B0603551D0F0404030206C0301606" +
                "03551D250101FF040C300A06082B06010505070308301D0603551D0E04160414" +
                "EEA15BD77DF9B404445716D15929ACA4C796D104300D06092A864886F70D0101" +
                "0B0500038202010046AC07652C232ED9DB6704C0DD318D873043771CD34CA966" +
                "91C04E5B7A311805E965957E721358B2ABA55A0445864251263E09BFE2EAEB64" +
                "43EA9B20F05502873B50528E48FF5FD26731BEA1B13CA3E3A3E3654E6A829675" +
                "B9D7944D734C48D36E19B88B77320A5096BF7462E1B4D69F1DC05F76A9417376" +
                "D5127027B1BA02380684DCEB75A4B14FA36B2606DDBA53697CE7C997C2FF13E4" +
                "66234CBE07901DF33A037F323FEF8C4229F2B7D030BAC88B4C3D84601576B0DE" +
                "32F3116C6DF7E9AA29028289E0CCA89F8B809310C37E9BD514536131D9E66AF0" +
                "6B1F36BAD55C9F9D6D1393CF724D75D3441AD67491AA76C8C31AADE22209831F" +
                "C079B49408ACC2C0D975EF8BEE3E0F6A01DA4DFC6045433BA6B1C17BB0E3E181" +
                "22770667CBD6197413569470BF0983BF358C6E09EC926631B0A2385D3BF9D9F3" +
                "0B5314170865D705CA773652BD66E1B3C112D7DA97CDFB9447FBFCD4DF674AB8" +
                "A6F430760276B7D8439BE40961D0B7F9F2B7AC1D291C0E67C1789FE038B6652D" +
                "24FCAF0A49CDB1E61FBA146AEFA3D934BF3B6AE8A5703CCA80AA500B56DF93FA" +
                "931E2D71E292042342CFB073431AF0AA1ACC0B5F53EBF66CFECD52EB08B11628" +
                "000F5EA01AB1D8E89F219178DB67B2CD68AFC2C0C248D8A65FD9AE1A0DBFF84F" +
                "3BBF2077EBFB373F6ED8D6C7CEA7BFDF0425494078F293949496B0BEAF63CAB5" +
                "62C297590A737174").HexToByteArray();

            private static readonly byte[] ValidLookingTsaCert_Pfx_RC2ContentEncryption = (
                "30820AAE02010330820A6E06092A864886F70D010701A0820A5F04820A5B3082" +
                "0A573082059006092A864886F70D010701A08205810482057D30820579308205" +
                "75060B2A864886F70D010C0A0102A08204EE308204EA301C060A2A864886F70D" +
                "010C0103300E040809F1CB8B4255EADD020207D0048204C83EF1251C78FF3815" +
                "CDB0648F20FF20FA0437E5AAC5CB91C0D547755113724A913C02427A12A878E3" +
                "B161F388F4DA9AEFBBA5FEB274CEF3A35CC2EC4BFE32442E5B6624888D41FC3B" +
                "EA02BBEDA034BB196A3FA41E23DCEB6F6E906FD86FED066D547518FD628C66F0" +
                "1AA4B38F3DDD94D63A5B336B587BAC4D2B739EF096A902ECC4216EC284099F10" +
                "C93785AFC3939A44C22FD027E4E643B03641FB3B76B21DB632D8522A365A495D" +
                "5AC74CF7765E294CEC55C73F6A4BB45ABD214D7AECBC3735DA41D8FC66CD5C34" +
                "54F354E16084D0E1984B20423219C29CAE0FDCD16A16C5BF17DB23DD8F2B1C1B" +
                "DFC9008B647D2FD84E4EC7952BFDF4EA0F0A13D57CD530109BFBA96DD3F37F01" +
                "7F7BA246C35A9D5C0A2294A2EEFE35B29542A094F373B6FFECE108D70CEDB99C" +
                "A7172B17C6C647CD6614D3FAE0C02B3D54062FD8392F152AB1B46D1C270A9F19" +
                "A48A393CCF22EC3DA958C35A8A6A3E7CFFDC2C54090F844B3B73C3CE7F7EF26C" +
                "982567ED621FDB93E402FC145E6D7E8D7F2F9C56F870823C98B81586F34C7C96" +
                "CBAA5A67544964FA1BD70B9C5E8284ACF22FFC69BF62E389C5678E44CB10D6C3" +
                "D3F73DA7BF12B44D74C0233C140ECC93C5F0118C8F0D9B6FFDFB23966ADC070C" +
                "8DBFAFE78AE1E05F8FA7C75A22FBF7A8458B9C61B1F2BF2AD2F0A6C6D15AAD55" +
                "D960F173AC3D9890CAF50228796FAD6C7EAB1C8A3C8E9E42E4A4DA99C4A67FB3" +
                "E2AC958AD989508538C96697A0CFBEEB38E9CF27DAE3AB9492729A205CB25340" +
                "85CA0D580DCD49E88DA7950F99CD05523885B5257BD302E1B449F03941BD0DA1" +
                "ECCAE6A22BC9F7E446F7F6FD4EE850CA3BDD7338F705D93C2F196712250BCB1D" +
                "A18C9661E195486768515BC2761A66F61770589A62D172DF8EC288549E86E518" +
                "7B04E1550154FF45060945BDA07014F14EB554A5A324F9B79DA192F79AB0745D" +
                "F30355DF580778773F2FFC76FB7B367EDBE173AC7F70698766DE6BB384A5C88B" +
                "66B163E8ABBF0AA44C4ED0D3D51D167E8BEFB2E71D36043ADB098BF2DADD409C" +
                "1F53F5546D1C8A1DC5E7BE566D14C494460161BFA7CB7E936790A81A45763004" +
                "689FA9BC33F31B4E86A152700D35B20380F87F4304D7850CA7BF583724328E0A" +
                "0D9B611B919975DF118B334D9DD96A46A21B00FC3B7FCCAFEA968FF030EA5D8F" +
                "9AD8624F668B2A7E433E54547EB89FB876A7E1AD2E9DAA38F30E938D8BCFB431" +
                "6E12FB8BEBF57FD0BF542E55386A6DEE074CFC6A33A55A57503CAB7F115DB875" +
                "C8EBF787670BE98DC03C65F2502D6F8D001ECC819BBB9C60BFC3A88DB8A117D9" +
                "9E09C13AC23324E15E5EE3C22B084D428FF2DFB37F917F7629F8A15093BB7777" +
                "B1AD8CACB4A5C6271E8B21A18DB95D6196E9EBD870521CA16930F2D1D43962AB" +
                "B8413016DA0117E10AB2622FC601DD08826429D8B8AE9BC6F15AE78392C36BC3" +
                "06FC19C90AD43BADD9AACDFA8CC16075529BFC8C34766C12783BF2F2E0B393CD" +
                "4F8F05D588B99218806D57CD5935E25DB2AE20DC4CDFD7F5111AF9A9EFE45349" +
                "42CAAA72F1F95636085FEC84BB98D209FD4222BC5F84DE5926A28FF7A5B7548A" +
                "B4FC3331431964454A0C532C205CF8D3872C561E83D805F7BD7DC811A0A90C9A" +
                "CB308E8F06AB739DCE97A840B4AFC0E884982CFC9B37897CF272ED1F46027101" +
                "BC97B11F04D64B07556DCFD5F609C5C9FB4B3F2AB345CAB46211EF0BE5ADD6BD" +
                "3174301306092A864886F70D0109153106040401000000305D06092B06010401" +
                "8237110131501E4E004D006900630072006F0073006F0066007400200053006F" +
                "0066007400770061007200650020004B00650079002000530074006F00720061" +
                "00670065002000500072006F00760069006400650072308204BF06092A864886" +
                "F70D010706A08204B0308204AC020100308204A506092A864886F70D01070130" +
                "1C060A2A864886F70D010C0106300E0408C10D9837801E56F8020207D0808204" +
                "78827979D15533A4A7210A435D0189C133B65B00407E85BC44368D066C65007C" +
                "256EE51A55E35BF8EE7C6FAC3D67FF7CF2031FECDCC356A51396B2977E044A79" +
                "E6C6CB8859E4AD49765468A4A467071292EFE47AEB39856FF8F00B5C6C6190EA" +
                "B20CC9A7C630C09E3F351ECB20CEC1BFE7BEB5A3FD534BAF8CDB658318A37279" +
                "269A11E8A87074FF0B111E2CFC493BD08D7887A7A222743B0C50E47F676F9B47" +
                "449F8FCBC6AE5F5A3456AE6BC3CB8A3CF28C59D0A16FE4336E0BFCA0AD74F95E" +
                "0F1010C3F698E16418E46B0059AB8F3DFD31FDB99132665CEC4CDAE8B6C1D0EA" +
                "9DADB8E591769261C27188CD5FF8D5C56E6866D85E1502254823940EC77096F0" +
                "6D3A261F49495AA60114BDDCA27C603F78314678CB08738FA2974DE03BE315F9" +
                "1FCA511446C68127211CF575948550DE4F7FDBF4AC31E395E12EBFD4BB99F470" +
                "498846940A92B6F85CF5D745B5230EAF994DD368FF55809F299213749CBABE54" +
                "C54D39B2165370DA43468409E73C55EF784D17496AAF8B6B5CC0F9FF234D7CF5" +
                "C84C248E91F8B08F74A8A953F60407940A2EB9576655CD9946D468DD6B3DB634" +
                "A30D4FE8E09C51AAB99C3DB5CC834A447EFBFDACEC9A49AFD675BCB247575B27" +
                "ADEEA9C5A6F2FCFF71A57EA99B467029C3E40D94D2849527FB90B9BF39294CC1" +
                "AD1F4CC18F75302FF241E05896A357BAAB88BDA8F7EB22B9AEFC5D7600154631" +
                "3C2E87452F99997CF4747CA936BD9199C61CEE5E61BBDB80C17978938F241D53" +
                "AAC97758FD1A77F3C7C28F88F1BDC7E74176CA41B73B7042A6C3C7AE918635E1" +
                "1EF4F2607588B0FF7C656541C32A1450AC71AEA42C14E94F0408A09902C30D10" +
                "ABE8396EDF5760991C2E02949EED06D091A2AB64827A53F29FA8B8B79036DA26" +
                "210A7C34EBE73590DAE990666E0F0F011EE8E41D08E16B28D49F30D8539EF18B" +
                "61860EE73D9AF98E4D3F17A50D06FC0592C218F38C33E0B526761D030E037B27" +
                "348CFDBB72C6B635BAEA497BA12618BFA47A9E358EE834FBA8A5FA9D30A66CFE" +
                "BFAF1E25FAA361F94433B40628E7423491BBA29116EA6ADDC6DFEFFA3AC2AD83" +
                "83C3EA05DE72B7CE357612A9C60CADCD42DECCD40438A9FC8E5D9413825178A1" +
                "9D67E7B13449F118A6B56A4F8A5DCC0AA05D0AB40B8FE0820A2399307331524D" +
                "B3389C97181EDC2ED2C5B72C1318B406CF99120CD784B85238488D1DA062478D" +
                "1EE6380F9954BE5442C1C00ABF8D284AF95937E2D515E2E8858F3C50516CA00F" +
                "E3F632BB05EE4AD63A6F2D72C7F2B06DC993866F6A740C1EF5159117437F646C" +
                "A47AA4BE5936C2F6BEF4C628A744FA2A2E2068752FF43BB9CB29C986D3A466FB" +
                "F3A107E8D610B8236E4E074CD6B32B50BF1C389CBDC2A92FDD19047450E04B4F" +
                "B963CA94860C1DF05D9FE1D202227E351BC2226572D6950D7E5242BE17DCC992" +
                "A989B7D13BAB9C8EC76FF5B8DED7696BBDAC058F6183F6F799365EA1DCC4F1B2" +
                "43A35027BA2F02E19DFCABEB564536EDA7B073519C60294C4AC25956D9F9DA60" +
                "3C5E4F335D7570299993460BDC5F20776F22D745A6B3F7E80EC69DA881AC72A4" +
                "D6B7AADF7EF19C77A2555C1CF8CE1CEB030EBF1C365D40C9C33037301F300706" +
                "052B0E03021A0414AACBB391FF9626295715807ED7DDEE57F716A5710414658C" +
                "344F4B20292DD9282953DAA4CB587AD48714").HexToByteArray();

            private static readonly byte[] ValidLookingTsaCert_Pfx_TripleDESContentEncryption = (
                "30820B1002010330820ACC06092A864886F70D010701A0820ABD04820AB93082" +
                "0AB5308205EE06092A864886F70D010701A08205DF048205DB308205D7308205" +
                "D3060B2A864886F70D010C0A0102A08204EE308204EA301C060A2A864886F70D" +
                "010C0103300E0408ADF347A4330ADED1020207D0048204C81696BCCDA96E42F3" +
                "38EF7182A7517E36157F11BAAFE0CC6F289215BA8F034FB30D498A0E5149BAF4" +
                "A40F3BD8A72F5B96A06AA5A7297A71A231A762EF9AE12C9E7D1AB37CBBD7D205" +
                "7234110A55F14FFB89CABC1BC04C0388F5A655AE50DD9AAA001291799C7902E0" +
                "BC942064CC53EAAE4E6D5EE19611EEF1B935131EA9CF38A055BD2D7CE3EDECD0" +
                "4B34B299A1A1429A02DBAC1A4C1DEB828A933D15110C2BF6AF5872CCC84C912B" +
                "9E14F5CEFAA7F0EB496B9E277765C9530784E4FA487252E12316166E78D1C6B0" +
                "C4B74A0AE19C068F9AFD2BF18A5123A65E93696DDC9ABA7DD1625F9B16D9F5EE" +
                "EDA9ED0507DBED944C6695B0DDEEBD3E2EC88F386FDC1497478EABA600F64E98" +
                "478685EFE70B07A54CE17FAE9C0D66D9ADE92BDF0ED4D1FB87362FDE3981492E" +
                "3AA4D47B6943B0DCC36CB5DBD9CE231980E54CE8A71D0EA3FB813D4903B88FFC" +
                "D7546A41003B688CD43EBBF59E1181FDD3DC9C37D641D1E9CA99167C746DD917" +
                "06571ED2CC7F04EEA2FF5097D859812BDFA491678EBE8D77DAE016757B3E4713" +
                "94D5B08ED1C803AF3621644C8D34AFAF13239B6B6DDCFC163BAE728BA002FE7B" +
                "19C7157A53BE0317073977ECF7072EE39EC1FA13FA26E392383BE1B57C91595E" +
                "C5B4DD4740B63E4BBCB09C00A590E211FFE5863E7C06859C1B43912BEF2BEABD" +
                "2BD1ACB43AD3729BED2AE491C1769F7E36FA91E63A8DC14A1580428817EF7451" +
                "F5C85DD349B0878652958E385897C063E72D30F2B2C96C26D9967F8DEB796F05" +
                "761C2406DE70D8D7275A40A78ACC802A4AA5B071FAF6218DC70288AAB29780A3" +
                "BBC84E919D6A1B1D3F9DE41C931F860A27B12C155BFEAAF56CC5AD6F0ECB592E" +
                "FD0B36BA166A121C66655A8B0C5DB299DC36FB5B75D634565DDF7A2498407452" +
                "B59B996C5C586BAD45B968A20481598C285E870C3E534B3E51D28CE409B1B5B9" +
                "138CC40EACC44C6859F83E6D3B6C1AD88FCD4414EFAD2CAA2658F8659905D856" +
                "B0E2257606C1C93A39A54E35CD75405C61C49D7A8CBF3CA614797065C3B12C18" +
                "4BAE8C222ADFC2E167CD6E6DF2B26270614EEEB009818F8030B5D5B7A5829C9D" +
                "39DCF8ECD5CD3A0E403B6F9E6E1C6E682A8EEA4EA5CE28D2A123729A3C0B9A32" +
                "845CD29970AD092A77BA1599339315F233F92CD741BA40EB029D36608CFCD711" +
                "03A4CA5A4F82C688D54DC83DAF5B5EBB50A99802D69AEAD3A145FD9598D71345" +
                "67BCDE414FDE05B922154BC841015C26DB93236D3BC32F125D1A4DF123D18337" +
                "0A55CA413A8427F49DFEE024B52EBC7F1B3FBC6C19FB816B7C85DFE309CDDA83" +
                "F078112432FDFA76F238C8C1BEAC104D59575907BF834618B36A15CD5C2FC817" +
                "004C7D52AF29064E19CEAC287E362DF2DB179CFBF373786A29EE2B65AC0BD6CC" +
                "C09AA409ABF211A69C357EA8972C3C5A413EDB8651660A2E1FE1B300CB402521" +
                "CB4990D6C7C4FC20CC9C9D929D9324C2EEEAA4E253F1E5503E456A6C5365E625" +
                "CFF07328E49322014036F4C86D6188647766F5072FC1DF6B6F1F222EF2982818" +
                "4C9B2B625C1644ACA5AAB8BBE783A7435A00EF44151EDE46DE7B3D438479745F" +
                "C31D620DE785C299B33C74479094D98B2CB6BA962C3DBE1840308476A5917331" +
                "9CE976048856835830FCAC610F5BDC5C9E80DE63B56910B388C0B7F10E3356B9" +
                "9B604DC107FE9A30292C0109552158D2D1A5960580E013C577946CB2A0DF9D23" +
                "3181D1301306092A864886F70D0109153106040401000000305B06092A864886" +
                "F70D010914314E1E4C007B00460033003400310044003100320033002D003400" +
                "3800460042002D0034004400330043002D0041004200440039002D0031004200" +
                "44004600340030003500440032003300330039007D305D06092B060104018237" +
                "110131501E4E004D006900630072006F0073006F0066007400200053006F0066" +
                "007400770061007200650020004B00650079002000530074006F007200610067" +
                "0065002000500072006F00760069006400650072308204BF06092A864886F70D" +
                "010706A08204B0308204AC020100308204A506092A864886F70D010701301C06" +
                "0A2A864886F70D010C0103300E040857A66BF90E9EA805020207D0808204789A" +
                "7480EA324EA3963561F35DB45508292574D9617690F85A4AF727E03703C85283" +
                "95C03B958F0FDF0F547A823BE0D4912EC54B26D6734989EC9DBF638F9D951E1B" +
                "0979D9432A44D7BC02395DA1675A9D12F30B2B4C5981FBCAB9B1EA258F9F8DAA" +
                "6A97AC923910D4AA0BB67CEF311861069CD4DD1141A1F34E01EE44DCB12C158A" +
                "76EE43CBE6097446ADE2AE14DE57B36E22BEB8FAC309F9B92385A30D364572F9" +
                "C499DCAE960C4DE6223303EB52CD12B02115C5A3CACC7133E04047649CE99F71" +
                "5F014B29FAEA55132CE6BF5190BF01583A58EDBDAB41B21FE9DC18E57B2BFE65" +
                "41E3796DAEF49AA2ECA8083D576A2E98258AD488D48F1832D6C64FC08CE56F63" +
                "97075E358963FC47F3759F6CCAF3EF37A679D542EE05FEBC4BB9E292B141497A" +
                "74C99219612C65D42AC450DE6EBD35BF700A1EF5B844C955D81D0D20C84EB999" +
                "161AAC32931F46D9216A85004858B3136AC2B5C94FAB360ACD441C0BF7870A69" +
                "1E3114816F43F6F5B5171C1EE85814EAB5A0640CEA23F7AAAB756DEEE703D4D0" +
                "848C071849DFCE92048C4422F8B94C8FAA430CBCFD1A736B1FEF01BB9D8F64C3" +
                "55D42A645D55F772A101A630803D964813C2A15C3891317599664ED1B6034049" +
                "1B7369C336A6E6513DE128E258F0FE2C2C6E1188B8C33F072CD11F4A72F68482" +
                "C5433DE391E9BBC58E1C886CE11F3EC733D29F7E940D695C7A56D5BBA47F3D13" +
                "8BAE578120F7339AF9DEB9930C786B83F16A9F56BFB31A0918CC917811E12787" +
                "10F67967E0EE429A9B8F20D683D9AAE0A860653CFA133F001F3442931396EB18" +
                "CB1D27AEE356CA2FDB374191BD59CFD5A9E58341FB32757459BCB568CED7CA8E" +
                "4EA4CA2DC98E9FD250C4308F65482D4E161E6D072D40CC9D03FFB72A069EC289" +
                "9B7E119ECD0AA49074E259CCD63D02395063790ED310446A3B49B0FDC8F05665" +
                "A07BC452474DCBE3B739ED79D52C946DB7BE2BE95602232D1DA5B9307DAA6147" +
                "4633BD04CFCF1FF20637C83B0B6A3EFCF59445428EF8BE2E22159BE5F42899FA" +
                "F70B50F24FA9131F8CEF2D01DCF3DD60A9780028F302FE813188FF9CB9156880" +
                "8FA543478A68503A5B8CC93167F0B8032029073C55ADEA9676D08C86C4FA0D6B" +
                "E9D2AD1983FD068997D86980663778AE94686BB19E77362B0A9289080B1A0865" +
                "4621F8ECEDB1D08224E44D8F0785DF2EB6A5B1BC7A97182926C30C78B345F995" +
                "1E5DD67F318ED5A7F7922BCD71993426E26067477C0ED06887C8790B73A5661D" +
                "B753E85433080616FBB9EFF7D40A7DD3072F9E89B51516C94186C72F3804C7A3" +
                "1DF5F507675B21B295A0D14B633BD515C382AC4093B92D32800C7F63E8C9DFC2" +
                "30B1692DADE7F5E6E71FAB8B40D3516F6F10AC3BF3722AE4F85FC6414FB4FD45" +
                "5B93D87171DDD04962949AE685448B20C27561A58F90E05503DAE995711C10ED" +
                "2881F2F15BF58EE206625937D620C715E397BA6D74C21BF68E1CE35915A9A917" +
                "E935308E724F336151CCA43498371F92A9C3E543559056FC4DF42857357FBDFF" +
                "D26A215B4A4F9880888683956931C11B020D8F69E392B4668A2E05D1DDCD8CB7" +
                "37C0EB82E91110CDED11FA41D5EBDB44499BE2FCB5E021303B301F300706052B" +
                "0E03021A041413439C2B57F81655034A005B25CAA60BF393100B0414A6161E42" +
                "F758104D24EBAE60A38A72EAF7F7519B020207D0").HexToByteArray();

            internal static readonly byte[] ValidLookingTsaCert_Pfx =
                PlatformSupport.IsRC2Supported
                    ? ValidLookingTsaCert_Pfx_RC2ContentEncryption
                    : ValidLookingTsaCert_Pfx_TripleDESContentEncryption;

            internal static readonly byte[] TwoEkuTsaCert = (
                "3082044030820228A003020102020401020304300D06092A864886F70D01010B" +
                "05003029312730250603550403131E4578706572696D656E74616C2049737375" +
                "696E6720417574686F72697479301E170D3138303131303137313931325A170D" +
                "3138303431313137313931325A3026312430220603550403131B54776F20454B" +
                "552054696D657374616D7020417574686F7269747930820122300D06092A8648" +
                "86F70D01010105000382010F003082010A0282010100C456AE596BB94EA55CE7" +
                "D51785F44223F940237C1F0A279533875427547BDC3944B73E8E6F4463800571" +
                "226147CEA3649972F96F128B673BCA6BBFD70B5178FE93D4DD7BE9E4D450AA0B" +
                "4D177F24DBCB2A7A13D7F10BABCE0E9AD3B853F01872196F6905F523E260555C" +
                "AFC5B158A82ED52D62BDA32142982EE8BB4E011E44622B59387B8A287F4DD7A1" +
                "5C783EAB5D4736CAB0E06A78EE50A7BA59DFE2C35DAEA0637FD581DB485ACA9A" +
                "57B94585A9E7D7BFAEE31F92F96DB5F95DDCE52B839BC06C30191014FE6804F0" +
                "CF4CA29412EB34D87303CE5FB49EC3E4B1A7CB001B6F117E1D5846A605ECF5D6" +
                "8FDC32EA972CC8521B823708518BBE59D15D10DB79990203010001A373307130" +
                "090603551D1304023000300B0603551D0F0404030206C030160603551D250101" +
                "FF040C300A06082B0601050507030830200603551D250101FF0416301406082B" +
                "0601050507030106082B06010505070302301D0603551D0E04160414B163E5F7" +
                "BBB872522615BB55850DF3D636E8EA6A300D06092A864886F70D01010B050003" +
                "820201008E5927A04E7D6361819B565EC9C3BFB59F1BBB57007B4534FC5B2F53" +
                "6A02BAFB2EC19DD8EE65DC77F34E2E7A4A2EA1597EE2019CFEBE8BA45BD35A53" +
                "D201465818F1B6E0E891FB380813E27FC1F6D9F3D5DC4C23267FB4326960E7C4" +
                "D69CF837904E694F4DBBD1AA346FC058B158AAC7EDD7F947F67DD09D9183B84E" +
                "B91CDE170895B830F4DE87850BC79C3E27534298BD8C7E7DAF3CC1242777435B" +
                "2A80CBD57C015B1B52A5B46D85187E5FF20EE68A96FB705FAFF4E335EF4FDBFF" +
                "CC5D981943127CE41EFA8C1D7E2E05D03D772E0C02F4443AB3420B6E3D1607BD" +
                "B2416268AB7D9D2B5824AD82B06ECB58B41C0AE092D704F77023F6CC8918F7D4" +
                "3128357B7504276C805E6C27A0A5872C89375F69FF14E91B4A11922DFE7B8142" +
                "8B103B55973214426A063DE2955AAB165CDF105E574F23C00E50FF5B8AB86222" +
                "460734EF5003A0D780DA6FEE9B8CEF0AF45C0BB1D0D142C4A2FDB2BD9776B049" +
                "B70FB4B52D2EF2141E3C3EC8F4BD914209C2F2EB31FAB2311F906EB48D4396D8" +
                "5C5D9B54FDCF52C00FEDB9F399607C1D95BEC1D81BBF8B9E35C36BC72FD90834" +
                "AE8D4A02DFF4FD65DB9748DB626FD4D387A4E0E5ECB73168AF3BB3C788DFAD4D" +
                "CECCE43F9513EA99F1AFFB2B900F5AC55DE8D7AF96B243BA663500A63E4A35D4" +
                "7257229376EE8C0179396C355DFEEEC03F8773BA1DD5B0807E44EA1E11257751" +
                "67020DF9").HexToByteArray();

            private static readonly byte[] TwoEkuTsaPfx_RC2ContentEncryption = (
                "30820AC602010330820A8606092A864886F70D010701A0820A7704820A733082" +
                "0A6F3082059006092A864886F70D010701A08205810482057D30820579308205" +
                "75060B2A864886F70D010C0A0102A08204EE308204EA301C060A2A864886F70D" +
                "010C0103300E0408C4A9C5FF5EE508D5020207D0048204C8344C1DBD71C360A0" +
                "B4ECAD9F7F43F25272B5C99F1D54A7A0926401082B4E1512610D52B15CBD8203" +
                "36AB72F4E57D083B49BBD606F5C1A268B66E7931E0980A06D8EDF5B5C0BCDA51" +
                "908E21890B24054867912D65125BA0F75B561175A053D2F875C1C846CDC6AFD1" +
                "7599AE877B8CF18585BC405B1E356FD49AAB207BC7C2BBCEF1B0E9FA2EBE205F" +
                "E5F98F825BD9564FA45A7FF011EA41A247AAFA06391C62BAC548A004A139F9C8" +
                "8039B1837066BBF5DD7E8DCEDA3B13ACA5214A53C8D6D748B4DA885CA59741B2" +
                "5799051D59AECE8F06EE0C637406A91070C7DE72B2FAF982BB0A9D937C517D5D" +
                "B0F4D0EB69FDB597F8695A9BC42DD5F87F56119AB5F3E051B0E03E4A069FBF39" +
                "3C5592E1BC28264BDED3ACD7D0CBB5DEE9B426101C6CC5752A5068DF4520C71C" +
                "F10875F1F23BF84D4B6D3A2E133E059A8B1F02B47258F36F84AF4EAC85045489" +
                "971E63970A614CB05C3FA28A711F8DA220C23E463E50B17408E05316F1CB32CA" +
                "37A0C4262081E9C60D897559FF167F2F9B58162399368DD5B85309E9B941FE32" +
                "356258D2EBBBBAB957F496742DB2CF7D8233EBF879887B3F07156BFF9D2CF87B" +
                "D495C684CDF46E451715D4CA1DA21F960868BA70CE88D4D8904EDB97EF69435B" +
                "DC89648A1C330757D51B9D94AF48814109B13EC4AAE6EA99B2ECE5DFDD384F71" +
                "E3A4D39328F5FC55344E2B6EC68D164F57B92AC17D6BF52AF153D431E06770DF" +
                "53A8F14AC579E5E130FC5C3A665E5BEE8CCACC5188191B00EFD13A3A0DD1CBA2" +
                "FAB565CCE5459DCE7CBF8332A3FA1A6E943AB05A2BD28A35025A19DDE18A63E9" +
                "123BBA96B0147221E7CD90CA3A8DFAD634CF8A24EC1AF619CA7B43D844B078CA" +
                "AC708A4D1775AE368B583785307CA3F73C370740DAE2163AADC966BED8EF2648" +
                "28557C1F10BFFED48ACA7ACBB60CA16724F0FD9A2C79B4556A71C7BEAE5549B2" +
                "BDF5B165760DF0C7A0977D923952C9DDC95D89D379E0DF0E292D1534753938DC" +
                "8764451A231F132FE3F40C472CDFED28564002A39ACD4B7059CC284E72D27ED6" +
                "4204DCE2CF30FFF82EE788950CF24358214406CABAB32332CCD7D14A141162BF" +
                "832B1091EEA2C845DE9338D96917065E0CBEFFD292232B20956DFE8116C724F8" +
                "EDB03BED1474460B0C3C45A894C7CEAEA083C2FD5162C926F5DD945BA3BF3A53" +
                "45E82A93F8BCF462AF4C51F4784F8618FC2DB64B4E4A497F0654F573A2F83426" +
                "DFF119440981C9ECBDBA7BDD2AB18F2D62B5EDEA6E2396537EEF3A4264EEC3DA" +
                "4843FF0CD344204C8FDF9C92AF1278937694B30EBE6238AC70D19719D43D77C7" +
                "6B117C4048A7698B822BF371EEC55C1A4C51A13E4A84822F5370616A67B25723" +
                "0BB9B14443A8FAA13414244CFA353E414C9E652C447BE58AEEF982FE4A12FB64" +
                "5FCE47038C15499277FD0EA308036497437DDDF39F596D48FAAC1177112E0929" +
                "234E3F5389DDC21CE14362729AF3EDCD7F2641A8633C13C1E10FBB808E5881D9" +
                "A19778C52E8A8D9DD97766B18EAA9F147AB7B184D7DA131148A70FA0D2FB079F" +
                "E4E4062211D0EF4C3E40D49025BC84C68FC2CAD10F2F5AF80D8174B2A05301D7" +
                "35F3688D854D5D9A2A4646D7F4FD49A16F9432197EB581FB71906AF7D2A0115F" +
                "418AA18C1F14285C7197F3508D374947A8769A91711B0D159A71CA3258529DA0" +
                "C918D8E53E0ADA32E8F32AF11552ED557DC1D8F0F1D027669221C00529B44031" +
                "3174301306092A864886F70D0109153106040401000000305D06092B06010401" +
                "8237110131501E4E004D006900630072006F0073006F0066007400200053006F" +
                "0066007400770061007200650020004B00650079002000530074006F00720061" +
                "00670065002000500072006F00760069006400650072308204D706092A864886" +
                "F70D010706A08204C8308204C4020100308204BD06092A864886F70D01070130" +
                "1C060A2A864886F70D010C0106300E0408E42BF45F0D448170020207D0808204" +
                "90BE13F1716CAB0C61D6269860EF56B1C6695C35AA4B8333CB772579033EC447" +
                "9B83662C6EF04487BD6BD3EFC7BEE8C17F5EAD6E73389A9EFD73FEE959F4FF68" +
                "D31616FA6D24D918F0377555CD68AFFF60F7E7AAF1619C2E1F4B057F5D2CC20E" +
                "DEA3A8683DF2DF5E6D3C062065B38FFC4C16E7AD27BD31742C8732D09114B768" +
                "FFAE7BEFD13303C64E8CA18F2571BBB1FFCC3A28BDBDE510D841FF781D7C615B" +
                "178252E13D24B89D0DE8E47D9160CAE6BDF5E3959BA35218EDC43708F68CB2E6" +
                "B37BFE52D05141B5BBD351C570B18848C68AC15E109467E373904F3AFDA06905" +
                "0C1596D2ACED9D2733EB1FA0CD503B06828944463C579986BBC24B443261F1C2" +
                "2C170F13F3EEAA0FF2EF63200612723AB4A0C768B03C6AC3C6B8D967DA306018" +
                "2D2E3B412EE8E6E0639C282ADEC3899F36D740CF1CEA888824FBCCE2A7A22AF2" +
                "06681597D80B907F50F7044B928BFAFC10F2580B5F7380E2C43BD6F273CD7EC6" +
                "36F3F4F3AD5F2DF9F48CF4B0A4EBB8CB3BA1DDE3448C5ADE45C75CB80CCD61A6" +
                "AFB2E29BA3833A6465C34ACEB7E47CAEDFC1A6B5DB7E3DE594026B648082732C" +
                "1A3804E882DECC2018BCC27A29AAB4B98873099025449B9709EB9C0B5F84EA34" +
                "4A7CE3D0829DE1ED1C89B165A1130DDC8333E54486A94E35A8C017B21DC38D74" +
                "C6C0A685D68103743DD7DFEBBDF0D9BC55DE11DB3F38F762415BA58B2E52CCBC" +
                "72B76D70EA1BA331B24F3817DC635FF59A1247A8F6EB1AAAA257388709BE7E44" +
                "E8A3716A4A2FA0D57E07853EBE9BCE05FDDD23215531BFBAE5E304A9DE44DA93" +
                "6AFF963085CE099665B03357C295D33A338EDFE664EEF200A957484D51736FBA" +
                "73924014152949BDF128248ACEF561F51F08456E03E3533354283A74D0BDA10F" +
                "53BEBD5710C0735C2960188781EBE4B9AC0E658723CBD09C8898FFAE79E5D066" +
                "EBE4586777E1039D38EFF6C74ACB80365341A868C610377C03DE1B691300FC4B" +
                "76A7192FD2D37864CFB336852EE9C2FA8AB96F196A0EDD84876B3278C4200143" +
                "11C933D8C759945E3565C80CE8E7782C4BEC074FA87EF25D951D846FC160A0E8" +
                "ADE67650FAC6B2A7C8F9798CBF8DAD6B23CE7A429995051CFE6941EF7258A10E" +
                "2BD776B6C7DD59823A50BB767E38D6CA099553AD0E3982296DBA8AE6D72CC473" +
                "9F4A9476B113FB06DE80B32F65FEFE0545CB44A3F5940C6776806168C534BC18" +
                "4E73AB8F7B0E855849ABBAFA9D832246329794682331A05DA97848262E9FBCD3" +
                "0A4894B6EEDF42150B23EADB3362C54CEC2CDFA9F36F67AD22389B3487CEDB98" +
                "174644FC2A7C5FEA11488F031AAA30A9F7FE15662C86A0AC5EBAF2FB47CF555D" +
                "B8D1C85410107C2CF40099DA3B281F0DF391E5602B64DA73B585895DCC465338" +
                "29BC0E72F178F41179238049BED59D0602BCAD8AC9E9BC140306D8BDD4150C6C" +
                "4DFD8EA353271B928EA23482E719A58D02515EA83DB54252CB8D466EAC9FD1EE" +
                "71D00DC6D39362883364E31CBC963CD295CDB490074F5C43759D7DD655AA8A46" +
                "EE5FE06E186AB3025E71AAABA9DFD8A105ED86B85FF7C772F596BEEE31CA5BF2" +
                "26BAC1F59123893945BB052A48E9BE2254FF1512B1C2E46D34111F7BA35EA906" +
                "74FBC63FC70A18F095C2AA6CD060A7B52A3037301F300706052B0E03021A0414" +
                "A519F103F114AFAD5EB7F368DB4D0748559CDD190414584DD2F41EC2DBDAEA69" +
                "FB2FF401BD9FC3B57572").HexToByteArray();

            private static readonly byte[] TwoEkuTsaPfx_TripleDESContentEncryption = (
                "30820B2802010330820AE406092A864886F70D010701A0820AD504820AD13082" +
                "0ACD308205EE06092A864886F70D010701A08205DF048205DB308205D7308205" +
                "D3060B2A864886F70D010C0A0102A08204EE308204EA301C060A2A864886F70D" +
                "010C0103300E04087D3CDF9B432B99CD020207D0048204C8ACFB62895927191C" +
                "320EA183A55D7F3557941E6B7C5ED28F91EB4C17EE01B6683EC313B12CF1DBB7" +
                "1A5CA686214B9916292A9EC0DA42B7195772368CC3F05DCD449B010F55F4EA6C" +
                "C6061C8F70356A4765D6FA5DB4DBF6AF14BCAF41020FF275DE691229BD5B61E0" +
                "C06FA2239387F3A09DF650919723E0DF02326980B46B7425D5330CB6CF77B0FF" +
                "CE374DE81A1A60BDA1DD3EA7DBB60AA36FF462C9806893FBB4774C3F7C1F0C9E" +
                "910C6D0771779C7C911949B48BB636F44A2D907AB7DEA864A4297CA8DC3E58BA" +
                "3654E41767A823F7BE90932099A07CB26AAC49EFB9C9046948B5BEBEB794661F" +
                "8E7C8E0EB73521DCC6D6DE25DAB6AA12194BB7FC3949174569270A781792A6B0" +
                "ADBB00B64944BD0BECE863D9DF9A177BDD11E929C76355EAA1814E65281E3672" +
                "F566031B423E5B9E514AE03BADB6B4F894CE7AA1BF1433D4CC2B2E9FE7B07823" +
                "6CB9B6766514026FF0F432FBC007951A51569403762956B5DDFC0B2EE7B69297" +
                "0A4E25362B6DF26454AE56EECBF8CD9FE7850CE9E85BFC939DCDD94990872461" +
                "CB32B2840DC87F200DE00CB13DA06A845ECC67AAFB77D53F06B216E2D1192BA9" +
                "150690CA3CAD285BAEF34F3AC982F0F9578915652A681A79F46B808603264A2B" +
                "EACD854CEE0D9D9B284F72F863E2BAAB8D40C102C9563BD806C7FE6EC5F3ABBE" +
                "0F3995F65CDA6AA3CF26F086499DDCE7A28F3AD4A15CF244B4A8519CCB181CA5" +
                "F7A1967A9A079B0E22629720544BA1CD16B20C34DAF0B0A192A838A7F41D18C7" +
                "AF8E3CD67D5CE688A8A5A7BC19B1852DED772A9A648B59E5421711EF59A765E8" +
                "550832DD8CFBD8BC19B8E88E8EFD14D384F747DDA2FDE6AE9189A2B8FCA2B9CC" +
                "901FBC6621C40EE0CD77F3A4E8B81451C6AF6B18FF53B49FA97697A54628F8F7" +
                "F87EB621CBBA472AFD1F1442F3C39051FDD746C75F7256B3624FECAB800D441E" +
                "AF0FCD857DE979E79B192F1FB9E330C85629F0E6727003F1D58B03629144F14A" +
                "F0761A09EC9B4AD459E798C2C20E32F1B67BB13DB2A04E2445DF74BBD536AE48" +
                "7329EFB6D5297F37F36A098AF4DF3078044BD2F44F64F8598D3CAF7732586AA1" +
                "EA5BAD85CCB8348B2CE595102DF4DA0E791B3C21BC860135B2F3A417FA2C14F6" +
                "F70E75DC0C2F11C0C4B7154BEEE2EF251062E1A6592E7BC7E45740750C0F2B70" +
                "0EFDBEF16A6F11D07EDFB6502419026F74929FFDB78DD2379BB83B02DAC29838" +
                "DCF3A71B19E3D1C8B87ABD7970A1E658949E9EC53147E54FDDAABA0A21A04229" +
                "0200B5EA2A9420941102B6218C16CC45EDB08627504623B7D3B8BB78300260E3" +
                "FA1C20CABDEF8E8A1F5D1E875A321F94BDA20C798CE784484E7722FE844D0377" +
                "D5B6FE150BA29D428AC0C42538BD41908DCEB1DB32E08939750D2454151CA85E" +
                "64F42D0C4677FD7E6E88B881F4B1959050CE4D22BFAE7D0340EBF893E6E9961F" +
                "9C72F1DD16466351A12A842672C0B79D7E05A272B393582B7B02F21BE35FC165" +
                "31E0B7D8A317A45590D32BB2F61DCEC5AC7C15EA86341397EB3ADB80F8FF5A81" +
                "17B2AEB98CCFDF6AA6E1CABA3547822EBD697EA2A82AEE5C9336C6E97C476D5C" +
                "4EE17336CB905EC26D13CCABAB82F553A39695DC3D102CA17805B2F3259B73E9" +
                "945617FD77BAB0E5E55B430164554DAD84C27454BF05D52F6B130AB69520D01C" +
                "CB72A1BEFF18FE1D24DF3E5DB85DB36342B75FCDC9D1DB924A6280969DE29DE4" +
                "3181D1301306092A864886F70D0109153106040401000000305B06092A864886" +
                "F70D010914314E1E4C007B00380033004300420034003200440043002D004200" +
                "3800300038002D0034003000370030002D0039004400440032002D0045004500" +
                "38003100460034003300310038004600410046007D305D06092B060104018237" +
                "110131501E4E004D006900630072006F0073006F0066007400200053006F0066" +
                "007400770061007200650020004B00650079002000530074006F007200610067" +
                "0065002000500072006F00760069006400650072308204D706092A864886F70D" +
                "010706A08204C8308204C4020100308204BD06092A864886F70D010701301C06" +
                "0A2A864886F70D010C0103300E04080242E511671DACBF020207D080820490C0" +
                "257A60FD19BB77BD42308DC6F9042BEE22410F9EEAC205F23C907B838043C8D1" +
                "EA8B9DD376DFB810AEC783603B99B15D44F14674F83B6D7A11DDE52D9D1FFCDA" +
                "28559EDE9C4313DE021D2DA3D2D528D3A5B92ABCB82F25EC2581CCA0742BFE92" +
                "EA499A8C4A9367151B7F6F6BCA7E07105A38C3ECBBDEC4B5C42063B12FC587C8" +
                "C44371A6E9B171E9C9D2AEE4463FC8C2FD9842C675A09B8B6EBD2BD3DC207FB2" +
                "C0F1875272B96A980EAB1B1735C0FA89E02FF690014370ED6FA544DA209F3981" +
                "D94946B6C2AC501B502AA3ADBA38C78ED5F433F70210F6F9B6CC4E47A4CF5EC5" +
                "DAC225C8F4C848452A2D8B9FA4CFA0ED53728FA970F48BF8C88341894A2E54AF" +
                "7D9CFEEBE56C146237B901233DD7B296D8C9A06F7B962AAB67D1721972B8C3EA" +
                "F34EC994238C500D077A876E560067844E3C9086C809DA7D9D04480858DCD311" +
                "56057816354497D9D41C1F34456FCF479B1FCC85427888D04CDE4E8252BCFECC" +
                "1A61A605847D0C7DB7F11225718F7CC889F9C9969524E8808B17925FAE2CDA0B" +
                "75F825F4F5F35A482DB0A0E3F305803C9EC9EAF9849A45DC3F6D09A018B4213A" +
                "E5DB24B82B4571A6242E41ABC581941A5C9927171BE89782989DDF72E6EE857D" +
                "2FE7112FB20D43071866B751D30A85DCC2DF0C66E169B7857D6430373E119B9C" +
                "0DD8DE260E12688D74DDE8FE2327D5D9D24680F95892F045E36710E2D25AA58A" +
                "9858F94202378242D77BB56D94A9EBF08BDB7EF9E893BE26BCA49B52D4881B6E" +
                "8AB0EA44C9D58D32905E63C6F1F1AFA2C5D1AC8C93D958626C9A47EE6CB6D7EE" +
                "1D28F890D56DBB1F41070529C4A029C7BBCF0CE44748D8318EB4C2756B0979E7" +
                "0136AB0B011CDD3D90EC48F416E63DECB183BA68507B3EA2A981F19F5D9A5EE3" +
                "F83E821C4CD0B8DAD888626AB3778D788F4DA9A46CD3CB039545A711144D366F" +
                "5D63699567F2015913D8C87FDB60C5E4DF2B07E120E9E22FEC81333429718B37" +
                "A878D6104FFEB0CA9AF281DB0C64441A24A3020A562E53C9F4558A0AE7C4DDFD" +
                "A165AA29AB73EF31C491F9DD7B3991F45786E9DDF3B44202EC847F8E99D829E2" +
                "6A1D8918D3EC7B84936CBF0E7CDEF220DCB3E246169EB1F5EFECB54B90D18CAB" +
                "CF0CE8628C8C36D5066735CFBE31C8062EE644FDCED0E2136EABB4D67A1ECBC0" +
                "530A7FB5CBE87233F76CF585DD4A6D15DE345A8664D4B44B5CAF753C4F5410E4" +
                "98DA51F31FCA417EB474B95725016ECFF71197BAB7541AAC3F679A493A5B6292" +
                "895967DDE578E5F0CF2017CB210DF9C36B23715BFBC64D40B5005F09171654B3" +
                "BE3424D9F749AEE51F5CDD350B11AF766218191A564A58BF8791F4B69D3627DB" +
                "6BFD40FDDD9BBEAC97B73CBF156514691CA7993978C416DFDFD9499C439BE755" +
                "95279EB20B063EA6F9B6A6892B4C95960639C5A76E8E7DB33B55D9DDD67F1225" +
                "88F4331C3742F4446A0412C1992CB7C5D71F7F3F4F631CB383D58E0971FD54C9" +
                "F9EB711D9CD97D4082B337B7E97859D70DC1326DFCBE7EEE678D544F7266424B" +
                "8CA931B46C750A2A156C55955F368AD9B3FA6A600126D3E99F5D605587D37FF8" +
                "E9E228426C65653412C92A380B5ECABEAF61BCBEC82195B17D036329C8454C88" +
                "DBD9AB3B4D5FC3AB6978088D09E690303B301F300706052B0E03021A0414E6C1" +
                "EB5F19E56891D2AD2D86C30FA62CF75AFEF10414CEB6CBF0022C852539BCB5CD" +
                "5CF4123B4A8FF97F020207D0").HexToByteArray();

            internal static readonly byte[] TwoEkuTsaPfx =
                PlatformSupport.IsRC2Supported
                    ? TwoEkuTsaPfx_RC2ContentEncryption
                    : TwoEkuTsaPfx_TripleDESContentEncryption;

            internal static readonly byte[] NonCriticalTsaEkuCert = (
                "308204243082020CA003020102020401020304300D06092A864886F70D01010B" +
                "05003029312730250603550403131E4578706572696D656E74616C2049737375" +
                "696E6720417574686F72697479301E170D3138303131303137323532335A170D" +
                "3138303431313137323532335A302F312D302B060355040313244E6F6E2D4372" +
                "69746963616C20454B552054696D657374616D7020417574686F726974793082" +
                "0122300D06092A864886F70D01010105000382010F003082010A0282010100B1" +
                "74BCC25C16B5B726821C2AD59F0EDB594832A15FE186A087FF40A8BC9396C55B" +
                "C0DB2B3CE3EC5EF26A11AA87073348435417982D29C439FA3EC94A62B4BCC9EB" +
                "CE0236B7E5306B3144E71373B05C24D3C1EE7A4D263BF11FC54D09E4B674F593" +
                "389AAD503930EB7CEFECCA3A64FCCCC15E32E4B304BDAA36267E6BF2DA9584A7" +
                "66E1862758D04E7FF9CC5C46CB074DFBCFAFDC355BF848337CD38331FE8556B9" +
                "F350C6780C7260F73FBCA77FC454247B018E937D2002C9590E07804233EBC28E" +
                "7BC712ACCF6A125EA60B86A87217B23A91866BEAAAE842D0D0D02E87F5F123AB" +
                "811EDCAD7A6819E88B0F0D0932D0748EE02726D7138B1ACEB7A6D4090245DD02" +
                "03010001A34E304C30090603551D1304023000300B0603551D0F0404030206C0" +
                "30130603551D25040C300A06082B06010505070308301D0603551D0E04160414" +
                "610479D21BFEAEC87835A7D03714613D566F25D3300D06092A864886F70D0101" +
                "0B050003820201006511E16CB557E9936363DF92EA22776DD9AD1ED6ECBFE1A6" +
                "B9B718BAF9828AD2B8CE5450F56E1371B54577DE1146FE82089A4331D063DA8B" +
                "3CE786945ABBE4D11F4465068B1C3E4CF987C2C014BAD8BCECCC946EF3C5C56B" +
                "0DDECFB11967A10949A6D111DC96C627D2F1D7969CA6BA7AB131367E89D950C5" +
                "0B756E516CD4CC6BE679BB53684517236846DCE33BB49235CB1601A6657CC154" +
                "8C2D57E5554B2755FD047922B0FAC86437758AB72C1A6EC458DB08D93CFB31D4" +
                "8C723494978D7F32B940E5519852F9B6EA4F0363B59FCA7F8A6CCDE595896B91" +
                "869C1D90FF2CA0CD29F9AA5FF452470B752052BFFBE2CBBE00C68F2DDCBE8C68" +
                "53136118CFD1916D711BF90021C04FB5879BE0974228F79C71532FB2B5DDA1E1" +
                "E89BD5EC9015F2207F7C55D72E976A87479AB62A46465E8D75466492F82CA574" +
                "D788B27A5CA251C9BE3A4EB90F6E7F238CE8AAF46E598DDD70094E148DAE4DAA" +
                "21275F79095ABCE8A47CC362386FDE498D102CCD42B2313AC88FC11E21DF8283" +
                "631329E61F35E8DB12EA8D33DD02B8987C7FC38242412EC97CD52A7A913C6992" +
                "D87E71A75F9654F7F9EDEB80B0BBEA25C5A22CCAF388D109DB0EA7C79247CE4D" +
                "C89F390EB8C6CCC19FA9EB5DFC5BFA4457DB30B27CA73EE1C19934C8BED63E58" +
                "F227222B86010D9521324BDDE04F47BF5190778C6B812ED78AC7DD8C82FBD0C4" +
                "0DA1EE040184500D").HexToByteArray();

            private static readonly byte[] NonCriticalTsaEkuPfx_RC2ContentEncryption = (
                "30820AAE02010330820A6E06092A864886F70D010701A0820A5F04820A5B3082" +
                "0A573082059006092A864886F70D010701A08205810482057D30820579308205" +
                "75060B2A864886F70D010C0A0102A08204EE308204EA301C060A2A864886F70D" +
                "010C0103300E040830D97AD5E5020804020207D0048204C857D388EF52DCE112" +
                "B6D73E8611F6971BA662ECBA8F80435F83126D213190D6F77369E8D1213CF1C1" +
                "7791AC985C7A0125D1D4728B841FE7BE3E9C44469CBE672377CD97821B49BBA2" +
                "75789D64B648F4F243136E9448166EA366EBBB973041C437E1BA70B609761F03" +
                "03A5ED03C041732D34F070555B006FACA2B8639A0B609D1A7F8A88ABD260CB69" +
                "3365D9181A638A6ADF69385C96BC094E6EEA507160F18A22E0552183C586A45E" +
                "C680CE10B56E1D3956F704C89A0981429C470DB378DE1CF21BD67E9EEB43B37F" +
                "E00117CD3CDB02398FD0D7FD4B48EA38CBF6B19254E957AD2D8A0A4C3AE72355" +
                "590407AC9FE2622C8AF04BF17E62CBB213F9D29999BF184532BC64E2ED1A5323" +
                "1501741A1352F492AFE713503A950DF12E9BB505EBE9C80DF4DB6CC9E1EE0CF0" +
                "02C9CB145E265F7D84A448B9C7462CE25EEBDC9AB3A6E1C996FC535FF7627163" +
                "1CA7E36C1614A9113C96EDD951F2B3B7508DEC6A2BE6889436A741CD170B6201" +
                "D546430CC38CE1E874D78A9D6E4D9665CE8261368FBA08C7456E5B01723D3465" +
                "7AD715328341B4BAD14825D1DB1B0030819B61607E2F4FE76D0EF1616E2C1F96" +
                "4395FFAAA4A9F7E833A1527B630D862DFF5C8DD6EE6557F55B429C9088020D10" +
                "309070D8BD46B1512C0D6B68C8C00EBF215C5DC3DE0BD8B4A92E4C3115687194" +
                "D7DEF524FEA4B02389388C7021BD85EDF13BE19086D08AC682EB8B37F1AC6445" +
                "67CF6213363D889536CD8A4287A9DC16DD5497A8D06A489D6AB12E4943784EDB" +
                "559FC02C7DC1E190A9FBCB8EF7D83AEDB31AE1BA8F356742E539E4A7B9D0A516" +
                "90FCF505BFD5DA6AC8DD67439C2CE9E8D3DFB78A88581BDF0EBB89B810FA7894" +
                "78D5A5BA44BF287BC8D98DF1B286F2B9109430524DFD5739405E46C755F9C943" +
                "03C95FF6E89400D1E1D1E814D795FF0B77ECC84AABFF6A8D3C665770778CE9C9" +
                "A9189DA1E257988AF6588A596F5534D91FA4505581DBB0F8588F97CC3177788D" +
                "131A2F03972DA2753DCEE18965E032A5326CF50378D7D98233A913387315C71E" +
                "3FB2D81A78B537BBBD4408C2E8DA4EFE975EFAA785BDEBA40C5CFF9E25CA07A0" +
                "77DFD9744FE20F783A38A274CDB85A374D1E7723473106DEA578B14C766FEEE8" +
                "6446C61AAFCED190892AE44C8BDC72D5178C3AB1BE9600CA15F5D3383A6219D0" +
                "675F2DBA9AED44BF8702777EA6902344AA572535217EF44BEE37C6E507FEB4FA" +
                "62EF557119608466CD1339C242AFB4F8E0E9EB403E41872B3C5A34B94EF2EBC7" +
                "91111687C764E4E20A25EF07DAE9E402FB08B79EDB4F5B5C3ADAB14E3CA9004E" +
                "FEDECD8DA9D4791BA96F6D6493B301698F2202DB9834B423ACFE71324716FDEF" +
                "6D70FEBC98503E914593A1F511BCD0C39425DAA9981B6BEFA122F8812564D14F" +
                "B6383F3CB8C2C41449E9B58B3D4EE27651C5B20CEBE786312878E641C20531B2" +
                "909BE5727E4C4C01FDEFBC635292A663B53A8EAB29A1FA4CEFF11A02AD511AA7" +
                "F2FD338A86E1876B568074F50B33835186C71C3854945AF082B4DBBD6865581A" +
                "139B3973A3FD5E62AD88C51D636D616AB18EA808BE982C7C51B20FA239D07014" +
                "CEE766F9CDF5D592B1D31881AA5939C670A7CCB48D234268D61031A27D99728A" +
                "4701EF7A241C45A26799C45A8A0A02EA054215973B6F156520544DBE0C3A89DF" +
                "7BBEEB7AB754495781A4A37F4CDD64B1A3A535826B00E1A710AD4D4A56C17662" +
                "3174301306092A864886F70D0109153106040401000000305D06092B06010401" +
                "8237110131501E4E004D006900630072006F0073006F0066007400200053006F" +
                "0066007400770061007200650020004B00650079002000530074006F00720061" +
                "00670065002000500072006F00760069006400650072308204BF06092A864886" +
                "F70D010706A08204B0308204AC020100308204A506092A864886F70D01070130" +
                "1C060A2A864886F70D010C0106300E0408DBF6A30F31A06E2B020207D0808204" +
                "783C0D88298EEF3EDDB8C04416DD28F2DA6654714BD9092529FE1181153A60FC" +
                "552B4D62B0C1F53C6F6337A7C774DBE4396690431E55A963AA509A2351104B91" +
                "E74B9250AA58812954181B1ED602D9699105960C7B82E91362946292E65C99D5" +
                "80DCD3B00FCC0FF4B25095AAF4B5E67886B817556D8B69B3016DE071E31F2B86" +
                "3612A6050FB7D97C5454CE63B842B02FBA72D102DFAFEB01192CCF33BDDEAFB3" +
                "950C53E1452B449950CA860BEF314B32AFABF6B9F2BB1CAFD064960C073239A3" +
                "EEC38BA9B30BDF0A9DBA3FCA6F22F47DC8F593BB7102FE0D8039AE5B2317C9E0" +
                "2DB059C99F06708809362C1676D14D7E5F3DB30E0090697366DDC9900A218E7F" +
                "99851838A111B9C9C9DD9A696DD096DDAA23175164034407463AFA4664BD5E70" +
                "3B6AFD659D6C7CD266ABE731F51F96A2D148B919B83888E4416759169C304A15" +
                "57251FCAE553B5A177DBB5366B031B59149CCE263601A0231544CFD7107BDDFC" +
                "4037AC0AD99F93D001C4CBC4DBCEDE235875C20789BBE8BEDFDE63D1959C25AA" +
                "1E410AB081F03CF5562D814C54A9B66A27E74DC261D4E41513927BADA1E993D5" +
                "20EA81D592B1D4ACEA2577929229B60A8B0AA7499037F3F7F24C1E8E980A06BB" +
                "6B953090844FE068B611DC4C880A4B2FE21F82002C4305A9AE27C1B17607D59C" +
                "89589F122721FFB6DC2D95EFB7D96625EE3C6C252E0E3FF25D4407549358F995" +
                "D9911E9021303FB711B71D7D5F61D6BA845A456B9A832926A098E0EDCA081E53" +
                "9FABBE54C09DF90D37D349FAA3A9259432491307C216E8D3C160E1D5E994161F" +
                "BB29C9BBD789CCEB23591B983D35CE3001D7D4A313A9D66356D5B3BC0BB061D6" +
                "49DFC15CAF2B8B70D3FCF40B1D412E60FFAD5FB4A0F2944F019CD2CD26108345" +
                "20771F437BF27A586AB0866BF1F5920C488648D463A2C430E217CFB080E91930" +
                "05589A9670FA9C75050E45100A553E9A21FE300DE621B12CCA03FE189CE65367" +
                "B1CC452426AD21C67925894905CD594E85C354F85748994C34AAE7E281DC3C71" +
                "81BEE53119708F6C2D29B2CA987F5620650BE2EA087FAF976BB58349E8B67F67" +
                "FAAE11559752C31EB34A72FD4BB23B363255530F92E8C0C82FB7ACD6FF4DD7CB" +
                "AFB831E624522FBDD47A8191957BAF0E9998AB61C5F839B6DFF3A568132A1A21" +
                "643A576D3562EF72294B13E6662ECF9CBFEC602C7AAC5E01D43759917D95BF3D" +
                "D572A45B2CAA118965764470C163FACA61D06693273D869214816431963087DE" +
                "B10FE0316352440E87189532D950E0A9AE5ABE9926907E52F1E4F135B467F4E2" +
                "8BA57B9F371FC934409526E261926E484B465ED10CFDB25A2F3E6838F41C37F7" +
                "DF6A88E7063EBAEB6C426B0A4C7842C58CA49581F14337FDF43FC22DAF79900D" +
                "306E34ED9DACBD6072DF4F34588F28F0F4AE13B431D9F47ACCD6EFB1C4EBBDED" +
                "D507512B820FB02713948D45F3C595330B271308F0E6DAB862BB917EE9E2B1F4" +
                "17CE21A26CC17358C11E2BCC514EB241D0767B4971E9B89F94731843B508C20A" +
                "54345F58E99D430065326FFEA3E1FAC40E24BFB17A51A884CA022944A27436DF" +
                "8F2C4E296A728496C38076FD3F14B007FEFF015EB42329F7453037301F300706" +
                "052B0E03021A0414A0E907FF695A237FAB54BBB94CBCE689EE0B4552041426E2" +
                "132250203B3235FD5023D999B747478D8873").HexToByteArray();

            private static readonly byte[] NonCriticalTsaEkuPfx_TripleDESContentEncryption = (
                "30820B1002010330820ACC06092A864886F70D010701A0820ABD04820AB93082" +
                "0AB5308205EE06092A864886F70D010701A08205DF048205DB308205D7308205" +
                "D3060B2A864886F70D010C0A0102A08204EE308204EA301C060A2A864886F70D" +
                "010C0103300E040885A220CF9DC89867020207D0048204C8493E01DA35E65667" +
                "AA4C7764114555B54D086913F0892CF30E2AF6C669E4D2A174448E2455FC889F" +
                "E1AFA684549D8C0A316A3FF42BEB3D223F76C65C78E834F083EE6D84C896938E" +
                "F66306F6A146C3E1D065B8994E234D50D0617B29531BB95B37F16BDB2C7B2AE3" +
                "834702AD58FB29BA6342C089C91699C2EAED188CD02DDE492DCC46E0D5C347A0" +
                "22EADC1986140686F8AF7F6AB77B324E80890A8CCF01C954A48E518B02835398" +
                "42FC74BF41AEDFD5F650537168434DD53E6B430BBC9781CA10D4A479AAC8A94F" +
                "25B2A594B786E8CBD781C7CC137C53A98AFF2E11790671CA41175A708DDB74B8" +
                "7F88548587BE23808A696B5CC165A4FA4777D9BE9F2F5BD0CE0BEB6C36353FED" +
                "472790064A42FEA5456A38530AC10ED7862FC44833D0CC2C8544A516AD4D9118" +
                "4A94A4D1356308FDCCA27599728E0B089CE08FD4B00D823162DEFAFE9E703310" +
                "00D2D363EC936E6DD8D100ED8145306E485C3ABE2C1383FCDED4D29AB2A6534B" +
                "AA939587492DA733F1488C308671A7EA2D6F62088B4AB497AE0267CEA5E80C26" +
                "14FD5513B0989B2B24C218B0ED98605CA56A4DEF167221E904CFCBF8BDD897FA" +
                "E3265672B3035C308E82222B6FC8F43F670B4B26CC26B644019EE959386FCD2D" +
                "F5DF9A8E48B1C4CD523C04D8CF49CBC06E7445B8046EBFAFF74AB822B38174FB" +
                "513FDE0930CBCF93F6CF8D069969270E3B486B3FDC34D31FC35864C5DE52425A" +
                "8C581F6CA430C3841DEB9D1ECC7FCF353C8023C254010AB993651DA372CE3F10" +
                "A0B5CE853FD563D9022B435D99D31603610600578DFDB5EFD127D1006DCD1942" +
                "7DE7347E3539DDEA35A6CD10FD414D854D3D293828914A11A160A08FE1EFCE0D" +
                "792DFE648F0948C232DB46222B212B6C4DB1DF051E5D399F34F695DA4587BA89" +
                "522D98E930E5FB602F90939115A64D10C854A4032033BA09F8EB230D285FA8CD" +
                "CA103B1C699B78271AA397B6ABCAF0D12DEEA3AF63EBA477D563D291537079BD" +
                "96BDCB960C2A8A2ADFB687B969069956D300E295355836926495B80BFEB2B0BA" +
                "848A7F05E3DACE177D1BF127F0A50572D9009F1E10CD8D92EB034DC2B729E3B8" +
                "34862A8C6BD39E35B83E6AADFA75AF222D52E64F882F5A0BFFB6BF5C20AC976F" +
                "32D42B53E7E857F25C4059E6E226E7E29CBF98E617A8D6B5EF9378E68BF705D6" +
                "8D48A1E687BF13347B782B0CBF11B9985A84B0C3B5C224E7519DEF2C40718C27" +
                "D6FC5F0868B0E6CF74A1858EFCF63A3B350C4BE8C0D561678B06ADB9EDA678A7" +
                "1E6F1ADBE74231C690B10F8DA14055BE670719ED649669F75B71A01726696868" +
                "D9A962815D7B2DD29E33DC47C0807614AFF891A3A687FAE688CC423766C7877E" +
                "A87A46E7F7E7FDEB4B763C7DAE055F5EE5383ADF8F9220FF4613F8E90D8D03B6" +
                "EBB1A5C8AB3AFE1A8FF0C4624800AEF4601E476C7E18CC3E9836D8E555EE7C4F" +
                "1306065EA84DB84D8A3E4AF0C2B1D897AEAC2F8DB839EEC0E68F8E15B6B9B8E8" +
                "FBB28608C246A449644AB8645830F11CB20594005EB6337A8159518F7F1C634D" +
                "E571E1577E1E6FFA89E9457F773079A3DC059214FCE5E09495D6B64F0E2D69C8" +
                "38BD14B10AB2D1793E6D6AA0DF74220F141A6958D73228D5AAC194B3415174EA" +
                "92572B367ACB5C5596A63C9E695E93EB9F6BE647FB0EFF6DAB3A29D6F3186380" +
                "17EE35CED6C5B213EE896DE44DB79D6F70923B0009D2FD6F1541AEFE3E95149C" +
                "3181D1301306092A864886F70D0109153106040401000000305B06092A864886" +
                "F70D010914314E1E4C007B00340042003900330031003300370032002D003300" +
                "4200440046002D0034003500300036002D0038003800340043002D0042003200" +
                "43003500300035003200300042003300330038007D305D06092B060104018237" +
                "110131501E4E004D006900630072006F0073006F0066007400200053006F0066" +
                "007400770061007200650020004B00650079002000530074006F007200610067" +
                "0065002000500072006F00760069006400650072308204BF06092A864886F70D" +
                "010706A08204B0308204AC020100308204A506092A864886F70D010701301C06" +
                "0A2A864886F70D010C0103300E0408692303616098813D020207D080820478A9" +
                "DCD7CA2C409B30FC9771D293816CF36EE63D1DFE738DFCBBE0C43C04995DAE19" +
                "24A9FA11922A0737E1458A760A6168686638DF4C6D3E1B621499C5CBACEF6953" +
                "6AB12FC6656DDA61D7DF31EB7A4FA0DE62E931C2B22CB30ED58C4584AED90978" +
                "858B24F416D6E1EE786F88CAD806D52522872B8D6C61EAA4533C12257E7FAD73" +
                "A797ABB89903D50F42C63DD4529DDA07D8E6442E91C4D2F4387404AF4C79780E" +
                "29668B63E1C0B926D0A115087D5060EFC657EC96D29D7565F031C5271FA77323" +
                "568FC7714570A578C68A6FE92B68EEA7659340DDD9ED1FF51085341ED408B91E" +
                "009A42D49732BA66A93E43CA5AB8658BFC2D496E4AC8FC0E119120C8FA108C02" +
                "6A5E4F212EBFB91CE1CF879875918312D289159AC4AAF776A4BF3F06BCA6BBD7" +
                "10792147EA54CB1BE54936D7987A0710D6CDF71F5DD5470A265050A8D7718E70" +
                "37D333A4C932912333695927FF0AED530B00E119F5CB18A3518E472CE0EEDAF0" +
                "0010AE887F9F7FA911DFB9F34D9122DA2CA220E543CD5691D1AD11BA926E0158" +
                "9BEA260EAE698F62212F2FC2AE194A188109EE4EDFF7E1D90439BD31E32998EE" +
                "845AEE83B4EB13F779CACC3B6C981A122A0EBE543052D4587D0C0E300156C24E" +
                "168503D4BAABAE11764BF13D3D037F283A824DAC692B645F539140E2EC8E0146" +
                "A090BB3BA2FF03D7F3235E73425D4F7161B02CA24B917AE5AC3A3BA540610946" +
                "80F3DFB7FD21A9CDACBEC8FA90956CF8F59E786BE57B003EF58324DE1BC54004" +
                "27BD07C841A50469551FD3A4F40788C2BE8D685AFE3D8F4110F14B9B1466247B" +
                "4A272706D8D2793457FCEB9B12F39F3B1A225F52C13FB7181EA43C07262AEE55" +
                "72EA80212643B208C9C708D2E813301953306A0F5DCD1C0F829D952ABAEF61BF" +
                "68137DEEA7F9F01C66AA2589B9F1519B382B7948EAFAF80C7B0A7792A55B9847" +
                "EE0A40C622B9D34AEB2EF5F7708D3BCC0D9947334EC8BCC89BC5EEB412DC98EA" +
                "5BDFCD322D67CCBDB2ABFF57E508DDA226C89B24AAB4AC5433AB6227DF11AF99" +
                "7FB8ED5C7964BD2755DEDA6A9EAF1FA3C3F13DCB878C2CCAACFD45D1F9233177" +
                "363ADCB3F6B86EC0C84FC0BFFF009192ECD08378055658BED34AD6443C7229B3" +
                "D8AA7D9B086A2952168F44951204DD67B18394E19E41166A60D52123EEFCA5EA" +
                "2001135734E3B18F98B92C08EA0C3D57A697032EA89691DC7C57651E498847DC" +
                "ABC31DD71CFA992EC7F22BFCBCF3AA8D6EF166032567B77D218EE831E00F4333" +
                "00D776BC1AD652902DD387A9A4D90E9A56A9784982F8E4BEE9DE15FCD74101A5" +
                "36BDB3AF9A85E7D6757F073C66E5853B9B65A153915CD453192155733D3EBE08" +
                "6811580CB50BB73D519896FEE3EF7DD91C3E50F8AA6920DC6B0331425DE7BC4A" +
                "B05B2810B1475CBDAD17674452F082AC43CDAF47374006131B73C6B55C8340F5" +
                "AF89B24BEDF6D2C257F761432ED864D37B1F36DC717B7E84D655C775360397FD" +
                "E0E7F5A3F9430565DF4D1550BB1AC8E4CA761BFB47C2AE5AB50927138A075A91" +
                "7D57E5006A81ABDE9E7361AD4ACF811E82395F0660B42145B4D81A027D039F2A" +
                "117F641226C01FE1A993CA8FD3BC1AC0F4B90396FB53ED303B301F300706052B" +
                "0E03021A04148527AF9ACC926A87B49C07238F570D60B0588B4D041467593F11" +
                "3A8077255DF0A79741DBB93B70E5E352020207D0").HexToByteArray();

            internal static readonly byte[] NonCriticalTsaEkuPfx =
                PlatformSupport.IsRC2Supported
                    ? NonCriticalTsaEkuPfx_RC2ContentEncryption
                    : NonCriticalTsaEkuPfx_TripleDESContentEncryption;

            internal static readonly byte[] TlsClientServerEkuCert = (
                "3082043130820219A003020102020401020304300D06092A864886F70D01010B" +
                "05003029312730250603550403131E4578706572696D656E74616C2049737375" +
                "696E6720417574686F72697479301E170D3138303131303137333331375A170D" +
                "3138303431313137333331375A302F312D302B060355040313244E6F6E2D4372" +
                "69746963616C20454B552054696D657374616D7020417574686F726974793082" +
                "0122300D06092A864886F70D01010105000382010F003082010A0282010100B3" +
                "CBCBEA8EFAAAEDF982CD046188D2F57FE9FE4F4FEA94ADFB7DE47AE731636624" +
                "3C4E5F8C2821BF0B01A32048E8A275FD9B3073E0DA2E8870FD9671BFA600A47C" +
                "8588AAE88150448E6B4C594C5EA288119BE9A3EA369F66EED7C499738B8EAF84" +
                "5E6B36BCEDF7887D061BC86F12D40982199C1C41CCF5440AEF592A482931B541" +
                "1B0E0400FB24AF838BA89A3E1393C24B80B4C67AB923DE68B0B8A2218DA93C2C" +
                "A4908E3B906BF3431839CA6D3FC2A4FC33C4CB58BDEF8982A39DD186D0BB88E8" +
                "E68819C4A7DA4D29F48F1C1F00098DF41C140FA649204564A3AAB9D0E7486252" +
                "77F40B9739ED07D385BF9C0F78528CC5DED08A84B6D392874B2A4EB98B3A5102" +
                "03010001A35B305930090603551D1304023000300B0603551D0F0404030206C0" +
                "30200603551D250101FF0416301406082B0601050507030106082B0601050507" +
                "0302301D0603551D0E04160414743B8DC143DEC364FA69992CB3306A9EDEACEA" +
                "1C300D06092A864886F70D01010B050003820201003A48278ED517B381CA420A" +
                "E3C12C31C22F49D1DC35EA60B018BCB88C6C3E0FF1E7A301E525B5592649B496" +
                "7BA62D89AC748DB026CBD73C9333AAE89C5F5D76FC77E970F214F1508CBFBD9B" +
                "09CD50EF93D61559004027E80BCE6809F196633C22A13CA2CD7596ADE5B4A23E" +
                "BB4E03367892433357636316F0D235D3DACFEB99631CB437DEB04E4A074B1CBA" +
                "6C6D722210B058285A3E124EC8148E31B3DE37DCBAECF560076FC0E50965D482" +
                "DCBF529127CBE2440BA746DC71765516D480E68805C7668A0B538BC489DA2FE5" +
                "E158BB6786A626BF0AAB74AF042347785B54323CC019CA84347BFF57C025791D" +
                "69C558A605D46C50297DE1E9576972053A3DDFE5EC8FD2DE0D48B80252C39EE5" +
                "4410AD46D8128A453758DA442CC2879290A50232B13920D9C6800D8773C2FD82" +
                "D11755C336CD6416FFE26F97599A29E5D18227949DD4385C74D29547D6C70ECB" +
                "CBE006AE2D18BCB8708C50E7C46D442A8DEDECEEDCDEAEC47042957B1D18D410" +
                "96350438DCD8223B17E5FDC3C9C0D9BD47D191E6142A601930817E30F8E0509F" +
                "D5F5FE269683494F08C55B9ECE6E3D503C81A81F40CC959B858E52BA086D95B5" +
                "8DC13492C128B64F862E7800384C98A1303EA570D328C003E2B944A804B9794F" +
                "A5C6040881E61510E90C20F21DEA0A73E0DA5C1A2D178A48D76CC8FAA2ADA660" +
                "2A98B50AC197B40348A012E83A98EFF2B8D64900DE").HexToByteArray();

            private static readonly byte[] TlsClientServerEkuPfx_RC2ContentEncryption = (
                "30820AB602010330820A7606092A864886F70D010701A0820A6704820A633082" +
                "0A5F3082059006092A864886F70D010701A08205810482057D30820579308205" +
                "75060B2A864886F70D010C0A0102A08204EE308204EA301C060A2A864886F70D" +
                "010C0103300E04084BEC68D6BA25EEEE020207D0048204C8E534D459438245AC" +
                "EAC40458DAE1FDC42FB28605913F53F859CA1B7D62E385E48352F956A1B4D904" +
                "B796CE577F18A5E334990367542F7EB0806EDE06892812F914D62BC029E6637D" +
                "60BB10F125350DFC7F7702D68984F79C9192ECC07178330141B2CD8D88974F30" +
                "DD2CDF6F82F668AA5BF9F3201F1A8ED58B2546A2683260751F8254C3AC574ED2" +
                "8FBEF421A4C8B1C2CFB4691875314C06148A801036E39827BF3AA57F9FC32DB0" +
                "00C0BB9CCEB0E828651AD82E903B710DE378B00533994AEA596AD4FFD5B075E3" +
                "4BA54099F8FEA4AFCC469D071C48A0EF8BB58B46D3666A251188209AB7FE80F2" +
                "238EC6977280497ED7833283D3C49DD9546190404E1018C1179DDABBAF18D9B9" +
                "FE18D71FA4A87976F95A533FAE01E96057CDB05FF19DD14673AABB7FB5C3B01A" +
                "44F7D8265B320E846244E7C3EE65D01F4B468084F3890D92A065D745F41275D2" +
                "9650B00BF3CB7B4DB64D14A78591147F9FEA71DE4469DE58D1E90A40B5ABB151" +
                "B4F24BCA90E1966C9588D96627FE1F69ED8BC7A52F03BEED75CA8E90EBB811B1" +
                "37BABB34129C5E7AE44E1B6FE3A3DE8EC923F05E4A471BA0D27B134E5880CC4B" +
                "20CE7404CE9EB2C114C7018E811786A7FFF5FE6A2C7FAC4C4B2FA0CA6223E9D4" +
                "0A9D6567FB659857A83703D9E995CA2E1BC96FD6EAC678204661CD866530E61B" +
                "40533A011A250B6632760FD942A8A5741410C1BF0212D66085BB623F5C53A186" +
                "6699CE7CAD843C9325D54D260D254B3273717DEBE43F4F7FEDFC984546434CEC" +
                "46D70E3B888A85A11252DC12E7D50A32CAD5D4544C161F81BBCEF0D0DE893F25" +
                "C762FDCCDE1DF91262C815C925BFC6BA133E5CD42D32E7D2A6FB0BF22AE8482C" +
                "CEAE15070F1692D5BF3C2E2CCC02D77DF879C4D4F188B80870A234714B92197C" +
                "7A27F39A7E5366D7A2E99BCBF8BC5576DD627754EC4AEE2DE118EEDF7686D109" +
                "0B3A59E97051624D56224423BC9B4A2D7A85549596D4F981E29BF4C7F6CA8F38" +
                "2E4138BF515BB72CF4CF5D44F49274843BCCED64A9CA5A514E29C2DFE75CA4B2" +
                "FABACCA238DBA202AC6A996BDE4F79C9A568450C849BF12063C0CCEFF4B81057" +
                "A386697D217F4AD02ADCA127271CC5A3E5E9E3F6722D4173B83B39CEF5DF0BF8" +
                "F9630D575F3A99BF83BDA8A5A1A8CA5AF876F14BA5360DC6CBA5D1EC2F46C3E2" +
                "F14F87A03052C3EE4994C6B661248401F40EA0843F011E5186EF09B8917B4218" +
                "DF0352289252463E2DE1FFED603F330D80D6349C36FA9CF9B721069BDD833491" +
                "52FB1BA9F994507BB22CA076AE781FCDC5B1A487C9EE8BC2C83476260F61D866" +
                "0981A7BA1F5471F56C067BC6A3C1C6F4F76C72497DDD8AB961DB8FF673B00EB8" +
                "7C498624B2C46A184B6D4F248AC9307DF046DCFC70811402AF06468E7E07D6FC" +
                "353830F4F06B5902FD823A4A94099AAD9CEEC531CC544F74BCE62777F188CAC1" +
                "B819F6F0F449372E1B3ADC45514C8598A18427D2957511938CAC5439768A97C4" +
                "8D05A62660628DC2C16D1CFE73920C00B6DBEB4D66D1572F60817FBCCC3DDEB4" +
                "C5DCBB69799E4D3BC155FCF564B6AEBF25C7AC3A0F0CE7F4F2761738A7485236" +
                "8B4D950EA6672BA4615A9A040FA5166FF520948CA1D3649D9AD2317B8380FE90" +
                "4644F2C06F6173C8FB52A572FC50D69C273DBB63EBACA717441C2530CDEDAC0B" +
                "3796FAA4F9BAEC808A01F70E5B48E42B2AA49AFB65054EEFCB0F10072A38DA5E" +
                "3174301306092A864886F70D0109153106040401000000305D06092B06010401" +
                "8237110131501E4E004D006900630072006F0073006F0066007400200053006F" +
                "0066007400770061007200650020004B00650079002000530074006F00720061" +
                "00670065002000500072006F00760069006400650072308204C706092A864886" +
                "F70D010706A08204B8308204B4020100308204AD06092A864886F70D01070130" +
                "1C060A2A864886F70D010C0106300E0408FDE1F090274C2210020207D0808204" +
                "8071C27010C2CFFD78D49D28DBB5AEF4269926F1055E9DF0471FFD7077366628" +
                "DB87AC150E12BD5C636BE9A9CCAB194C87F32749D9544873A0C2360453A5293D" +
                "E03D5DCC1E3F12FDD89CF5BC37099A08DD9700CC0B6D1E351D5451386513E774" +
                "C3FAAB375F7A6731C8923C7979F8B283D33C09CF1DFAEFA9EC014845CD779C24" +
                "A1B085C2F9A8B3AF35197ECAD885B926463D785E9C192326FC6B956695970920" +
                "8B571E383D1083AD446DFEFADA128E247BC038DC74C43E9B5832922814DCEAED" +
                "F97E989BDB105642466FE2B2E2FEC3A19F556766DBBD0BCF9C5627876B1D2A19" +
                "42B3EC2DFA957DE74E669AC767811C6CE6133ED50F97D29FA002FAA0A72980A5" +
                "CC66C5FB86B1F657BE940489087D473AE6A5475A952B24A585254FC27AD50D80" +
                "0C3A77B13AC17401453645C03FFC66984FEA1ECC385852EFD7EB008971E31652" +
                "43A3F672FFB8590F756B4481EB5FFBBF5935C8062741D7AB38A4BE28D141FF16" +
                "6773ADE4EC3C021DDB5FCC43E2D03CDA4AD6BAC0EE63CA816089B8C971A3244C" +
                "6186BBEE09C09A31BC29DC8EAC665E7D9C6CC148C3FAD9028FF2E0A9F727DE1B" +
                "08AD8561D918677FB026A8EBD9BCFB0E106C058C6EFFD53FE13996B9C40F12B3" +
                "90DDD3CC63107CCA59F1CF861692629F4AFD7D3257218D9F888DF2E7A67BD601" +
                "2EEF48B82D09F01D1D1EDCEBEE878A771399D58EEA1D50C8F8ADCA3432696F2D" +
                "49D7253F064F4EC28C335853CD4EF9E72094517EFE61EB3E0A7F50451C1F9CA6" +
                "B95FCF62F36E6D79B16D99119C9E0D4AFFB36972E68A1FA2E2A099596176EA5D" +
                "C09756632BB4FC82136BB554FB32A2997DD982BDBD058F9990403FF786342C91" +
                "F117A98F0076CAB6AAF35F8FD5A5E795F9F3368049849D582E9950B42924BBC6" +
                "5EA3A0FB8FE87FD0B74C1621B52B673E92FD6475DCCB138425740AD1C006ECE5" +
                "3633C9AA2B02404384A31F7FE7DC9F41F2CDB4584283B48E6C5D1EE659316F2C" +
                "FAB3E6D5AEE5F0F986D223E7077C417BD890035E373BAC6B90E62451B14BB701" +
                "30C263872865F8975165452074A53FD036FC3930FF2F781D2805999DA7955BB6" +
                "4D4A98D0E67E4C18F73126177D047E89B24A409DEEA7B7B8CAABD0DD39DF3818" +
                "59E973360EFEE67D8FF9E24275E9F47D5A67E812DD68F8325603F3040180D85E" +
                "2EF61BD821CA666550E55F8A8C172D93315C6D7E14567BD9DE84A0C298A6A586" +
                "A63611495C66DD7D9AF83E20CEF31F3A3D3CC25553F2045FBB547C97C1AE43E1" +
                "F0147C116C82AB442562534E0898023CFB459DB66B7320DECA41E26477D76D5A" +
                "562BAC06023B778271311E3BC8E2D5ABD90515650A994CDAA99D38F6169D480A" +
                "122AD01731819FEEB8FE3111C1CF06FB9B5882EB74232B0CDA7DCAD2988FA9F1" +
                "8F4DCDC7DDC45983128E07F6687AD6C1FB219B3295AA74592B3D8B05C7D9851D" +
                "140A71CAD10D18EBCCDEB9C9243B39772E29D923D9D5014EA4CED9FE5C7859BD" +
                "6AAE8902726D5ECBB6C0D028728F316034F991CEA8F435FFDF35B26583C1CC44" +
                "959BAE4514EB478327BA76E6F9B759E176FDD8BFEDA8BDE6E7BF5E2F9C76CAB8" +
                "055DBE8A531FA77690A620BC5463431899B7577E3AEF0D382F31955AB5FCC951" +
                "973037301F300706052B0E03021A0414F2009473DB4B8B63778E28E68613246F" +
                "D53995810414A911A22D051F1BF45E7FF974C4285F6B4D880B08").HexToByteArray();

            private static readonly byte[] TlsClientServerEkuPfx_TripleDESContentEncryption = (
                "30820B1802010330820AD406092A864886F70D010701A0820AC504820AC13082" +
                "0ABD308205EE06092A864886F70D010701A08205DF048205DB308205D7308205" +
                "D3060B2A864886F70D010C0A0102A08204EE308204EA301C060A2A864886F70D" +
                "010C0103300E0408157C22B1DC09A4B1020207D0048204C89340FF05561FC850" +
                "8D2AB6A7F4CF1B5C6D0AC638CC40E58A2518CE947393B550E4A2214B188C479B" +
                "CF508130F4C130241B926348AB97EB8F9602FCCF05351CD1FCC938A717599744" +
                "5EF72D77FAC8780ACB0B2E10E12C4D48110A0FBE516A5FA2345A5C28FFA2AA01" +
                "3D7A7AC72D3A8C27BBEDAF9F4BF8FB44F54BAA23F2D45E07CE669F1F979BEBA1" +
                "BC5475BC82E74D3BDC5374FAD9C3224C979AA10F4259755E8388C8803430AA35" +
                "87716E7C0FC3B469A88A61F9328CD5055EB6309C58DAE2749DE02A110DF13249" +
                "33BD8E7B94CB6DD31403D03B8223DD3B00E94465ACC1AC678502D8FB4599B310" +
                "D3252BD7E19EB1804D87BFE3B0C000B7DB7EF955C9B0AB96F98BC6971CEFDC6D" +
                "E9759D0291DD0C9A47DB5B23202906E1788EA1809E8B837EF67A9AFCCFF979EB" +
                "D202F85EE213CC04044303A30911043F1BA340DD3D5D44F9F2D038C37E8BA095" +
                "46BC781AF70AFAA07CFD2CE72DE02F31346DAF2ACE9C71E026C5178334F5E1B0" +
                "2C2560DCC243A83D162F145C2CD6679D2889D3E4BDF428A03B6089E7D565BB0A" +
                "1D4076AA304D39E2C293BB5C7132216B8830A97E646EEA849123B86C2234D0AF" +
                "9AD7FB623DF5D1AE53AF1FAA84120F724DD29D09C6CBDC9AD69FA8BBFB255DE0" +
                "2814A4D40EA3A2DF65DC6239B5C87EF0A13CF0A96662987B99CE5985A6703432" +
                "C3AFA98A1B1EC2E9064E166F3B5694A04C1B5D8762DBF79F4AC8CE785A826AE2" +
                "E79ED26EA800DF797BBEF20CF971730EA895456728797280EA53133000BD89A5" +
                "5ED89348E0F435703D4A19C5030919249FBC551056EC27A5CA1366B59D210C5A" +
                "AD65DC1604D9DB61ACDE4FC48859937532042872FC93A6279F2D9EBF1CBADF77" +
                "9CF1F0CA3660A3B8363D316A79CB14AD76513565314839EE46DF8A79A588AA46" +
                "2A41E1F63AD8AD5D34DE9FD2362784CD7615F35DB86575FD123686BD26A53CA3" +
                "842427778F227F633C28B50E86490E337004ED11089CB5965C4877A6C4A205FC" +
                "A0FEF43AC05E1C66F24BA51DB861617BB3C0A287B63BD03B0DB02F9D3CB213B4" +
                "D785DC7A6D9CF11C89A94DF734AFD1685CA9803987947454A8B653528196E8E9" +
                "7078294A7A9A19BA5F326EF15F5B1AB9A3180F774BD8C8DFAEA480B0187AA08E" +
                "9354719170D4DF7AB9C9F5086135B4A4DB5555FA9A283B1052F4E7762B5D2FA3" +
                "1E23E6B150B56F6551142D9EFDB857379A7795CAA25D01255AB55A656B325C24" +
                "29094605536DCAEEE1E4A6237118DDCFF39028FA3997907621A2EF2EBC007127" +
                "5539ADB45AEE3E44ACDB70B39C7613FF42E5084BD9DE9B38D8CB166D34B58E98" +
                "E6FF9A14D5F667D94EEEC2185638BA467E026538B264B6905E09B8B279163B3B" +
                "FEF0C0FD9D44EB3D100D79069FDD25A730DC380959C9043272F24EAC30AEAFA8" +
                "2BB27F78141D11FB5BC36C6CEE545B4B3E541047496BFE02C539B5160BAC5A5E" +
                "0D97FC1270501E77E7461F06D8C41F843F624C821DDE87E66D29D0D97B669F69" +
                "6868260BF2F4E4797D65F0833D72DA997B81941E766DB263C212EDE8D79DFE19" +
                "BF50C5D1A4664CB2FCE52A729B091215C69CFAE2DDA561CDDBD4C9A598B11E50" +
                "B26162FF757357DB2127DE4FD5A313A64694BB0292F482F1711515975ED49A60" +
                "78A856D7BDA9D3976DDB75FD68D5A3743ECACC9A2A715CF1B58F1B32D5A46E5D" +
                "4CC7E4743FA7E15E70362C773B621B39D9C996C8867F8CDA9333B9898395D43B" +
                "3181D1301306092A864886F70D0109153106040401000000305B06092A864886" +
                "F70D010914314E1E4C007B00340045004200420039004200450038002D004300" +
                "4200370033002D0034003500380030002D0041003800350033002D0046003800" +
                "38003900300035004500310038004500370030007D305D06092B060104018237" +
                "110131501E4E004D006900630072006F0073006F0066007400200053006F0066" +
                "007400770061007200650020004B00650079002000530074006F007200610067" +
                "0065002000500072006F00760069006400650072308204C706092A864886F70D" +
                "010706A08204B8308204B4020100308204AD06092A864886F70D010701301C06" +
                "0A2A864886F70D010C0103300E04083C4FC94CA167D9CA020207D0808204806A" +
                "4AB218D2BB0B213172A7988955F8E22DC0D8FC9B5B009A7933B911F2C0B0F483" +
                "12C8A3A6AAF9DF1B9729FBB82C6D0683BB1BCB57EFF0BD551A247F1F0C903043" +
                "6EA13F2463B85F18D94B68ADCC18D5EC979BFB2062A5ACAE4D5CEF22A86BC591" +
                "AA80BECE63688F8B3090888BB1E07F7D9545A71E26ECE75F5AF728BC5259FE89" +
                "938DCB1C1B6149AE02A99FD463608955464EAA1061E6B16C377602E960FB73C0" +
                "90B86381EE3BB2146FB439DA8CC14E2FD7B88D08DEB69250D4F274506FB634B2" +
                "01C98297DBF121168534DE416366F1ED4D96FA67520F197F2AB219DECEEA8241" +
                "C9503C8379A18176BCA4720AF05A820653CC63366C625347BAE20DDE0D07E617" +
                "3EAD3637706820946CA6190393D242FF645BC009A06B812E10C4A83259DB1931" +
                "A0D91A9B4835FFA93A5F86DF501AF3793DCABF3E1A6D8789968E449DFF77725A" +
                "DA0CA2F87F7F545E151E62C8C84CDA7197187693E2458A480F26D531C68D74F9" +
                "E9DD84B656DC62C73B9833583CDF8FED3777EAE99E06BD33BABB81214987D8AF" +
                "28F56AC8D411646A8F1DD7C4847ED63937C7B10683DA118059ECB73A560A289C" +
                "8B675E7EC35810912DA5201DF36BD8CA14428506D6181B3BC4BEA807D7C2CC1C" +
                "8459F48F6DCF9D5842C85DD374B1D9B03E6F8340B197FA5BF8B9858FE7D56391" +
                "81D486428A1474C8299959FC1CE0FDB8A3912D09BE3E4F05BA58AA2B6999F09A" +
                "3A4ABE26EDA71C6C611C139F8C8A112106036BC55F4320344F4791A59CB02A31" +
                "8DC0820625D531A0A79FD5052A492DA6E0637ADFB6F90F672CE9EB8016D0026C" +
                "AF155553FE2256B4FE791DABEA84244F154EFA22CEAA47BDD151C1EFDA50B44F" +
                "985EC26342E775D9ECBF1D6C03B29A57982A6513283ED5E39776BD96B8196B72" +
                "3A48C02E058DEB7EF6145A4153867CA24357AB670F4FF5BE3E707B4ED9A6FF9E" +
                "E24E13A874AB2985F4478AE3A5958D4A41249A021558EF56D531AAADCA69A6AE" +
                "92284D764F229AB09EF4529BF329FB91895013F808006329D2C5DFE05EFE3712" +
                "CB214F01B55834E70C5AE483522DD7D1B712E9EE93EE5647618623423DBC14D5" +
                "78C2B4BE1B038D8CEF8C70DD6201692CAC995C51B0366EFD01177ABD3D4C153C" +
                "5D04D62EDE624E978F1CC9A01BE84E5D52CF114614340454E27AEC783BCAF6EB" +
                "FAA03C571A487581408B9DBC2F480F262E4E8DE5F0645264BD04D0FEB39ED15D" +
                "7602C58E2A896791CB739BFC671E116B8CC253F3BC84A3F421F17E9C9FE77738" +
                "4916F8F8C822A2576E7CC00E0AABB9425699EBEECD5BB2D3F9A536E42D953AF8" +
                "DC4C7692E333E7D5134849D42AABA751BD3FD79F66B8C138D6EF0207BF3DB900" +
                "E44AEA2F112E61547BB7187B99346F0B851CAEC1DFC20773ED26CA67DA4B9319" +
                "5F38E277186EA536E8BA075A1A93DEDC63AAB5B21B40A83A8954905B637807BF" +
                "494A059DCC779009DD41C3EB397ECDE6057B5104B0CE675F29590FB564B426C3" +
                "9E5144392B57D56326BC42807712B1FD56FE0BD5E7BC0C5CF86E8F820E0CE6E4" +
                "C971E816E68CDA9C17178FF0FC6C3DCC11841D8F1B8C69881501C42F657EA8AA" +
                "06F9AC1BE74D48907F7825B97CDAB88C9AE8738319F063497B7EC2E626C8A430" +
                "3B301F300706052B0E03021A0414F71F2A9273499EABD45CD57C42AC2D2FD7DA" +
                "4D17041406C0432C5D5A1BEDD4771A8232801BB5933EC037020207D0").HexToByteArray();

            internal static readonly byte[] TlsClientServerEkuPfx =
                PlatformSupport.IsRC2Supported
                    ? TlsClientServerEkuPfx_RC2ContentEncryption
                    : TlsClientServerEkuPfx_TripleDESContentEncryption;

            internal static readonly byte[] s_RSAKeyTransfer4_ExplicitSkiCer = Convert.FromBase64String(
                // [SuppressMessage("Microsoft.Security", "CS002:SecretInNextLine", Justification="Suppression approved. Dummy certificate for testing.")]
                "LS0tLS1CRUdJTiBDRVJUSUZJQ0FURS0tLS0tCk1JSURhakNDQWxLZ0F3SUJBZ0lKQUppdWpocnpi" +
                "Sk9XTUEwR0NTcUdTSWIzRFFFQkN3VUFNR014Q3pBSkJnTlYKQkFZVEFsVlRNUk13RVFZRFZRUUlE" +
                "QXBYWVhOb2FXNW5kRzl1TVJBd0RnWURWUVFIREFkU1pXUnRiMjVrTVJJdwpFQVlEVlFRS0RBbE5h" +
                "V055YjNOdlpuUXhHVEFYQmdOVkJBTU1FR052Y21WbWVDMTBaWE4wTFdObGNuUXdIaGNOCk1UZ3dO" +
                "VEF6TVRjd05UVTFXaGNOTWpnd05ETXdNVGN3TlRVMVdqQmpNUXN3Q1FZRFZRUUdFd0pWVXpFVE1C" +
                "RUcKQTFVRUNBd0tWMkZ6YUdsdVozUnZiakVRTUE0R0ExVUVCd3dIVW1Wa2JXOXVaREVTTUJBR0Ex" +
                "VUVDZ3dKVFdsagpjbTl6YjJaME1Sa3dGd1lEVlFRRERCQmpiM0psWm5ndGRHVnpkQzFqWlhKME1J" +
                "SUJJakFOQmdrcWhraUc5dzBCCkFRRUZBQU9DQVE4QU1JSUJDZ0tDQVFFQTN3Y01hNS9YTm14UEc2" +
                "NWwrZmtjV1RUcExzdURXTjIzZXVsMGI3OGUKK1RNNXpHbEUrakU0SHdJdXF0TTlXVytiRG5Ody9W" +
                "Ri82ME1STk9HWW5wWG55ZzZ3cEtKQXRxcm41aUpYY1RKeAowRUNUUHZBVWhLYWdzaHJMdVFCS1JH" +
                "U2pCdjVRQUg2cWMyZUZkdEJZS1dxVnBPSTBvUm5nREdkMTJOQVRiVVJECnEwUTJabHJEeVVJWnFw" +
                "VEZzaHlhdU00Nm84NmttZkI3Ty9TN0ZwbHE5S3lrZkxXTGhpTWVGVWZ2a0tSYnJpeVIKNnVneDRp" +
                "b09NWUpsR0Zzd2x5NXRyYm9HaGdtNTFKVEdDaXM1R29GcjlvZEJiWDFiSkFDai9peXM5bmYzaEpy" +
                "NApsRmZGVUJHc0hkeXpVRUYzZVZGSFo4U1lKdHRIdnlHdmJWWUxBcXpwTE15UEhRSURBUUFCb3lF" +
                "d0h6QWRCZ05WCkhRNEVGZ1FVdEd0aGs0LzVoa3ZZU1VzNU45b1p5ZkJ2cU5Nd0RRWUpLb1pJaHZj" +
                "TkFRRUxCUUFEZ2dFQkFIYnIKYzZFRUxpYmdsc09ReU9HYVlxUFZNa3JyWThhb3FveXVQUks1dzVP" +
                "dmlHVnVOb0R0dG1vbmRVek9pV1kybFlmUQpPak9UV29hb29qM0pQV1lrOEYrT1VrL05NRVB1NEFN" +
                "Nmx1dm9oNDRCNThBTXNYS0JaUjFXdFpncTRKVmQwb01lCm5ockY2Tkkzc3puaEFkZDdZQnozNzUy" +
                "bTV5bzFqQStKc0plVTdQRjlZL2F3bDFBWlp5VlZKYVBDdFpMY21KTFAKcVlWVHNlZ2NxVFlsK254" +
                "MDdHVitja1QxaGp0bjIvcFc5VkZMaXFHWUt4RkI0WXVOTnAvRWtUeFY5QTdJeGpDRgorQ1ZyVy9H" +
                "Z0NhbC9DVTE0b2tSaG5zVUxsQWw0RzBFeWorQy8zZE90alZQMjhWWFdacEJ2WXB6SWkrUHNxK1RH" +
                "ClEveU9MN2psMGFnT1dTcngwWnM9Ci0tLS0tRU5EIENFUlRJRklDQVRFLS0tLS0K");

            private static readonly byte[] s_RSAKeyTransfer4_ExplicitSkiPfx_RC2ContentEncryption = Convert.FromBase64String(
                // [SuppressMessage("Microsoft.Security", "CS002:SecretInNextLine", Justification="Suppression approved. Dummy certificate for testing.")]
                "MIIJqQIBAzCCCW8GCSqGSIb3DQEHAaCCCWAEgglcMIIJWDCCBA8GCSqGSIb3DQEHBqCCBAAwggP8" +
                "AgEAMIID9QYJKoZIhvcNAQcBMBwGCiqGSIb3DQEMAQYwDgQI2rQEBhq4HAUCAggAgIIDyEn+zCF3" +
                "wteqkmcbOFO7Aa2HbI8d1Dbp8M2QESYZhIZr/nX69etCuh4UewErHwwxfJYVQjhKO9NhWO4wB3UP" +
                "defCCrbrCmU2CIiTsgydNw8Qfyrdp4nMUNJ8AGCK9+atNX5MmLnLoFFT182SCZiJFVspDAlI/aEr" +
                "YNF3hvtYIJPce3om82XFHPotezxpFIma1MBuhQXEjcA12tfMAjPGdA1snqjMPPUOWK1f2Zpq7gCi" +
                "Q0E9fR3O2g4ut2FDpMAPIsG1HE4TlnShJ7qMT1wj70BcXstTVNFALw9emw4guIbbeqJmPrFKuqpY" +
                "7qi+Pk5AuBRMiR/g/A5ltJuP2ZemdlF79nI+Xz+uJPXzEHk4g4wueWCaR+lCeDhxGXqWrht4gVDV" +
                "sCp156WiDdfaUD5wkWPKihXlI42S65dvltHEG/NiD3zY324rAM+A8fAp4S1aILrqXjENehtncgCQ" +
                "6OX1RDvW2HD96pRwmdz0g+qGkQzQzzEXx2qdjqIO4nNWyedw/CrfEvC7mGCqoQ3x0mbHPtJNoRMS" +
                "AyNzkk2jQ7d8EpQJhbhgqsox0kDwBLXF4DlX9IhotCZBGI3em47PV9jFD7DRbU8G0nVjqb2Jo12d" +
                "5nkyvGPF+VIOxvqLO7yyMgptpEut2x6AhRsnCP+uPdzSCckPTJBlmGiHWIhRwWbLk77OLXOmF13v" +
                "TpE98cZojKH026CfmEtFrNVaMkC74YBBbHQR97Q6aTUjb0KIMpfJ6zeWFs88/E8h9B0K0B3RNg5d" +
                "qKjQhjix+j6KWAyhnP6iBSsFHtT30t1izEMPOK3Rn3lcB4NHOp3iRixAlb+abFPiCmA7Vo5ox7Fk" +
                "ns4+n/CeVCZaMie0+F9JFE9c9wxi7wYp9YGR28cAB7jNsVgQ5NJypNI1m7S74D7ivsN1D/4H7w6s" +
                "7+2NJyChQ6d1tKriDrpaFXBtuytieRGbmoI9hEqTvVRNGKHR/g1CJk705o+Iz25Trkhb8I7HOJmS" +
                "U6TS4Nm5eZxaqcnLiwwq2bkm0O4yeo92gX8aynMAKVqp4Zdf3nunGer+HIbp7vOA3i+3CscqgIY7" +
                "fFIQip75V3sr2acpQrSttY1sZiGQ7r7064g9WWgWmOloT8JaCPLzbqstA0q2+3nNsqOJItij8ig5" +
                "e04ZJdkHNQSVQ560A3Y0+6+w0AOWJJ5U3fDzG+E06jdO8+2Y0lzlBmDEnlUx34OD5RBg/tgAjC7+" +
                "SR9flnMTMwdyhfoKWMUYhtQsKk8vORGLjaJxyfZG33Ndt78nYazRPKW8G4sYdqi0bh4GMIIFQQYJ" +
                "KoZIhvcNAQcBoIIFMgSCBS4wggUqMIIFJgYLKoZIhvcNAQwKAQKgggTuMIIE6jAcBgoqhkiG9w0B" +
                "DAEDMA4ECFwry53rQT+rAgIIAASCBMgEOQRCGM/ukQAEA4qrBattYzlC3tiHD0gqEh6SAxvPabaD" +
                "ISMLLoG0aZ0yjo+wDN7apOb9ohRIxoVyadJ9d9N4/nPIn1TnzHc7yQDTa1cktMSEszJgK8nWso+A" +
                "fQeljqqMmlTkryMiUdp+ekRu4TiDzq+zezQp20Lw5sfTVgGuMYfTUAYkB6LNzjbUw4S7/mEEcpN7" +
                "bKFKfcf3xi19yRwIO0ThLA9nF78Pq1QaORRpUuDi2zxSacHxN0wvM5HEIVEqL6+U7BgVAZF8r51x" +
                "1/+0do4O1XZpDcDQC1/yjW39dSFzWbNMOCm3aj87Tc5wurIFW1JqPmHc01k/Ks83dqSqscEBte/q" +
                "ez0XKtFGizUaxieo0ZQRHkLWE1wS7vmHIzhmYbRpSr3/WZO8u5Q5da3c1Gt5Jo6MAk3qp3gbowgp" +
                "I9vPFW4NIhq+XIx9rn8BdwKIpRwXGB5vK4aWh9GK4KBMk3mFVxWYa0ICf962R9VMzSvSLSSIJyuW" +
                "3UpbzXxQUjOVs99oKyfuDLyLQPeOVgBAVl5orSE2hcKAN2EZpHOF45TZsY9Z+HJaSwpaUDEPcvT/" +
                "9JD4uLhAcH7nQ5HX/Usp9nO8N5zC+ozBT1Hp8o0+g5pb8FBxh5NF0Guj3TlQ6VDXFX41lArdJy04" +
                "rZfWkGU9QE4VbMrBve/vyAiqdMIJ4nUCG0ghMr7VjFLPD4Gz7+9EZS2tft3tpu4VihM/TxSGgsWj" +
                "Fiyy5YZRU72KLZ+ztZvdLqYOc52/Xwxcgzdb2owMkpr3LAJONm8yXI+PFkOAVF+vkBdf4zNIzmja" +
                "ScsvnIhK7TwvlfAHEJBu8Q8pkO6sp7190WA6wOWhb7RovDMFMCJFaH7AVG8X7iu0fDxN/EqXwKH/" +
                "XTz/+FRGDIRmmS/VtavQEWkUsfSKKxfKT+glZ/sHyo5T6Cml3HaUkZ/aW8sQ7klIW1/uDepYE4ma" +
                "tA8znoDPT7fS5RM9cFfvn3Uh3KqfLB8SCBAiFkDQslf/vsVvFHxxIClA4++0oWwJ2rBibW3RdQPJ" +
                "acgd8BHuTy65TMZgbkthVBFa4p3Ne2c7Zbxhazw0Pt5Fqw/+kOp+o9chMxJsu75VMjzvVpMgRott" +
                "vn3c0jPj8olsiPn6mEWoYhdQQupTc3qmmhSHRILHc5FBY9UKoVRdg3fWeB8vf3OnQRqK1kvmc/yz" +
                "3+CUX9Kl998lYlp/YrFpqIv9A1LkyF8l43hXU56yuQTgaoYiMVTBEbdQ6/Luwg+ZAhLfl41G8MCS" +
                "Yn46mQgko9BYr9uIFpR78yzWbLnjm40vQ9mP2BE+5LxCg+QE0Frw7tpLHYrs1zIAhFw9nkZTDeTl" +
                "6VvOSmQegbw5uP3VV/SkkX2fO8DzU9M5nDDmK/CCklaGy2nzFAKrHIhsHhML4FWjiLxA0PGh6aBE" +
                "/2p+n9W45rLNqCOTejsbE5LVz/lUcDXbaCSKU28AdOUSSmv6f2hYV7idMmCORvpgKZmtjrjwVfGp" +
                "Rph7S0574FrS744psLILb+aSGaiNBPOFiBRRcUhPyVNL+Y4cOj/NMpgW6EsUpWpTJAr/1dnjFMbz" +
                "lz1b4B8KZkeQl3smXJaJz/7oU1jJ96QRYnGRyx6NYXOvu+v3b56g3Lo8MvBF59duDNcxJTAjBgkq" +
                "hkiG9w0BCRUxFgQUsdOM787p6dBPlEljE4Lazjja1NswMTAhMAkGBSsOAwIaBQAEFHcLzfQtoDAa" +
                "sbfisMs0Ll8CgjH2BAhMwg4eZANcOgICCAA=");

            private static readonly byte[] s_RSAKeyTransfer4_ExplicitSkiPfx_TripleDESContentEncryption = Convert.FromBase64String(
                // [SuppressMessage("Microsoft.Security", "CS002:SecretInNextLine", Justification="Suppression approved. Dummy certificate for testing.")]
                "MIIJuwIBAzCCCXcGCSqGSIb3DQEHAaCCCWgEgglkMIIJYDCCBBcGCSqGSIb3DQEHBqCCBAgwggQE" +
                "AgEAMIID/QYJKoZIhvcNAQcBMCQGCiqGSIb3DQEMAQMwFgQQzMqKF6TLC64Jd+DF7mgQmQICB9CA" +
                "ggPIoCkOExBhWQ0rnLj6ZlwP9pkKhnC07NHHXBuRKQTLdZgQlchqjz6wMebmc+ZlwVQbaPvJmXMF" +
                "QR46wbjJdbxVeIX34ng9tUyAo0mEUaL7PEl76ajS2lrfsGBSv3HmLL7+uGmY29/lxMNcTWLt8PbD" +
                "S/Com9sQB2MWSwsz7NsT4FlYs8P/n89Kf5yqG6RRqcdB3GlqaFZuqNi0Ycb3kMM0urOlLAEN5U1K" +
                "TCEy+YCg5+1fibgkTvLa4aJTjDsgX84bg2IuKne745RLb1kPAeLiQ3BscVyj3BRcMgujm31VqESs" +
                "OeKdCwNk8bN5DE8iSx30XlhxrFANMMtl+3Ya45+W7zmJS1BvN0DSjDGP8M9ibq9/b7dTtX88xvAn" +
                "OpK6Ecpq01rV5QLIuXnk5knQV0mWXEc97l0dz3gTso9xqUs7H422Eg9OOIuJQqJRpsbHB8sCWjyP" +
                "1Td6fLsnhmEn/Ri3+w/QhHF1oAKaNZk39vzruJVYxZL39Ny8dVmrKx/Myn9qErLQqgcx+v+DZVEP" +
                "ODFytAgJ0pfB9SaMSLqi4I6pFl7yevKt5kFif6JmiI7cvgrjaz+LCkdDY4y9WZLcQNmzKgl8+eJg" +
                "BQitWgZCnk73Jwlj2KoZFATdTWr7RYx/3F3Ra8UI/eUAE0TxJXAVUgJ2VLYsEKKtlgLkKJulQ2jr" +
                "49q3+q/oQDNOLLMc7hjr+B2kknmAnvmtayywK8w08Mw/0o5dPBrPG46aeLqw6McNArZASzUGyUSP" +
                "8cZDDGMBilBYyGRFaw9vxANDo0y9mHJSx8Ip4Ow+VU7tng6qrlJjt8LEVWM947UGygHc2EbvwWa/" +
                "gmX4OtVGuwqlANyR8Rm3cufUNGclcxt8IKRLkTRq+DQ3KTCRqZSsRfxq/pl3QDICWb2f7v6iP29I" +
                "gcCDf6ujQ7sQn5gP9KdkkAA0ZJPH7PSioJRUc4zsq3EKJJYA+YfMTJVZSdrQMBgNFhHJyEfiyxbg" +
                "o8Ge3gjcUx3ZnAN0saIC4/LxPfdu1OCrHGFkJDiTQeqROqbeYF5YtKaCwnlL3dn89eUicHVvl/W8" +
                "WRfJS60F6KFWrAOmKjx620V5/vbMyYxA+LgAIcwdmrF9k0OtnpJqcZ9qB4ie1FhtW6UsIsPQVnN4" +
                "g5JjxMLoGdMwTY0LH9YjEDdyVzjcGOMX6SVv1KyeGYuiBEfNjgik4PUAZuVqt04KO3Yp6huj9ljn" +
                "WSdJVVA4NBB4FiG/PjmmVw3MMfw2lx/M/rlI1BN+hPAW207Vuesx806cqxwdV4RI0eEe5Z5vjJx7" +
                "KiAwggVBBgkqhkiG9w0BBwGgggUyBIIFLjCCBSowggUmBgsqhkiG9w0BDAoBAqCCBO4wggTqMBwG" +
                "CiqGSIb3DQEMAQMwDgQIXCvLnetBP6sCAggABIIEyAQ5BEIYz+6RAAQDiqsFq21jOULe2IcPSCoS" +
                "HpIDG89ptoMhIwsugbRpnTKOj7AM3tqk5v2iFEjGhXJp0n1303j+c8ifVOfMdzvJANNrVyS0xISz" +
                "MmArydayj4B9B6WOqoyaVOSvIyJR2n56RG7hOIPOr7N7NCnbQvDmx9NWAa4xh9NQBiQHos3ONtTD" +
                "hLv+YQRyk3tsoUp9x/fGLX3JHAg7ROEsD2cXvw+rVBo5FGlS4OLbPFJpwfE3TC8zkcQhUSovr5Ts" +
                "GBUBkXyvnXHX/7R2jg7VdmkNwNALX/KNbf11IXNZs0w4KbdqPztNznC6sgVbUmo+YdzTWT8qzzd2" +
                "pKqxwQG17+p7PRcq0UaLNRrGJ6jRlBEeQtYTXBLu+YcjOGZhtGlKvf9Zk7y7lDl1rdzUa3kmjowC" +
                "TeqneBujCCkj288Vbg0iGr5cjH2ufwF3AoilHBcYHm8rhpaH0YrgoEyTeYVXFZhrQgJ/3rZH1UzN" +
                "K9ItJIgnK5bdSlvNfFBSM5Wz32grJ+4MvItA945WAEBWXmitITaFwoA3YRmkc4XjlNmxj1n4clpL" +
                "ClpQMQ9y9P/0kPi4uEBwfudDkdf9Syn2c7w3nML6jMFPUenyjT6DmlvwUHGHk0XQa6PdOVDpUNcV" +
                "fjWUCt0nLTitl9aQZT1AThVsysG97+/ICKp0wgnidQIbSCEyvtWMUs8PgbPv70RlLa1+3e2m7hWK" +
                "Ez9PFIaCxaMWLLLlhlFTvYotn7O1m90upg5znb9fDFyDN1vajAySmvcsAk42bzJcj48WQ4BUX6+Q" +
                "F1/jM0jOaNpJyy+ciErtPC+V8AcQkG7xDymQ7qynvX3RYDrA5aFvtGi8MwUwIkVofsBUbxfuK7R8" +
                "PE38SpfAof9dPP/4VEYMhGaZL9W1q9ARaRSx9IorF8pP6CVn+wfKjlPoKaXcdpSRn9pbyxDuSUhb" +
                "X+4N6lgTiZq0DzOegM9Pt9LlEz1wV++fdSHcqp8sHxIIECIWQNCyV/++xW8UfHEgKUDj77ShbAna" +
                "sGJtbdF1A8lpyB3wEe5PLrlMxmBuS2FUEVrinc17ZztlvGFrPDQ+3kWrD/6Q6n6j1yEzEmy7vlUy" +
                "PO9WkyBGi22+fdzSM+PyiWyI+fqYRahiF1BC6lNzeqaaFIdEgsdzkUFj1QqhVF2Dd9Z4Hy9/c6dB" +
                "GorWS+Zz/LPf4JRf0qX33yViWn9isWmoi/0DUuTIXyXjeFdTnrK5BOBqhiIxVMERt1Dr8u7CD5kC" +
                "Et+XjUbwwJJifjqZCCSj0Fiv24gWlHvzLNZsueObjS9D2Y/YET7kvEKD5ATQWvDu2ksdiuzXMgCE" +
                "XD2eRlMN5OXpW85KZB6BvDm4/dVX9KSRfZ87wPNT0zmcMOYr8IKSVobLafMUAqsciGweEwvgVaOI" +
                "vEDQ8aHpoET/an6f1bjmss2oI5N6OxsTktXP+VRwNdtoJIpTbwB05RJKa/p/aFhXuJ0yYI5G+mAp" +
                "ma2OuPBV8alGmHtLTnvgWtLvjimwsgtv5pIZqI0E84WIFFFxSE/JU0v5jhw6P80ymBboSxSlalMk" +
                "Cv/V2eMUxvOXPVvgHwpmR5CXeyZclonP/uhTWMn3pBFicZHLHo1hc6+76/dvnqDcujwy8EXn124M" +
                "1zElMCMGCSqGSIb3DQEJFTEWBBSx04zvzunp0E+USWMTgtrOONrU2zA7MB8wBwYFKw4DAhoEFDV/" +
                "r3z/utYI1K4Y+CvDRILGEGbbBBQMvVmizsCIdljG6t477/4bInm7XQICB9A=");

            internal static readonly byte[] s_RSAKeyTransfer4_ExplicitSkiPfx =
                PlatformSupport.IsRC2Supported
                    ? s_RSAKeyTransfer4_ExplicitSkiPfx_RC2ContentEncryption
                    : s_RSAKeyTransfer4_ExplicitSkiPfx_TripleDESContentEncryption;

            internal static readonly byte[] s_RSAKeyTransfer5_ExplicitSkiOfRSAKeyTransfer4Cer = Convert.FromBase64String(
                // [SuppressMessage("Microsoft.Security", "CS002:SecretInNextLine", Justification="Suppression approved. Dummy certificate for testing.")]
                "LS0tLS1CRUdJTiBDRVJUSUZJQ0FURS0tLS0tCk1JSURhakNDQWxLZ0F3SUJBZ0lKQUs0cjJQai96" +
                "ZnF2TUEwR0NTcUdTSWIzRFFFQkN3VUFNR014Q3pBSkJnTlYKQkFZVEFsVlRNUk13RVFZRFZRUUlE" +
                "QXBYWVhOb2FXNW5kRzl1TVJBd0RnWURWUVFIREFkU1pXUnRiMjVrTVJJdwpFQVlEVlFRS0RBbE5h" +
                "V055YjNOdlpuUXhHVEFYQmdOVkJBTU1FR052Y21WbWVDMTBaWE4wTFdObGNuUXdIaGNOCk1UZ3dO" +
                "VEF6TVRjd05UVTJXaGNOTWpnd05ETXdNVGN3TlRVMldqQmpNUXN3Q1FZRFZRUUdFd0pWVXpFVE1C" +
                "RUcKQTFVRUNBd0tWMkZ6YUdsdVozUnZiakVRTUE0R0ExVUVCd3dIVW1Wa2JXOXVaREVTTUJBR0Ex" +
                "VUVDZ3dKVFdsagpjbTl6YjJaME1Sa3dGd1lEVlFRRERCQmpiM0psWm5ndGRHVnpkQzFqWlhKME1J" +
                "SUJJakFOQmdrcWhraUc5dzBCCkFRRUZBQU9DQVE4QU1JSUJDZ0tDQVFFQTRJVUVaTmpMU1pja1Fi" +
                "WWJIUk1rek9lOW1lNzBWQVFHUjF0QWRLeTgKbHB2bkFxWkw1VGxPWHd5di94MktXbG9kVXlsOGpZ" +
                "dEhRc3lpb3BQRHF3QVBVUk5tdEZtQnVsTi95QW5Bc3JTZApqdWVHMWJCUjFZYVZzSFhpeDR0Mldy" +
                "SjFJcWtUbzIrTjIzb3ppa0w3ZWJzM0RtdDE4NlNLckJGVkdyODAvU3hYCndlMThiUWJQZ0RHYStS" +
                "MUczalFZakpPVGRFSk1rQzkzMUY0ZEFBYXZVcjgxWEJKNTg2T2dUZnY3SHhTMi9TMVAKS2cyNk4x" +
                "M1dWZEZRMmxieDVYYkZRZVpoVlovZ09uL2xDckVwM2NFcGdlSDdVSUt4ZmFrRGFQNHF5WDNhMXhx" +
                "YQpKU3hsa3liTWFwaWpiWVBiSmRzd3RKWkhIQTJxblhhTktwVDI4NldLRG5lWnlRSURBUUFCb3lF" +
                "d0h6QWRCZ05WCkhRNEVGZ1FVdEd0aGs0LzVoa3ZZU1VzNU45b1p5ZkJ2cU5Nd0RRWUpLb1pJaHZj" +
                "TkFRRUxCUUFEZ2dFQkFObFUKbW1EU0FrSk9rZkxtNlpZbFY3VTBtUEY1OXlJQjZReGJrSjlHUVVB" +
                "VjZrSnMwSG1lcUJRSncrQWRHRFBYaEV0dwo4ZDlGR0crZUtGeUdjWEY3Q1V4TU45ZGNjaThHYzFQ" +
                "WnA1UEZmbk04UVhZVTdLSFhRQWVlNm9yWW5FelBPZzMwCnljTklLbkF1L09Ram1ZcjRrci9zRnpk" +
                "Q2I1MDZRWkRxMnF0ai8wZThDYUY4T2RXdHNTWGIvcGtQM3liRzZqNHUKZzF2L0JtU1VYaUdWWFZx" +
                "M0JDUGZoeEowWGhyMnBBOS84b0VENjFBMG1aaGVmOUp5RTFhZFd5Q3RTaWMrelplYwpBcStDTnFs" +
                "R2o2cUpuRWhTaDBXVkdwMlRmMFhRaXBTTDk2REVCNlF2NlF4T3hpb0xNNG9zY2tKUEdhTGl5cmNj" +
                "CkZDUFYveUJyVm5Md0ZHR3NheFk9Ci0tLS0tRU5EIENFUlRJRklDQVRFLS0tLS0K");

            private static readonly byte[] s_RSAKeyTransfer5_ExplicitSkiOfRSAKeyTransfer4Pfx_RC2ContentEncryption = Convert.FromBase64String(
                // [SuppressMessage("Microsoft.Security", "CS002:SecretInNextLine", Justification="Suppression approved. Dummy certificate for testing.")]
                "MIIJqQIBAzCCCW8GCSqGSIb3DQEHAaCCCWAEgglcMIIJWDCCBA8GCSqGSIb3DQEHBqCCBAAwggP8" +
                "AgEAMIID9QYJKoZIhvcNAQcBMBwGCiqGSIb3DQEMAQYwDgQITLms5nr095wCAggAgIIDyPyxib5c" +
                "oCEPC3OpfmCzzieARKf56sQZU31qlDJNDet7v1R68T7X9vu236kHOphKvqedPVNXKEPpaQXIxkSY" +
                "Hus2Xg7Zv7jwW99/A/coitXNEyOnqdv3vYRu2f69iwZJQmiPsQKC8uVbDBgPBZPPe+tgkY5e7dqZ" +
                "mwob4oJ3bjKR4cYPk8QkSOxgBXHLNZ0Cv3fjiAVKHnPGSo5pyvwwUtSp0bBXniyq9wnzYheiLBSg" +
                "S99M1HVvnbbFuhZYeyTP6gqRSda9bgVa0NNGhFYKjnw0H2K2KRtIskL/P4vcxZWTSdLNxZKQDfgx" +
                "R0Om+f8qFiDzaW5jo3ljJ9pZKrHWo0tDAhSBicOvFsNTX3zu9ldRJLyHO7SZObvHr3BCa6LoXRNA" +
                "BEhYmihmKwCv0dXfUgEZ6BSwSeHdNEWSVQhLLHI5oL2eRjHOCa8q3Md9GDn1hdnwUQftmvn9w2Dg" +
                "EhJxJe+DjhLquh+fB+JXFnrNg5s0birl1RPL7wsdadGBav9UhpRcIgxux4eaOkkGE8mySOxBY66s" +
                "BV6upSxJf+KAQXBQOqMYoC5wK6hXzllAWLDjpEHuU2q40bTWoUZk2lw8rSzxW/mhLoVl21Ix3x6h" +
                "Hm8VUaGkOsQbpMhI68B47yNgnkUyX5FciFtxnV7cxAqxjtaq5yO7gjzUSs7uyV3pYhdL95f1tlrD" +
                "goBz/OKSFZYnII8UnTQniFdqZc6rhKnJpTvuVJy6I3Os9DrJpZ34MagAkb6lwIy/BqV4hPDx2AaQ" +
                "j3P/Qk4xPq0EcbUSuDZHAe4nLygpnMtDynM5UDcUjLLEtGIX9MpUtAPUtZEH/ZRtWF0wIkvFCUDc" +
                "Gh6v8bZtRYvCd4Vbnf99ueRKjyA2PREcriO+gsdEUd+rmkBwCMPPRmIjX8tr9yuqM9kRlmvzYbBB" +
                "3RcUsfyl9I39S1kJQkrr5mOH2hVMTl6UQ70jM+RBet4FJN3TUWaoBNm5m2+9FLpL8GcFPH//WPnw" +
                "S2AqXl+iTysAUxqjtZ8jWkyl6DmybKHxPbR6Vy/6w/2WOUQOOsnFADm9NJPV8TE1Z6qnhlh4f7fz" +
                "zAue+vnkZE7920JN/ThEkreIJ1vvyfPQsfo5XSoK7SA7OlP0JX4bKEyWsahFzmFOaGNJUCKTR/gh" +
                "JVB6CcgrvyAE4BFsZeq79teOCKVZiseKwtmarTCMGQOdo+h5ceWXnII0gjndwcZbzURh3sj86AS+" +
                "GXk69sNKC6HvbineMu/oL8ZgsifD/OTTRC5ewXENyoF57vMCkFV19K95PNTN6icGEcYeMIIFQQYJ" +
                "KoZIhvcNAQcBoIIFMgSCBS4wggUqMIIFJgYLKoZIhvcNAQwKAQKgggTuMIIE6jAcBgoqhkiG9w0B" +
                "DAEDMA4ECITzIA3DVAowAgIIAASCBMhUF7mWIkOeh/4zd9n71iNtHgyZIm90cF31YRHS3syGcsAI" +
                "B4LH0wnHthG1u5GaEhTjMzBTr96hFkOFwmCW+8Kbg25oIcy12aren2Rn2LPkoHmmV/aA5ot7bbsK" +
                "2vTgbae4GbMwXb0oBMKLU3wzKcYyluhVGVket2+szUHUsMKZxhFUThkvZqnNC1BYKDidiCP0npEb" +
                "bgX4ARBEB++bL01YFvQbX/72JSPKTtvTs3aoYTlfoklwUrxSw60/wYn+3y8KVyBIkdZKDVKS0AbF" +
                "uNEQZYmyf4AiSv7q4h6WKfwI9ojl7wv9k0/h/a+x+8rAITHdX/ZaSNcmoWa/zCKWNO+YntvrRxLZ" +
                "xGE4t3S/4h32bbo+dDD5iLucuUGVMFt6b9CBph8+QYIbte3zjQPjPWY1YexRI4LqiK5/3ZFCRPJK" +
                "Eq4bYmhur34RmQ9vZy7NfANgW+lJVit1vfOMHjQj62svkbabMjVrWVhnpAJ7jhBoNRzbggY/Ehhq" +
                "G5BXY6fEjwPWFivHM0gmmBX5VlCTmhJck2sCJ4rKSpc9ZB4J6ucPH3wHEyO1HIDgB8yI47v4CZeB" +
                "XyfIrozivPn7OCmItD4tUdH7w4/BJxNOMZ+Gpoz5tGWoeMcPWzb3JoUBJgpZ7YnqFbG8I1lTlG6P" +
                "EbhNuyXtB83R65CAASteHww7PMfZDmUQO/KJwNvmZFPNox4IocTKLf7cOMXt6NR5hPQpc/tecDdu" +
                "QzQG1nEITq0MP8Ex3YQgu37gfpYNiRAki7yKrwzYpwVrUSwap0RPIn4+UxwoINXngwZTmNkhXEr+" +
                "sfW8uiB8P5ZrhqF+9dCGBHKAAy35NMEGXXHK79YOrnP2qNf8IbfnopwnZNYSniKFF7XvPmjiVANa" +
                "ARQ8wWIUbhsweZIz9dz0GH7mdk877o0k1xrqpXagAeaF7wwAz/HLFbIG4+XXD5fFjkqaFpuTYjOt" +
                "FgneN9wXDwgvQWyqPbjx/kJBD5k+UAW1r1z+LDwD/VdPMJ5bdfWa5QFCIxWw137VsieGXNZvB57Y" +
                "hdI2D3LKk+0DhGJjGTxy6g/ZTwvGUlPPe20+Ip2+zSoKhE2YVBAbsy36+cr77buOPg8L5aNqUg5l" +
                "rIYe5eLADLg8Unavy54ySIy2pZ0gX2G2eGQebwAIQEqDYBIRUyiEhbjGKprTTRQj6Ho3M/mmHOKM" +
                "fHZeWT+BGOAa0+eCz436QS7YsXl2AOwFBqDI6ZONYTBRu2WLvuK3nbwb+h1Rv6/pd7NGI6AXJBKJ" +
                "1+iBLGl5HBoOIYmpG+yGgajXGXDVqkRGESUnrJQGlXbwq00MFvtKbGI9+iQBgw4pbrUzc5xId0OZ" +
                "1ynSTYrKzaGfi1KqMj2dQbQ+b0SshD/BVrBtYYVEnY9LaPokHna0cYcUGXxnycesk2Av0kQziEmH" +
                "vltVZDj1LyMBJzF/auHY+Pbx4NIN33fOdKXByQSE4ulZqgsftn77B25yxwjO2Sks+ess7+pGBbO1" +
                "ikOqm6zol92lwRibj8bbbva86Sd3yd846Yxc4QRcSldTPMOQsBrfs8ZetusRCiONXb+gKDne/Z/N" +
                "u94mkxMkJvLRJHPesvNkxiXYhC0mSFl9RDSkIgrRJ2uRSxv0V5OdcAiCi3d98OlrIYkxJTAjBgkq" +
                "hkiG9w0BCRUxFgQU++MZ+0nVwxEM/9SdcPKsQcAk4ycwMTAhMAkGBSsOAwIaBQAEFFNlHKZQXjbm" +
                "f0PE6idyhcH963CyBAiMLpuBSuzCdgICCAA=");

            private static readonly byte[] s_RSAKeyTransfer5_ExplicitSkiOfRSAKeyTransfer4Pfx_TripleDESContentEncryption = Convert.FromBase64String(
                // [SuppressMessage("Microsoft.Security", "CS002:SecretInNextLine", Justification="Suppression approved. Dummy certificate for testing.")]
                "MIIJuwIBAzCCCXcGCSqGSIb3DQEHAaCCCWgEgglkMIIJYDCCBBcGCSqGSIb3DQEHBqCCBAgwggQE" +
                "AgEAMIID/QYJKoZIhvcNAQcBMCQGCiqGSIb3DQEMAQMwFgQQcikkI3mNYPzp9KgATYd+sgICB9CA" +
                "ggPI8jMlpeNA98igwqedfn4BlSDksVxqex+KPgGDn/UYx/S5yfQMUxq2vocr3mXGBGm5B7/4v7mV" +
                "X0CbLu32ngd+lWovtp/n+DyNdsLiVpAasjY07F1ozdtJQ9ANBgkz5PH5h/s2tzvcUDteIyclZ6X9" +
                "UWuwY9/LnnwQKfZAnvdMuYe9Cvygt2hEvp9xJSnotH5b74JcKGDm+VXhyDL9Ls2hvyjg8/AonfeX" +
                "t8MimMDX4PL4DruRbm8bFTU1SP0io9OLenZ8P9QJ1oEzIk5n30taxfMonQcImaBHLnugE9HeREfT" +
                "pPUlAjkLQR7muqnsabCMq9hDYNwdts+lF9cgKWO78VBDOgcfwJ+X3HdtVp+dfah9MDftC5PyuQE2" +
                "VY85CoPqMzsvk9afgxOdhIR9n/LmPZM+cc1uyXhdU9+mDUP7jJFnO+36uFn8cUDy1C6d1TqHZbFo" +
                "Lh/R2+Jf/txawMfGvanN8K6z4DPv5y6ARMBzsiIAsdfIQVWS397yJT5kKPcfDQb7icbpFdta/efV" +
                "sIbwz/2Yvku9+ipccGgqPjhAvBj1m549L/UM0+Oumm3iltDO4GYdWDHWR72rFEDN3wurrFgPADZk" +
                "3qrwH0T/GIUGYZeF1Kp7XEEgXcB2MCQiwwkVWy3HBVXL5RQFhFfPGSixXu1JPeugLGEsIlm3Jv/C" +
                "7D6v4aPfa2SpEAebYhSvfN3t/0J1cFzUeN4XVt0g44t2XnFDGbJVEG+/7odsArL/jYsp6jmR02Hv" +
                "0vhzA+4qyDEtZPSOWtiYRIE6g6rKfsrGZ8lVf/M+0jgXOSW4spoZEBtHoIeycQSvvPN3kLH9alcu" +
                "fJYf1PtQGFKR7EdGsJ88Ho5e8lzOn3Ka7X/M+vegIxdnmPclv6yD3FFCRS2r2DD31a1dJfWVO67x" +
                "ydXuhWEE20/rbEmblVHpoG9pSXrn7A4NV9Wyetwd/UE8XvGocuLFwXyLikHYs+KDTa7OW5BFebox" +
                "72QuNhQteU7ay+1i/rfhNvvqq+B0MALQAamAVVz/ynsdsRuszikcALK9Zofq0ap51ZpVcqKXI82V" +
                "9Gg0LdmH8AicB+RWj18MeEEqVKJAzbU7QJGmG4kNgZxSPbfAWek6axEB1SoKFlq+C+HcPl2O/r8n" +
                "WEeNd4lDp1v9cFbJ0XEnoAf/0yd7jDBr9DtMNyamW4wBvoCWgmAeF7ojuztitsy9Cu2Lh8r/idR4" +
                "dA9gZCI6kXw0pQ86ud2X1p41goxHcHa5ic0oHj3jZAYh1X6PPUo1IGg541jdnhQCk2hVmkIwCh+g" +
                "X78wggVBBgkqhkiG9w0BBwGgggUyBIIFLjCCBSowggUmBgsqhkiG9w0BDAoBAqCCBO4wggTqMBwG" +
                "CiqGSIb3DQEMAQMwDgQIhPMgDcNUCjACAggABIIEyFQXuZYiQ56H/jN32fvWI20eDJkib3RwXfVh" +
                "EdLezIZywAgHgsfTCce2EbW7kZoSFOMzMFOv3qEWQ4XCYJb7wpuDbmghzLXZqt6fZGfYs+SgeaZX" +
                "9oDmi3ttuwra9OBtp7gZszBdvSgEwotTfDMpxjKW6FUZWR63b6zNQdSwwpnGEVROGS9mqc0LUFgo" +
                "OJ2II/SekRtuBfgBEEQH75svTVgW9Btf/vYlI8pO29OzdqhhOV+iSXBSvFLDrT/Bif7fLwpXIEiR" +
                "1koNUpLQBsW40RBlibJ/gCJK/uriHpYp/Aj2iOXvC/2TT+H9r7H7ysAhMd1f9lpI1yahZr/MIpY0" +
                "75ie2+tHEtnEYTi3dL/iHfZtuj50MPmIu5y5QZUwW3pv0IGmHz5Bghu17fONA+M9ZjVh7FEjguqI" +
                "rn/dkUJE8koSrhtiaG6vfhGZD29nLs18A2Bb6UlWK3W984weNCPray+RtpsyNWtZWGekAnuOEGg1" +
                "HNuCBj8SGGobkFdjp8SPA9YWK8czSCaYFflWUJOaElyTawInispKlz1kHgnq5w8ffAcTI7UcgOAH" +
                "zIjju/gJl4FfJ8iujOK8+fs4KYi0Pi1R0fvDj8EnE04xn4amjPm0Zah4xw9bNvcmhQEmClntieoV" +
                "sbwjWVOUbo8RuE27Je0HzdHrkIABK14fDDs8x9kOZRA78onA2+ZkU82jHgihxMot/tw4xe3o1HmE" +
                "9Clz+15wN25DNAbWcQhOrQw/wTHdhCC7fuB+lg2JECSLvIqvDNinBWtRLBqnRE8ifj5THCgg1eeD" +
                "BlOY2SFcSv6x9by6IHw/lmuGoX710IYEcoADLfk0wQZdccrv1g6uc/ao1/wht+einCdk1hKeIoUX" +
                "te8+aOJUA1oBFDzBYhRuGzB5kjP13PQYfuZ2TzvujSTXGuqldqAB5oXvDADP8csVsgbj5dcPl8WO" +
                "SpoWm5NiM60WCd433BcPCC9BbKo9uPH+QkEPmT5QBbWvXP4sPAP9V08wnlt19ZrlAUIjFbDXftWy" +
                "J4Zc1m8HntiF0jYPcsqT7QOEYmMZPHLqD9lPC8ZSU897bT4inb7NKgqETZhUEBuzLfr5yvvtu44+" +
                "Dwvlo2pSDmWshh7l4sAMuDxSdq/LnjJIjLalnSBfYbZ4ZB5vAAhASoNgEhFTKISFuMYqmtNNFCPo" +
                "ejcz+aYc4ox8dl5ZP4EY4BrT54LPjfpBLtixeXYA7AUGoMjpk41hMFG7ZYu+4redvBv6HVG/r+l3" +
                "s0YjoBckEonX6IEsaXkcGg4hiakb7IaBqNcZcNWqREYRJSeslAaVdvCrTQwW+0psYj36JAGDDilu" +
                "tTNznEh3Q5nXKdJNisrNoZ+LUqoyPZ1BtD5vRKyEP8FWsG1hhUSdj0to+iQedrRxhxQZfGfJx6yT" +
                "YC/SRDOISYe+W1VkOPUvIwEnMX9q4dj49vHg0g3fd850pcHJBITi6VmqCx+2fvsHbnLHCM7ZKSz5" +
                "6yzv6kYFs7WKQ6qbrOiX3aXBGJuPxttu9rzpJ3fJ3zjpjFzhBFxKV1M8w5CwGt+zxl626xEKI41d" +
                "v6AoOd79n8273iaTEyQm8tEkc96y82TGJdiELSZIWX1ENKQiCtEna5FLG/RXk51wCIKLd33w6Wsh" +
                "iTElMCMGCSqGSIb3DQEJFTEWBBT74xn7SdXDEQz/1J1w8qxBwCTjJzA7MB8wBwYFKw4DAhoEFCyj" +
                "rTpupcG3BPfLBuT5GlHPVA7IBBRe4QUbn/DUvN/IRNbZin9j7WlOtgICB9A=");

            internal static readonly byte[] s_RSAKeyTransfer5_ExplicitSkiOfRSAKeyTransfer4Pfx =
                PlatformSupport.IsRC2Supported
                    ? s_RSAKeyTransfer5_ExplicitSkiOfRSAKeyTransfer4Pfx_RC2ContentEncryption
                    : s_RSAKeyTransfer5_ExplicitSkiOfRSAKeyTransfer4Pfx_TripleDESContentEncryption;

            internal static readonly byte[] NegativeSerialNumberCert = (
                "308202C2308201AAA0030201020210FD319CB1514B06AF49E00522277E43C830" +
                "0D06092A864886F70D01010B05003014311230100603550403130953656C6620" +
                "54657374301E170D3138303531343231303434385A170D313830363133323130" +
                "3434395A3014311230100603550403130953656C66205465737430820122300D" +
                "06092A864886F70D01010105000382010F003082010A0282010100E2D9B5C874" +
                "F206A73C1FC3741C4C3669953929305CF057FF980DE2BAAEBA5CF76D34DE5BF8" +
                "ED865B087BF935E31816B8B0DEDB61ABEF78D5CBC5C3389DBD815ECF7BDDC042" +
                "BD21F68D9A7315671686613260B5D3505BC7C241C37C6E42C581228965B5D052" +
                "B8ED3756966C6B678CDF647E5B659E5059915B16EF192BD313F6423FD487BBB4" +
                "06BA70A7EDA7DBC55E15F1D93017BC92238D22DD9A15176A21F1BF7CF7AD280F" +
                "7C32D76C8B136F0FBD0A89A373CF194CB5A0CF7AC1FA8098458290FD66311BB8" +
                "DB17E9CB70D6668A8926F26198586C796C61EEDBFD376525130BAD2C452B6A66" +
                "97AF7FE8F91408785C20DE79F4AD702197734328B4642FB898FF730203010001" +
                "A310300E300C0603551D130101FF04023000300D06092A864886F70D01010B05" +
                "00038201010057E4CE63680B165CD3DDAF5CC84F31CBC6C2CCDCBE33893A3FA7" +
                "01781BBB9F003C0E301216F37D4E9C735F138920121983728EA88324DBEC66C7" +
                "824BB25EC534C3AA5999C8A6D9393F9B04B599A94B415D0C0392E0FBE875221D" +
                "A4348FE76ACBE5D845292AC475A9104052CCEF89390118E8EA8051519F423276" +
                "164D8BDD5F46A8888C540DF5DD45A021593A0174BB5331223931F7756C5906A1" +
                "94E7FF2EFB91D4CBFA4D1D762AE0769A07F2CC424692DB37131B2BEBDB3EE4BE" +
                "07F9B1B12B5D368EC9317237CAB04D5B059F9226C972BADBB23CA747A46581D4" +
                "9240E6FC0F2C8933F1844AD5F54E504DDAA3E87AE843C298DED1761035D2DFBF" +
                "61E1B8F20930").HexToByteArray();

            private static readonly byte[] NegativeSerialNumberPfx_RC2ContentEncryption = (
                "308209C40201033082098406092A864886F70D010701A0820975048209713082" +
                "096D3082060E06092A864886F70D010701A08205FF048205FB308205F7308205" +
                "F3060B2A864886F70D010C0A0102A08204FE308204FA301C060A2A864886F70D" +
                "010C0103300E0408F693A2B96616B043020207D0048204D8BEC6839AED85FA03" +
                "2EAF9DED74CE6456E4DB561D9A30B8C343AF36CEA3C7F2FA55B1B7C30DBF4209" +
                "58FEB61153386DCE8B81DCC756E06EC872E8B63FD6D8341123FDAE522E955357" +
                "76B5A09941595C79404E68729389819D05459515314167A9E2585F3B2C8F9F24" +
                "860BA42612DC492137051063A8EC8CBBC3D0ED59B92D10E6612C0C9AD5AE74DD" +
                "999007CFEDA7A1AD2A2D5C25E41201F5A810D2654A3DA1648AF5D9EBBCFDFEEB" +
                "FF8BC78A7BCE26EF0ECB1B0F07B33FB777160ACDF3DE00267398D771A35740AF" +
                "661B189CB7394C02599417AF601231F3B6F40869E2DA8909D8A6D9565CFA389C" +
                "5A002193B3CC374D973F0A587697CE6812126E3BC390093E56B671E8FA77E1C1" +
                "E56AE194AD7695BED88A1139AA149F4E0875121995661B4133E9242FCAAF8A1F" +
                "5721A055FC523EFEA4EEA98B6FB48EF07891B2E80D6EAC824FE9BBFBCB6A72A9" +
                "52F12C3E3DE63F33F9A126BAC8FB0C7F2BF98A781598E6FE5A86409F4F22249F" +
                "4A38B4670A2BF0EF93B0990AF33672CCAEFF15BB5813ECEDFDA38CA2BEEDAAA5" +
                "2B880609057B3FC728F6D1964B7185A933B1526666FBC42C1B5D9A6FF3E2B3BD" +
                "1BB8893BB6B9170BD4D7C6EB47C573FA90E19971017A6FAAD2B36CC746EBB238" +
                "2211248404A09E2ABBC546D85B212286334E164FE24E03BAE053C3C12FA21F29" +
                "FAC8A69F0EEBC688FB50834B5D7F2CB7CECC9BD5D6C06200AAC11B33AF2B6E11" +
                "1F3853AEFBC086881561860093AE6C66A488BECE95CC80ECCABCFE91BDD302EC" +
                "B86DF4B99530ECD707943E76ECA51E07DC109B9B788D1C0664C560352B251FCF" +
                "D9F57C6C18DEDFB9B848A2A57A96B0BEB6C20FEFADAAE468D96B48AB061F6BE6" +
                "194865E676009BD48E5951C534C05A4C4186DF7CF85B596E2F4FA2495440071B" +
                "C13ECE7AF0E58FA13B53DF115BBAF846A5E2AF1F71B3F8AE895C62ABD90FBEBB" +
                "815993D122B4B6816DF95C8097E9A1FC66F0181441B3BC3320BBABEE02004138" +
                "93BBFEB480123A28B14AD4D4B8C918B851DA11C4347C8D1E46B2F211F0E6F049" +
                "0674C2B1C2F6AC020D79E50A45363AD54D5CD2D6668DE917111111E380ACC015" +
                "3CF5A0E3B25A4DF35305308FA2A5C0FFFFA31F6699D5F0699DD82AA5502EA6C3" +
                "B6B782FDEDF5515DBA4D1FDB295A77119D49BC098D35A45E8DB2C88B7C97DD1A" +
                "BB54B0D2EA27AD7B7A79D3C707C482376F8F612129C48024F4A1065BFCFAC2FE" +
                "A176AAA2E0064BE2E77D9E3BCEA0AA732B59DB213A9A0EC972E376D8ACDE5BB6" +
                "4C668B0E24BFE9479BE7574B07D4BEADF821769720A0B7557411826C97206B3D" +
                "176262F4BBF64631ECBD31418487588773E39D46784A4361E31742333BE1DE2A" +
                "BB0331011CA400795E306EDA88F82C78B6906B22E5D0CF68F9F349B9725178BA" +
                "6B03D972E27C8EB376BE51206DE070FA45EE430D6CE17FD2744A7830E7A8E2C4" +
                "C5033D39BFB14D74214ADE082371C49246540B0593487B0DC95BC9B887FA2222" +
                "250D59EB2BB04983AD59004E3921ADCADB30E5C713B9B5FC31A6FD97D6694483" +
                "2E29DAD105CF81265F8149B5EB32192CD0FA45F53621AE7A4D372ECCEC013D08" +
                "B7D1BE73E5D3331236E90DE00A3AC3044F1A5462C9E84CB621A9C68C8B954824" +
                "B684ED3DFC2D2839B19AF443F54ECBF3EC580BADC3CBECFDFE05BDCBA201C984" +
                "C7DDE01EF79E57C5E8EAF9C1E4C8D48DCDA8C6A51F8C0DECABFC4BA6B776C4C8" +
                "BA0110C6D5CEEBC7476E86D4710F5E773181E1301306092A864886F70D010915" +
                "3106040401000000304F06092A864886F70D01091431421E4000380038003100" +
                "3800350063003300300061006100660038003400370036003500610037003700" +
                "32006600610036003400350035003900630062003400320064307906092B0601" +
                "040182371101316C1E6A004D006900630072006F0073006F0066007400200045" +
                "006E00680061006E006300650064002000520053004100200061006E00640020" +
                "004100450053002000430072007900700074006F006700720061007000680069" +
                "0063002000500072006F007600690064006500723082035706092A864886F70D" +
                "010706A0820348308203440201003082033D06092A864886F70D010701301C06" +
                "0A2A864886F70D010C0106300E0408EBDF03E906A5A3C1020207D08082031043" +
                "B764B7A7FA6203394F0625BB0D59B39D3F28197972CBEC961476D76BE560A312" +
                "1ED025A9455F39C33B2ABC1FA897EC29FAF01970321069F8CA20EE4DF6F9E98B" +
                "0DCB3A290DFF4B9B1D151E4AE3AFD7D7383B40B292B0F369A47823A7C175F6CE" +
                "A327283B8B33712B88F06ADDA6CD626036765E9E102E110C8E44EA75E53C5FEB" +
                "661B08D7A9C06B59A35E76F293E8EFA64AF425B877D0321C13BA0079DA14D3A9" +
                "A76EB69CF22E4D427A35C7391B650B822FA91FF0D3DFD1E037C34E2F1110C1A7" +
                "F88A35E560422AE31203ED22BA74D1A6F8FB2AB5DA6E5E66633B7081E6F5681E" +
                "DE01BD28E6DA6B6E0C1444A010DE229D6AD6E44E99400EC22EF1DD6E7E805503" +
                "4B1A29ACD78F6D561E2DA35B84DA1F08BCDD0A74AF49E509744A91F4304F7EEB" +
                "6231C18EC9159D31114183C0B9621AA227663AC67A5DB4B0982B539578DF0ED7" +
                "6B577FD370FA4330E2DFAA015E1E6E2E4C6B71967071390D4B1CD061F567D072" +
                "2BBBE027C1350B1679D1AE7DE7522AF62960403C15F1B75DC2AA0A1339051FA9" +
                "E16692773046ABE0EEC29C1E6FE0ADEA33F7F878B004BC161D362ABDCD2F9992" +
                "82D967B9FA46BCBF0D12CB6659FB2BABA258DE158F51F961CAD7AB5D68D6DFE8" +
                "5E0DC6F043C4AB203E2F549EE0A3C2E25E13CD1462CD03AA3DB4689BA1DD0D8E" +
                "D8293DA3D195901405154E98999E60BBA39A543E64963BE607BD48275508946B" +
                "D27DA136C9F53C692DDC81FD3F6AD56419589716EF6F3A66A35049B29D438D5C" +
                "F778961F74E954CBE1395D8A98971C30C3941CFCF2D3B8851C819D79D164FC71" +
                "CA414798F4FFF3A6362A59AA17F543B5B4B2F2B7E14ED48B8ABEEB68469161B0" +
                "6D2233257291C6F1B67D0BDC2422F3E04213CA29EB648219A6C3E339C1352E55" +
                "F6D8749592723934693DD54F1D098DACFE9D308556B060CCC75D2F7AA8DDEC3C" +
                "B1B127DE82AC8489FEFA4A9F09D49C919F703236036853D5E802975D4B3DA619" +
                "EF90CF53AA38D0D646B69E75751DA833C619337722CA785477343614ACB67AF4" +
                "427E3EAFAC5900F0A573512BD81F1A86A848321309156093665E39193B0867A0" +
                "5C86500AF2760F8A87DB8E6E5FA2753037301F300706052B0E03021A0414AA38" +
                "B6B4257EBA3E399CA9EDD87BDC1F82FBF7D70414B796F49E612D1A2B5A0DEC11" +
                "608A153B5BCD4FE9").HexToByteArray();

            private static readonly byte[] NegativeSerialNumberPfx_TripleDESContentEncryption = (
                "308209C80201033082098406092A864886F70D010701A0820975048209713082" +
                "096D3082060E06092A864886F70D010701A08205FF048205FB308205F7308205" +
                "F3060B2A864886F70D010C0A0102A08204FE308204FA301C060A2A864886F70D" +
                "010C0103300E0408EADBD31BF1DB9D12020207D0048204D860781235FA36052B" +
                "1142E6E63AB86FAAB05C8522758F20D4104714E6C756173419C61A7A05CF57B2" +
                "8B716B01FA2509DD8F57226706BF8678D11AC4C76369CACF2A54A9E5A53BC658" +
                "675F326CC565CC7127BC50DA682DE3F1C57921FEA921F797370C07B194936A5E" +
                "23F0583B4E911DBED2D0ACE32190CB0513AEFF385B046D46EDE7BDDD1377A807" +
                "1B1D1AFA5400DF1E3E4DC2177617DB23D81F15B51CA1697625D8208FBA1F1548" +
                "B3CEA06D62A7F947997923A790608D6C11F4EA294BF5C394276C4988A1B95C51" +
                "BDE8317CCB2EF77FC20EDB1993260F7D581E89E03E0EA93569CF39BFFB531FF0" +
                "D51C5773661B5F8FB37E34FA847D67A1D8B3A17F4F67D123C81CA84C01974E18" +
                "07B6BA9AAFB816565E20F91E4D2D70D31EA8C523423AE4621ECFE6026882B6CE" +
                "78D858D518B1C52D49AFE4442E2FCD81E961153F5ED85E4B17ADB7C25AB71101" +
                "42B97C09297002058A0180E88AEDFF37910A2DCB275380E428B6CDE0707DE988" +
                "37732DE2C7817AEDCC872C0E0E8862DC553DEE8224A196840B1AAF82EFBBD988" +
                "4E37DA2BCC8EA859B2671947F6863034360F0F58E736BD2B5995C34C44E098DC" +
                "9B340E08F2A1CF40DDC0F6407212630120203DB17F27EF11042BEBAFECAEC5BA" +
                "5176468C5BEFEEC1233F8AD9F691F0D09BBD9C9D7E48E036FE14BFBA460885C2" +
                "3C417CBB0C58B962EEB960AE9FCFDD8BA51CF2056B26C506B0A2B46E159D19E0" +
                "EF9678B696A25DFCE06F6CB71263B7D7CEA23997A32EE148BB03B2BB5563FB76" +
                "AF0E10E428281EAA83AE16F9915C9D4D835277556193A3C1A3E3737A7B596F19" +
                "4AE3C2A5CE05A36F0531F8A173217DCEC91E328AE643ED721ADD3585DF2AB832" +
                "6D27066F701C329C6F3D43AAD1BE51E7DCA157EBDCF082C889D26CB91A54EC45" +
                "C84D67F291BD7B16DF7C590CD1FE2172A93C8693954F15EF913DE032CA677E30" +
                "03A00606CA194B1C02CCEA873730FA1EACE02CDAC838D9C6EFA2467EBD4FFF64" +
                "E1404F636068151DC4C840055816EC1C91BB6B7F5166AC2758BFE7B02A8402E3" +
                "6857DDEF537CBBDDEDD7B0461602147BC453FA5637D826273C9AA47DA96E94C7" +
                "A0ECFB5316A551DCD7FCEE25288CB5EC26F919915A1B724774ADFBEC3A9B44F9" +
                "5C838366BE34A8FDE892CF8286B90B45005E0980114A19D351DDD0EF5D45005F" +
                "0570C019E94622A79E0122B72B15EC6F011C0C5E4CCB12CF99164D5B4ACEF3D2" +
                "8C4F33773FD3C71A3F7CED5013887A4FB3804D487ED9FE8969D80F3705FBFF61" +
                "1D9BC5CF0101AC321609EBD735A8BE23F1CFA01087212714AA7A035D7BA1028F" +
                "30EE685E66E943105218FBF9F7ACD5969851D48405B8533754AD8CA5836F7B18" +
                "7EAD45C7CE0603798FFBF6F58DDE2392BDDE986AB050D00953684CB7FCAC2151" +
                "EF293100E894C5AC5CF44C2795C41BBDFB9C9252BE289B9ECC6822B97883E765" +
                "D958EA9AD84ACB9601930B4FB9CCE8F69C771D5D02C4CF9FC5EC9394B01C6CED" +
                "60500F317E4D5AEBCE5744EC8DDD5A03BB28072023CA7F7310BF04167692B03D" +
                "763169E142644DC2CEB369DA9C03411DE4C4E4FAD2532DE81EE3E7151DAA94F1" +
                "BCD89071F94B0DA91139064912FD697AB520F0AF8F8C938B50EF1CDD9A54C974" +
                "E6665851C42485E82A5E98E1E278309F45F3AA198D22DBC16321DE04E26E7109" +
                "1148E687B5D8869AB1EF942083DC2AFF863C251B0936FB73E4849D1ECB553497" +
                "7D334F3648F9C2A21A40C355FC11D7673181E1301306092A864886F70D010915" +
                "3106040401000000304F06092A864886F70D01091431421E4000380038003100" +
                "3800350063003300300061006100660038003400370036003500610037003700" +
                "32006600610036003400350035003900630062003400320064307906092B0601" +
                "040182371101316C1E6A004D006900630072006F0073006F0066007400200045" +
                "006E00680061006E006300650064002000520053004100200061006E00640020" +
                "004100450053002000430072007900700074006F006700720061007000680069" +
                "0063002000500072006F007600690064006500723082035706092A864886F70D" +
                "010706A0820348308203440201003082033D06092A864886F70D010701301C06" +
                "0A2A864886F70D010C0103300E0408F2C9235120339743020207D08082031087" +
                "A84399911830599C956CB78AC54D2AD6F3D3D90A180C9F71079DD90287703FF2" +
                "82E74E6FFE0F4C4335E2A3547EC8B4AFE5B618C5DB15548879B7CC4A2E56D514" +
                "FA7231F6EDB429385C6C8E44EDD9303089236371C2ED982D5F7EE32CA63B3A6D" +
                "F1D43DD9BFF4133AE4DFEDE56FC96CA3603A861EA52B9B6D21EAFA961ED97E53" +
                "5ED290A462912AAA621185F9E1339305AD7D2039120FD213A4063FDFF4DBFE14" +
                "FBF580ECA3A6DFE956B96A737962A2018F3525A472A90E133F022E5FCEEE25AA" +
                "CE9635182659049471BA03C5573E2F68E3F4484796669A9BE81398C232C0247D" +
                "AA8615505FD394104F11E7749227239628F666B2C717CC1E4FEDDB9F91F218B3" +
                "6726086B66C27CFE0395BFD150C5B868AF3D472EFF6FA8A0980EB7494AD7B059" +
                "6253F65C94585D2F86FD3E70A0E49C330419D2BDE70000FEB7DFFBFCBF33E1DF" +
                "12E08B3A8A6D82D9BC0AE0FC20CBACAB9C62D6C1FDE9505BA27CE77DB7C94475" +
                "29EB5C3FAF3F2E806D88A1A46BF230DBF7B8E4CEF6C12DEAC6ED805F4028AB5C" +
                "7CA99C23CF4E2419D8E6EEDD05BA6FDF061C3251E8E909E2DB35893AE96F3118" +
                "61AE4260EADABEDB22F545A85CBCF234324C95FB171AAD72DE503F2EA1409ECC" +
                "BA245A2C78DA485373E85F643C64F9E3DC4D20D971B7A960A6C76B1819BFC02A" +
                "7FFCA7768D4DC36EA7FCF19C80F3B082D9BF30545C17885875C571D050EBF2D6" +
                "952595043945EADE050CEA8C99433514D3B03D1F0082750BFC0965DB05444C11" +
                "311EE037DBA4579CB0D9411B54613DF8FFF8407110F0E8BEB5DBB4D4E286455F" +
                "EA748C5BCFD72B7BBFC1B960DF08C9601D5C49B5BF747B029DF87FD3255BA9A4" +
                "0224F2B97E8770BA23F1C8A04FD099E63AA05A172D883FA4CD6298479046B4B4" +
                "DB0A4CC8C72ED0BDAF6E846F795705C6D330E4A50363003CD23C55655AC332B1" +
                "30F6BF625896957877DC5C4B8D6D8357764DFF6FDDAE19D2B002062FC54A75C1" +
                "45E0C85678210CAA04B4E6A8B5C88F596DD769EF484927096415BAD1355DF038" +
                "DC4C150F67E6B2A3FDADCBD6DB1081ED29868CDDCB4EECC83B2ADBABB14AD09D" +
                "E8D0E90ECE88FCD440959F706B4028303B301F300706052B0E03021A0414E431" +
                "D7F51D316EB967B5AC6CE0632A882B4F61F0041404CA58CF8EBA5473BFB0AFEF" +
                "96D11C7E403BB3F3020207D0").HexToByteArray();

            internal static readonly byte[] NegativeSerialNumberPfx =
                PlatformSupport.IsRC2Supported
                    ? NegativeSerialNumberPfx_RC2ContentEncryption
                    : NegativeSerialNumberPfx_TripleDESContentEncryption;

            internal static byte[] RsaOaep2048_NullParametersCert = (
                "3082032830820210A003020102021000B2DB376E004B56F8089A1034C37D7330" +
                "0D06092A864886F70D01010B050030173115301306035504030C0C436F6E746F" +
                "736F20526F6F74301E170D3139303432313231323134335A170D323930343232" +
                "3231323134335A30173115301306035504030C0C436F6E746F736F204C656166" +
                "30820122300D06092A864886F70D01010705000382010F003082010A02820101" +
                "0091A009FE3025D0366735DB93BFCD714979B4982B54374E60C4220A2C0F7E74" +
                "A9BF62EDA8D87BE09956A73F4C527FEF7A53C2DA2BFBFAF28FD49C786E09963A" +
                "948A10D65DB8AEC829EF88DD5FB7386211AC8828124CBC24582450FA11337B4A" +
                "72FE8FFC9C95C30F4C7B0019FF591D0C92F88B7A21D25C8CE47D52E55E4823F9" +
                "CB06B7F62E10586E2DBAE84D48654F1D7A8EB5D160F6B076A82A3806864B09A7" +
                "58E9121AE8EF50D76CFFCFF81FE4193E5D0BD3E0CC4E58FDBC3CC973A2907C88" +
                "EA3E7BD2689384341ABDCDA2525C275EAFA4884F548F4FF20A869E9F598F48C1" +
                "571D42DE9E8B4C4CDD6B47865F9BA0A7BB342B29CDCE00D6711B79C2C8D50405" +
                "330203010001A370306E300C0603551D130101FF04023000300E0603551D0F01" +
                "01FF040403020430304E0603551D23044730458014A791F70FC3C8B363A49AEE" +
                "505055EAD33659711CA11BA41930173115301306035504030C0C436F6E746F73" +
                "6F20526F6F7482100099356AD75D355A305839B87F087B5F300D06092A864886" +
                "F70D01010B050003820101006F0FA161D224E2E27F35CEB37B92A8090568C9AE" +
                "F29F659A1DF3484770369751359849721A544759230AC79E8487D38CFDEAA31A" +
                "492E286B10E24B6B2FACF0E6F103B1D4CD6FD2B8784F536C1397B6E791499DCC" +
                "5B82C006458793815B2FA016A9C2DB1A690A8005803B60FD1876C11D290F77ED" +
                "5E41F63E907F21E645CECD96F063EB3E22BCDA5D80EEEE8BC23379BFD6B37184" +
                "82ADFBB84C6E28F86A8367F28A1777E61A1E0EA50E50993957E9715D95707A22" +
                "5C7B4071D980FDC13E0D72DEDC6205884B14E523941CC0F1ADFC64109E85ED67" +
                "83039EF14C45C5CE54151D679E8D38FB90F5F64F96C8FE23CBBA362C1A17A922" +
                "32DB27C43535DD8ABB9C054B").HexToByteArray();

            internal static byte[] RsaOaep2048_NullParametersPfx = (
                "308209BE0201033082097806092A864886F70D010701A0820969048209653082" +
                "09613082056606092A864886F70D010701A0820557048205533082054F308205" +
                "4B060B2A864886F70D010C0A0102A08204FA308204F63028060A2A864886F70D" +
                "010C0103301A0414CDCA906D503CB0FD77952B3E473FCDA78409868302020400" +
                "048204C8580404D87213470B672926374C32E46601DF1B0CF28A28C89E91B402" +
                "BBA6F12557D46C36D7C50447651BB14AB488671879C278BF4ED7B3FD1C2C387C" +
                "B5491192C552F2617683DB4B8D752CDCC29FD16E711B3D715BAFA893A606F767" +
                "8D464A12AD34D74EAFA7872CFD90413407723C3A2344B37BBC801BA5A2FC50F7" +
                "F23ECEC60FFF517B942EFB269F7291A568E6CE294A89231D309639228FA3559D" +
                "D72A9D51A33A6B112422F63E55C65E8C5B7AC69D04FDB785D9DAF53FAFEAAD02" +
                "CCE7A10690EFEED1CB61335B663E8AB05D43148EFADD170205CFD9C0BAE14703" +
                "40B2B10BDD0F6607ADF176026183DD62D2E0E52E77AD6945B616F979956ABE3A" +
                "E1C193A8BC7AD57DF90F02D52B88761FA85656039035B15D326C7C400D0C1BCA" +
                "CE265C8B1B3F48DE1297852055C37C89718383F7AFFEA171795570A2A13EDBB1" +
                "22B257A47752A67AC0AA922C69C50939A225B6725FE826A2ADFDF6672196547A" +
                "71C25A1FFAE1BCFBC06A20D25F1A0A1C35678E1DD9ECAF3BD1DA1F8B803CB3BC" +
                "FE39EA2BF26D4B74F44B6F8BCC43B39B1D51F8B935BFA55DFCFA504B5EA8792C" +
                "35D30E40F908C4B9D200C65F89492559265DF46B7659C2F3934EC1F4590F6294" +
                "319E03E88105CDE763347B77C30FE66F538DDB7C353563238224F523EC0D7469" +
                "1DB8C4D4FD5B99CAF8576A5508A0D537E33864749D790864D78886C47362EEF0" +
                "22E5E32B1A8DDA19949317196D641D5B08473A234817E10B2BED5DD9E680B30D" +
                "E5E545A1B93819200435D73CE39E083E9F1D19B55E8B0A0BC9144A5072908127" +
                "DD9031F5FAF0111FA7D1511ECB5809E49E29DA27954C78CFBE384D2944CDEEB8" +
                "3AF2B4C861AD92005ACD0A6374544DC8C89C28BC3FD7A9287BDA231902D8C6DC" +
                "3417423BFC82075BB29009C7CEEC69043AB209BE934C6A08A11362E79083C25A" +
                "716267098C9B8C054FCD55B8310D559E612273E84DABA24CAFB3FCBFC2068591" +
                "0762FFF325599FF882AB2C013BBFED97D21520DBCBEC0EBC03170591A4A98229" +
                "0FE8ACD561CE5DECD3C4DFB8C32BB6AB1B8799A8D6515569F37977CE60603D43" +
                "87A5D2BF4D62FF0C971B9EE6AE735FC4C047376DEA0EE16A12CDE810A5AF738E" +
                "4B454B2CE484D608A07FA18CE4868B6047F1BD377F7A8F16A983E19D9693D677" +
                "7F734F7417221B7D2ABC0E74CF9C039FBE16DD34106F88AF25ACCE8F407A5179" +
                "EC3BF209E35F5F315F0F0CE064627EFD830DF50CDCF8D21403D533924711B442" +
                "3E6DFB65B5AA000EF6DB8D90F57BCC592FE9C6F1A30A7B7FD4D651F4B3E37F70" +
                "C6CE0267370B5960C74EE23EE476AF8290114E41A2535FF3DE6A629EA2C87A71" +
                "7B3C164E6103222F9010ED4426A2B766DFF4271C1C6A2CF704F1DB1B8A27215D" +
                "91115FA4E873A00487DF7EC18E07013D47F59A5014AEE87FD1F94BBFFE72D8E6" +
                "CD0BFE40ACD32DC3F8AF7E8E288ADC9ABC75E5FEE7E56C676D25D94F1AFBADCA" +
                "7ADB09ED186751211D5A07CE4735508489178B1958FAE7E96A5AF25ED52FDEFC" +
                "D72E351096336D4C65836BA62693D6DB3111EB61679EC9AF0D837C1B1C1E08FD" +
                "E4413EF1CCA6B61E029195E477DCA8F92DC503A3F90A917730A0B4134F5543A0" +
                "877E7A125E7B2B87BF21CE84A811E2D8063C3506DF6ED4BD3E5AD0EB1119876E" +
                "C51172FC664C4D2771501D9B977B0579059932DC6F4B6811593CA1B063316150" +
                "A4385EF3EE87C2D4A64A2BD2313E301706092A864886F70D010914310A1E0800" +
                "74006500730074302306092A864886F70D01091531160414989A932D9FF76F8C" +
                "FCAA6B7420276D27B96236F9308203F306092A864886F70D010706A08203E430" +
                "8203E0020100308203D906092A864886F70D0107013028060A2A864886F70D01" +
                "0C0106301A0414692101FB773A3B21A23929C1ABEFC05DA7FB2C970202040080" +
                "8203A021B1E0409E6D0C4D02435D73CEF56BA485245E75AFF29214F9889B3AFB" +
                "80529C789BB4F0F366A850F89B35175554DF88A0CD453FB4FCA331ECC5069B11" +
                "03B670B745EC8F612896FE1451857232E5444FBA5A52ED963984ED79906D815D" +
                "B3F59FB19DAE35D8EC29E71016581AD3BEEF9992A76B776E221C73C47930D7E8" +
                "FD378FF254EF23CA3B34B6E75F1F7FDF58688129E4973A18BCDBBD9C5469D6D5" +
                "7B08E5377C2C416BC55642EB8C27545BF641500916EAE7A37B7D3346B9EEDF32" +
                "3DEB35D07D1E28420950BA40FCD45F2CFB7BADFDFF30D04454AE32A7583A79BE" +
                "002B34DFEA8259FC93BBFC67F6FDB428F4E73BB5228FDCFA1B5E4BAB67AF73C8" +
                "B53683CCE9A3A255CF66AE2DA79BE727F378E74FA9CFBBD6F30AC1C33D25501C" +
                "DFC61E5FD79BC13AD0091FDCEB00BA11A26DA4E8FC08DED37C8E4176C7D9A4D2" +
                "A6C91F847AEE5EB07EF37FB45411CC21CA230957F4D9BCDE96CFE8C5E476B5E5" +
                "7CC87ECAE0197C77C7977DA5D5E5002D5F51DBA06359B9F5E99448D700ED65FE" +
                "1618B347E2D1C9B4425F664DF796DF08110BB72E55D4CA34A9C19D4A00DC7161" +
                "8A5C047DDF25AA1BEDB4649ED421F51316B4EAEA1D6B264E86743BB92BFA172C" +
                "EF5CE50715D1226505F780D3BB9ED84E30719B88B776A44278360B052A6BCD98" +
                "17F0129FAFD37EB2566802B54533D345974E55D3C6EA59A224EE38203B32595E" +
                "5FFF6BBD41DADC4220D1202C8C71633A026901F08F2FFF54D4661EB7E4E658BB" +
                "8F059DA7F300405E17007694BA1939E473610802F427848B0C7A9C5AAAAD1F24" +
                "0CEFB2BDCF0F4F8200B1EAE67430C47FC460B57E65D973FD9852B20B9F7C8009" +
                "B9B99A52513264081BE6AC7C4ECEA3E8F702D32C82DB7221F5FD8116971D2473" +
                "811BB2C13ED6349F907BD1819B04DF7DB213FB0AEF0EEEAC47D9DD74FB387448" +
                "519E03F43803CE1A0CE168C225B8EC2FF5A4F7B7F2E985458DF078C1152C8D8E" +
                "0835A762A307AC352A3B6D37CF84B934F857DDC70306D0953CB7797356F374E2" +
                "6DD3B96C32BCA6DB98DFC8413B87171955B6758F382C2EC43DE012D9F02E5335" +
                "1F8E063C1E0B1896334672D621AF38E153EBCAB7D5CCF61087916CD6107416E9" +
                "11069B57359839F7D64FEC244CC8AB35F65C3F84DB80D1657016F69111DA7B63" +
                "052205786981FA3E3047FA4BB4876F315235F96FB83F02282ADD202168123C73" +
                "A2A8698209BEF3D3FFCE19EACD67708D97194BC77D94784CE85C33ED1BCB82D5" +
                "4256318773B4E6A58D1518CACEB03CA74999BF7A898A6BB543413803F77D925A" +
                "6D9A23303D3021300906052B0E03021A05000414A0CE79F0E969E2A8F6959B58" +
                "6B99CD091DE4E4FA04143FFA346F6E7F8ABDE22A51853C06F365CE21D7D70202" +
                "0400").HexToByteArray();

            internal static byte[] RsaOaep2048_Sha1ParametersCert = (
                "3082032830820210A003020102021000CD1B955EB1EFB9F3BAE8C64D49C16F30" +
                "0D06092A864886F70D01010B050030173115301306035504030C0C436F6E746F" +
                "736F20526F6F74301E170D3139303432313232313734315A170D323930343232" +
                "3232313734315A30173115301306035504030C0C436F6E746F736F204C656166" +
                "30820122300D06092A864886F70D01010730000382010F003082010A02820101" +
                "00DD16801AF1ABCA05C53DA0863C7719C9719DFB363784F5ADBC0367BB98E873" +
                "FD46B06CCE3640BC3BAB450756E8CA7FB336D5EF1FA15BE76C60F06E71F3843B" +
                "9155D6DCEA8A3C9D361068FE4AB95064AABB997731E47FDD00AED791A377774D" +
                "36A204C851E521325F2BA01616CB5684C8CA628D7A42C8CD02CF423FF9C6EB80" +
                "74F5D61E8226A39DAC0E4B051701FB7301C4F85DD4D2DE0FD1D5CC143B18FDE8" +
                "007ADC853C08003B86F5D4B97E87124906CF4D3C746CDBDE3B3B8431FB1601D6" +
                "4EC6E9EC99C6A5288023F8C15803DD8E85A6621C0714C60FA8062D5E6208FC5B" +
                "F1C32B7DE521C0AB1A7E60E444A1230E6BAF605BA35FC1111CDEF6B6E188DF84" +
                "8D0203010001A370306E300C0603551D130101FF04023000300E0603551D0F01" +
                "01FF040403020430304E0603551D2304473045801419AA399804F91ED61132EC" +
                "CBAAF578543F46972DA11BA41930173115301306035504030C0C436F6E746F73" +
                "6F20526F6F74821000BEBBAC206CB54DF85376C2D4E0F19F300D06092A864886" +
                "F70D01010B050003820101000B8F34A7BBD7E2DEAD1E1025B505DB439A7CC9B2" +
                "6D9BD4EC8E0D5A5B3222E0926F5B2F97B888AD2AB8632CE2EEF9878FEED75A23" +
                "019B4FFC6583B95FE8E033A75D2293E6A37084E811275E9819311A9E9A5AC9AA" +
                "F27031E814CDC7A7DEB99B42D8D24334BD00C4B8066EEDEDA9A19F5AD4720A87" +
                "E0EC4299A8F527929ADA9AE72A6D28E5E8EBBC7AE565D645065E0D64FD57F87D" +
                "8A3D542EBDF21F3464FC3B3AB33A74A46DCC2658E73690B7BB800B20E30AB038" +
                "6CF44C4211FA82A49F2E52BD168590955159B2D82A37421E0A72A11E6AB2117D" +
                "F9AB0E0165F0E4C5D4075FE9E21C8B6E783C3EAC67027B4F9FBCA3855FA361C3" +
                "C322E14CE938B2FE3A5AF38A").HexToByteArray();

            internal static byte[] RsaOaep2048_Sha1ParametersPfx = (
                "308209BE0201033082097806092A864886F70D010701A0820969048209653082" +
                "09613082056606092A864886F70D010701A0820557048205533082054F308205" +
                "4B060B2A864886F70D010C0A0102A08204FA308204F63028060A2A864886F70D" +
                "010C0103301A0414F4D52466643FE5A7B6C45AD1800F828B2FED17DF02020400" +
                "048204C8C5779FCEF0DE7E5D03ACCF6BCE8126CEE1F35DE299F7CE2ECF36F54E" +
                "A95DDFE98D164F8A5BC7A9F6CF39C8B90FCD40323D0C885E6AA5C54D810CD706" +
                "A42909972CAA471BF8A6AE1E51D646D95E29507CC018ACFB060B36268BFABC8B" +
                "82D01A31B1D492AC016D06BE487A5557793DE1DD1197C9D723C1AF3E9AE2DBC0" +
                "640FA18D3998356C2FD6D2A06DFD259E34083543AFEC7F72B85CCE6BE565C430" +
                "6EE7B19A039D1F22BF4F3AA3EA272C21AC4DEE6155041CCC0EC2DB68CAED1609" +
                "3F328689478A7A25B36A47A6655691F10163220ADA2F436943A68A6AB24D76A7" +
                "D12D400EFAE7C6D85FDAF8434CC8C524CB4B3759893536957F06BDB03711DB60" +
                "315EB84799C29C00186D4640278640815928679266CD6D767E83EF10B2CCB277" +
                "4D1A90D8F64A23CBB3A823A6A7D0BC1BD8B5D2E56E91F3F117D5D88EB88489CB" +
                "5CF4522D47DF401D1C8525A89A79D04E7A6A17CC04F4F2D7FF355F91FB0EEE57" +
                "C8B8B8CEF5AC41704A303013FE95DDBE70646D9A34A77A4CA283018213E2BDD0" +
                "92F3F45A8070E1472A3A1D6D8AD46190620CBD84FC5183F3A03921D9C815C1FA" +
                "D1140B6BB1008301B60E58239BDE8AAD68340AC3BAF2FDC00240DB5965C6BCC0" +
                "5F58C2B63F36A25656F645425B019D9EC57893245C48C0709827B208FA705716" +
                "3CA6DA5441C46B1A76187B7B6D2A286EEA26035ADED73B3B2B3229EC53F04608" +
                "60FDA1413CF2B5D16DB55806D90F09CE9D4FC41CB9FDE51869E2650270B042B4" +
                "063572217AFF48B3D6B5EFF29575DE8178B2D3397DB909A1F6ED2BC4623D4AAC" +
                "F2E56B20DCC6064BD5673F9AFB317372BAC63B9B12CC18E3A8C3EAAF63B2A96D" +
                "D04EF4A6580F98C353570ED1F268C4CAB268F6409F972AFE37D42FEC5896AD95" +
                "F576F745BA5092D7D0591A894B3C2B922AB76A8BA5523DB6EBB7CA3C63A39436" +
                "31CFB8958DC8C327296C2892FBFFD40A66F34195C5D9C8CECEB0596BD09406F5" +
                "CE417BAFDF31D0438409D12A8A657B093BF48FBEFE532E6A54EFBA99439CB96C" +
                "971BA7CD42CEFA3F5CE46A1290AB16DD451CB5228D6C6B611F1DC3A3856D2689" +
                "495949D68921C7AAF01E96F5BDFD365C1D09776EFA1367C8C1613B3EE01FE02B" +
                "338600062391E21341C2B2EF0E918FBB30E382DE3011374E852C8AC37EC74220" +
                "C1172AFDF61229F02D7774D5FDB33E5B5B44A961CDCF386D3DBB73946E20ADF4" +
                "D3F52F21A433AB71217E6AD17CFE85224D4E6326F6BC6FA0DE4B76B433C24767" +
                "7D7DCB754C2C5F30F6C50817A7AA7AC0520DAD79BDC5273ACFFF2E08A38349E6" +
                "6BE4027344D0BB590C8F9080CBD0D9C0C0DFF360CB53E74ED5922C75CB87703C" +
                "3601A7FE69BEFDE8618CC23D05855A7ADB58F18C68A970E6B9B9D16C0147B2E4" +
                "279460B856DEDAB4855C529CFE3BC76D2AEC253911DA6A2CB3BC965FB9B43A85" +
                "B3EA673724C310698F16E24F9DEAA547095C70736AA559447DE34EE8A80FF9AA" +
                "EF79929BF7BDB474A41E7F4D7B509FD195CB8C6B260D92D76AC50B3650CCBA7D" +
                "FF11C04C1DC8F5CB1E99CD651234882269212BE7957732C3EF74731D2D349B60" +
                "8AB1DB1DF628C1D50842E481ABD7BD15A95196416ECA8870A2AE9EE6B1724933" +
                "44DD0B843AE53362CDF4C23BAD8E96BF205B783CC0B810B56E7AFC50F1F91B19" +
                "C6776F01D452B29A87326D262F1742902072B4143647E28D821AFC27BDDF4F36" +
                "CF6A6857159C43582396C7A2313E301706092A864886F70D010914310A1E0800" +
                "74006500730074302306092A864886F70D0109153116041481AC4AB421E0EE3E" +
                "783F04B1FD5314632A7CA500308203F306092A864886F70D010706A08203E430" +
                "8203E0020100308203D906092A864886F70D0107013028060A2A864886F70D01" +
                "0C0106301A04140003F5A90C3AFBADC5D4B269C00550E8291F952E0202040080" +
                "8203A085E9E37AC65A1539B96463E46790E461AC4FE12D09E72B584C47332342" +
                "1446095C926D54A8154CC5237E41F3DB93D5AC96600A3FFA115BBA90942D35D3" +
                "4C540FA5FE7C06BE6EBF432AE0674D0BBF51F5AF54A00FE6B0779F6D24133632" +
                "EFCB75AE44022F8DAFA05D82BE8704E4A2F331CF4DED131C9DE7C2BFF7871539" +
                "724AC6425922C9D8C4DD45ECB1957FE3A97A0CCF8AC38FF6B8969C310ADA5AE9" +
                "DB43DF7556AD724702D82B334738543B8B85723F1DE520EFCF3F15BA72BDFBD4" +
                "CE9A3B858D812317341605346DD17569FED134D7FD68F628CA4C888883FF8BE1" +
                "9AA25E24AD46AC4FB9672B5FFDA7FDA340C611E4D7E8761EAC8A3D9246E4DC1E" +
                "D1A92263E8ECFC9A9BB676CEF3E413D159F475EEBD7707889E71E6F799740B3C" +
                "E56A1EE57752C7A979DBD90EFFE1E2AD2B17685BCD9AF73F59532225C648E2EB" +
                "7BEA8CB81CE5705F922657F8E5EF4F32C8F4426F2E5C9664EAF7AFF9C7FA0975" +
                "E1A3AA5509E0931AB471B7B9C64A08E9A103F47C4C4430D7D9EE848094B6268A" +
                "9C7CD811763CB0344B6AA1CB3C3A9964628BA348A93C75C2755F7E11315609A2" +
                "27BF54010948206501C92D3551B106DBC15EB596E1141A0CA48B7D655A13D0D0" +
                "E35473701AC38936102905C4723CC84F01C74DE854DB3AE352E0514F4AFF54CD" +
                "BF4146A03ADB72909AB999CD21733B9C5065FA5165E27A6ABD89AF8754D9C010" +
                "0C76A2976CDF958FBB0B83C15E2CA8C80397B74E43ED388C80FF4A05BA13928A" +
                "61077A56BC6064098D4EDE315A6E96B2C30D3B48102896BD1B9E1D0774F1CAE8" +
                "9AE21A0A0162CA05336E82351DA659F3E8231467C40B28774D639848FC435B12" +
                "4D5BEABC70765F021E9B04823E33D0A62C00CC3206DBDC5A65E3A6404241E715" +
                "2B3C942592A0B1655E9C1BFBF87E5643CC227258B64F3BE8266A91F2D4DCFB48" +
                "036E6AF7B81069EB66AEEB50C8BFD9129C51B518E15B4BE8BBA60989DF77852D" +
                "C3B394BC2A17BCE4A9C935DD1E51F8F672B5DFE18EB1A414791651B4875C1F45" +
                "7BE8EE0D3D1DBA0869513879E8D9338F993D4E57267DE23B7015D744572BDBDD" +
                "135E07793F3E24E04638772645DCE8325D7DDB9803E2CF7BBCA9792F0EEEE001" +
                "5D9621D316D83CD90E7F5451F2A8F9264A5F6FF6AE85325985F31854421D82B4" +
                "60F4420B9F3F08608CDC79F15F1D0096A8580225AE9BB81EFB85E56C80A08AA7" +
                "D70C42B1CF6DADE96D635574847F84AB9DD15FE1E393DEAF368FACB2F03D45CC" +
                "52C4F6A3BE155B7D32D46738200062BB36B53D7CE3FAF2DE04A83B42544AE212" +
                "728658303D3021300906052B0E03021A050004144200D15CFF93500F241D12EA" +
                "D690D3DB9CDE40A90414E9656D02B74AD5F7C1C9ECBEECDAA9F91B3B3E3A0202" +
                "0400").HexToByteArray();

            internal static byte[] RsaOaep2048_Sha256ParametersCert = (
                "308203573082023FA003020102021000E632D0CD2B7885DEC18B874E9DEB2930" +
                "0D06092A864886F70D01010B050030173115301306035504030C0C436F6E746F" +
                "736F20526F6F74301E170D3139303432313232313734325A170D323930343232" +
                "3232313734325A30173115301306035504030C0C436F6E746F736F204C656166" +
                "30820151303C06092A864886F70D010107302FA00F300D060960864801650304" +
                "02010500A11C301A06092A864886F70D010108300D0609608648016503040201" +
                "05000382010F003082010A02820101009773FC00EF371FCEBE2145563A29A08C" +
                "608EC3F24FB1976594876E3D168BD744BC665C061B75F0B0ADFBFE90967ECFAE" +
                "522A30A38457656F0D620C8CF4632D9B879F80240A0F6304CE276F5641C3B756" +
                "995202CFA14E6BC777235D942B517CA963A5AFED7B595569877E3EA645A6D887" +
                "10975FD8BB213D58F50DBEE58A3716D7EFD2171734CD17E099513F3225366C6E" +
                "B91B0946FDC8F6745A15FF15EB71E7C2C6537C9E3CB95F38E9A571496247C9BF" +
                "118B2A361AD47B6122D4E098095105F8BE7C031C7CAD1044F44849B794379235" +
                "27AED01CC9E8605BA7431956E7ECF4B4A829A36C77FB8353AA8966EAE05219DC" +
                "0BA45F0E41F5B6B91D954148455055450203010001A370306E300C0603551D13" +
                "0101FF04023000300E0603551D0F0101FF040403020430304E0603551D230447" +
                "3045801419AA399804F91ED61132ECCBAAF578543F46972DA11BA41930173115" +
                "301306035504030C0C436F6E746F736F20526F6F74821000BEBBAC206CB54DF8" +
                "5376C2D4E0F19F300D06092A864886F70D01010B050003820101004629B34FF2" +
                "9208886EBFFDB0C54F935394F33FDC2F4B82C7EA1A18E0630905906D8DFF23F8" +
                "0D79848584453DA67D65FB8E4C8C5D53EBF864B1414B43AEB89276C6BF0C0870" +
                "6545E90C14F033ED8F47B6020E6FB87AFE0C41B3A5DCB5ECA881D519759D7EC1" +
                "093A21C38CAE7202EBFA99FE94121EF89944EA68BF92C8EF605D6C9EFCA0D849" +
                "D4E775AE0EABADAFB83019A323A0CE3544B9F877546C5B0043730E8F1342F332" +
                "5AF4683C73BADDFEF3518B0A4E015A10BC6178C10E3AF6A11430A9F102E59B5C" +
                "C44593D76E1A80D29932CF3A4F170F4516AD3666108BE1CE17FD74E266DE6B7E" +
                "2F7DB1215D41CE515B615167083E8EF4A3A54607EC7D14A0839A58").HexToByteArray();

            internal static byte[] RsaOaep2048_Sha256ParametersPfx = (
                "308209EE020103308209A806092A864886F70D010701A0820999048209953082" +
                "09913082056606092A864886F70D010701A0820557048205533082054F308205" +
                "4B060B2A864886F70D010C0A0102A08204FA308204F63028060A2A864886F70D" +
                "010C0103301A0414E5DDB04EF6C37EC860C8F2CD89330E233C6732A402020400" +
                "048204C8500596DD0681F7DAD596776609BB7B87D9D4163DB7FDDD4814771FC9" +
                "F022BF51CAEDF359C2FE122ACE3409EEAFD0231AB4F935FFCE3ED94FAB187A72" +
                "332AB58C59515412F10184AB793AD31A483BEB0D4D51757E9C069BFDF3803801" +
                "95DCC13ECA7DC5F8F89E25DF0AE364A88B6621707352AF566C8E3342ACD4AB0D" +
                "DF49651B755EF8837AE1690EF1FEB85AB3550EB618CE9FE74213F85806CF8A1A" +
                "3575F994F4F6CBFAA81DBAFE07ADEBDBDD526054749219B1745C246E66CCBEA0" +
                "2B8B00A68AF82BB2EBA9F57C58B0BD10D0CC646735928AE4AD5229192416658D" +
                "BEA6B46E7C161E1ADC228478BE8DAB604988A41408D95D352B7270B837102784" +
                "2C9FC6763770702AB4FA0F408CE67AA65650C8E399F311E32D7EA162304460AF" +
                "5719B36D2064A6A00B41B264B26939158C261FED9E176E9BF7C29E66C54B9614" +
                "0957315A65F04EE868B0A18018A763073C320E322C1AA9E767AA788B7B045626" +
                "D1C51DE5BA8C74A82D4FB7892AA6E628BA874A79BBF17E6764BE4D600DE83673" +
                "86A8D182E44E67EB74806606429420AD3A102821323A58718FA5CF45711AAE82" +
                "072DEDE5122CA829D9B29DE4B6AECE54D5C1C3E6732A943926A658E8BBAEF062" +
                "8128DDB63000F7958894F38167F80F1F7CB204C4EB7A7F34329E4D0D9DC5F75B" +
                "7619AA40A7D7FFAEDA481BB0720E9705F1FEAA93B2A30DF3A1E5BA5BED5F15C6" +
                "CD2B4E58EADF9C533F03FCEC1ACB4FEF64A11A190AAD270B33CB6E088AD6EF61" +
                "C16757FE5B5907D7C7AD7B0306521703B2363D3CB8EB4C2D53FB6941DA9A05DE" +
                "41854F18F8F7DADE6D615DEF8F8ADB3C94722B68EC811F4C26A59210005C89A1" +
                "5D3B8D440BD859919E3D83427A7A758CD3F01E5050275F3131A25624AF2ABA7B" +
                "98F38A0D5867EEC386F69A6B4A02FCD906D5DF1A70FADC0DEFF1421530395894" +
                "84C99689821BF286FC9074C5C42AA31790947B0B5D86BC775F764EFF6E42A514" +
                "603624FD8AF0DE2C1A76D9AE94807A69449B569979B1AA3FB5C9C33E12C9A6CC" +
                "4B4E02C4FBCC8DEF3D3A176ADA3BC21615E01596AD45E2479929570BE4752C38" +
                "CCEEBB5C1DAEDBB85E9529E5ED6A5D427B4ADF572C8F5BA75E538BEB3A8CE059" +
                "3FA05AB3779912CA314CD2DC4B801D5FFAE7F0FDDACD71EE4483FDC32F0CBB62" +
                "AE18D4DE521AD50EFD4E8D34586CE208D6CD74800A3FB0E869746D9D9F64129E" +
                "9308C0AD94B4A788C9B0F9E689F9B7B132368C701D79308D716347BAB6F66219" +
                "94B00B06A260B3F71B3477BFE363C4D2CFB26ECBD54B0D83F7F399A1A60ECE5F" +
                "7DEA546ACD40F94F01DFBA2814F760BD9FFC36F7C04104D1897B6928745DFB53" +
                "5F9B297EE6E0F1301402CE8413D2B4AF18024D356EFEA7444FA4ABA371F49706" +
                "FE050E2697F8A57D917A18BA1049BC6CB4CD5BB0E60313C06259A792B18E086C" +
                "47B7052D47BA524391F4B03BC33B175176E9E223C0ED55CCBAFB4C228BCFA862" +
                "BD1A28D38AEF249B7CD6A5D54B27A31CBD48081D26DD4F8ACEB54519969E98F0" +
                "E550B14F9BD123D0BC687B12FB5D4703AE971C6528435DEE49C8C919DF9FF27A" +
                "B1C28944694890C3CAF085ED60CEB25B2F77B71EFDB0E4462C42A2D4ECB9BB80" +
                "ED7E63724548295851BEC017A2FF662EED7DE5496C3A03B2E66722AB61437EFB" +
                "D1FD01A1D38FCF8D7088E44FCDDB46B04C3ED8B1A5957B414952EDA21F134BAF" +
                "B63DAFB9EA61D1E6C1B7FF07313E301706092A864886F70D010914310A1E0800" +
                "74006500730074302306092A864886F70D01091531160414A818F5D36906905E" +
                "2F6FA5BFF7829502427B9A573082042306092A864886F70D010706A082041430" +
                "8204100201003082040906092A864886F70D0107013028060A2A864886F70D01" +
                "0C0106301A04145D4F6CE75AFE72F271F5ABF7D4222AEC5F689C1E0202040080" +
                "8203D086644BB639A1348939D6F67CA12354E69D27B92D36C6DBF62D6565B58A" +
                "920992B8E5EB63078E73F4CA3FC66EAF9772FB35A7A6C514A9EEA4C84CDED3D5" +
                "0D7CE9B5105D5D43273995E83BFDC2E30769007712FBF949B4B9DB584508AC25" +
                "78ED447FB17B6D289601F1ACDDB5D5C9D511366F48E0B1A1D1F80F81957B0C71" +
                "1BCD3DE8D2B7C166DEEE93994B9845D6719376F0DE335F8130B8AA0478837809" +
                "D71BFC6430BCC4939C5E6329694E6E80097888007C80DD6733633470D5234728" +
                "4C17257AD610BE367A7383C1AD98F4499C0ED644EC840FF119EEA82F9B910E2B" +
                "53BC6C430D629696DA5AC7C24099E334A4DF8FB28B5A21CA05A72D3590307B5D" +
                "D0B68C6F76BAD7EC14D4732F0BEAE723D111A2DC31AFA06E55E6837FB8706289" +
                "45EBCF4B4AA6A6998E973BBA5EB853B3BEC95C81736149A8173ADCF319DB9CFB" +
                "C10BFF450B9C026ABB4A5887DFDD6F5B8F19D6D59ED2CBD3113F867414E21647" +
                "1E963878D7292B2749FABCC7982AB187A79BF74BD591D8523B4E2587C4714E94" +
                "FFE21193A28DD72C4C947889674A7EE0305433FDE0572D8B69881491EF03D963" +
                "9A0A9D8E65F234DF51428C662DAFDB68C7665DD917EBD1338AF36CA418B67632" +
                "B4C057D96E400E571C31F7FF75291013E8D092648109E1366F378F86C4311678" +
                "67932071DBEB100DF01F95E7B9262F06DC6393ADA419AB583AF78F94E7FB72D7" +
                "C4AB90F52D94E309152F07C349625AD67D7B922990E82CF79436AFB5DA314DE5" +
                "AF55FEF6E70442334E8574A7FAFBC5DA56A72E87ED56ABC84D6389BCBF0C54F0" +
                "8AFC841522F41A90CA11B5597E0F822B20177FCD15B74AAF975601DCCD628C0F" +
                "B33F5BA273D33CD8647CC681EA44F7DC452F2DC6750CA88E0D8FF1C2A9132D53" +
                "62F1A57F87E778969B88ADA6D9F8AF1C404D4AB9730E44BFAAF7080481C8B066" +
                "D4FC8B0C47AF90926D7A91C95C6FC5CAD275DBD65BD0835BE93DC30AD0ACC2A1" +
                "942F29A8D28C20D0F1220C9AD981896BA50E99718CAC9D48963DEF98BEFDDDE0" +
                "32FFD89374716F1946E922AF828533BCDBA7865DD50F190F526763D008B8F44F" +
                "EEC22E7C90F86E24EEABAC404FE49AF49787A31FB7FC2AEAA04C114F64A4DC68" +
                "ACEE48F7AF3939CED423A9664D25B906EDA43AC2EAE0E571B857199C9AAF4858" +
                "F4DFE1F83BBB15D2889FA7B1EE3CE768445007F5949C2F353C3A405EE7842E4B" +
                "6FC77F1A26042DD5D182C44A21AC6ACF1A43FC9498664A7A722E90AC297078E0" +
                "633BFB83D85DAB26CD9462364FF161BF9CF194103F6D91A6D28B7284293FB55B" +
                "3909151C85CFAEB5F5633E82D5364725EFA4CCFB7F62454399BEEDF1FDCD64A7" +
                "788822C1BE1A53B5B4FBF60039AF86B12E1B1E303D3021300906052B0E03021A" +
                "050004144E291DFD7C40F48494D521D4939815FF16938E220414732425C273E6" +
                "923C8F31AED55ED4BA1110E70AEA02020400").HexToByteArray();


            internal static byte[] RsaOaep2048_NoParametersCert = (
                "308203263082020EA003020102021000D90B171E0AE0BB7FC73832CF72AB8530" +
                "0D06092A864886F70D01010B050030173115301306035504030C0C436F6E746F" +
                "736F20526F6F74301E170D3139303432313232313734315A170D323930343232" +
                "3232313734315A30173115301306035504030C0C436F6E746F736F204C656166" +
                "30820120300B06092A864886F70D0101070382010F003082010A0282010100D2" +
                "12A4429CE6557322B0558BACB0DE8CABEC6FCA8E57C58125AFE1B91F9A79FA0E" +
                "59CD6D580F139A80378DF000AC6837E303B0C873A31ECE01C604CED463160DDC" +
                "8842447FB36C4013D3DF4039BF0C24DF522914AB6FE836E465114A80EF5C46D4" +
                "6ED19222A7C92294BE39966D814A24A996E9618A055A24C7A5E4271BC6BBB7AF" +
                "E29667CABDD4216CB286E8D95DEECA2AAC0D6E5D15B7BC6CA40BE80C0FED9830" +
                "C8D933F871244BB69A61F8CC10BA1D6BB3402B8542D79BFCE2E9BD7D91D4E118" +
                "F7E0181148EE8F6A665492B00E28F2A4F2481343A1BD063B86E11C67C74B0D49" +
                "8DB0AD1B84A013D6E89184BEFDF87CA9DA2DE2C62FCCF056B48953C696AB4502" +
                "03010001A370306E300C0603551D130101FF04023000300E0603551D0F0101FF" +
                "040403020430304E0603551D2304473045801419AA399804F91ED61132ECCBAA" +
                "F578543F46972DA11BA41930173115301306035504030C0C436F6E746F736F20" +
                "526F6F74821000BEBBAC206CB54DF85376C2D4E0F19F300D06092A864886F70D" +
                "01010B050003820101007522F8814F534DB7EA5D2A1DC9D0344D4C4877F291FB" +
                "7C4DB42D172C431A9C81267F114D8E4197AD90854522E4D55D4B211C7367A5FE" +
                "45478971642AC940A6D66FB6384AC437628B8679C2495FA35B3998B852DDF59B" +
                "3944FB15577B3FBD9F104A823A92E09AD7CFACD886B363EADF2478056D5CCB97" +
                "B233762DE31525E4FC6733E2CBDEBFC282E3F6B467BBD5932C89B92D0EA7A8DD" +
                "5C44AFCAD2980C6A24B8DF83E743CDC2DA3BB9188D01C8C8F596BBA5ACFE2E7A" +
                "21FE0F3D753D409AE08E34208C1AD07F70B73B97ED62DA6B8959232585BA5AAD" +
                "6AFFDD7D7298CD80D42B50AA7DC6A317BB7B2FE9EE95AB1FC8E3C1B2392D5750" +
                "45DC4D41362F8C83ABED").HexToByteArray();

            internal static byte[] RsaOaep2048_NoParametersPfx = (
                "308209BE0201033082097806092A864886F70D010701A0820969048209653082" +
                "09613082056606092A864886F70D010701A0820557048205533082054F308205" +
                "4B060B2A864886F70D010C0A0102A08204FA308204F63028060A2A864886F70D" +
                "010C0103301A0414993C8510854D2E0B2E6A93146230548B5EA6290602020400" +
                "048204C8BF5DCE211704D21DAA9C8F855657C889DB6F5B29ADA8F9DF721A159A" +
                "A3CB9899904C80B382EEC08767A6C7AA2B3FFFA13F41BF5FC8205DC70EE9F4B0" +
                "49D270D48EBD290157267DA0393F2746032B9D6F6C7594FD3495DFA6FE4D0D84" +
                "DF9E8091620DA4219BA65031CFCFF3F0024DEF64E47681806FEC30B625DBA88B" +
                "823B7120835E8DAAAA8A8E110B9DBA537426C50099B7824AF3451EFC406AAB22" +
                "C5B3698CE2A73C222392960699BA3F3AEE9F089C299DF80D720489424596D1B1" +
                "8538AC963F9C0E545ED6DDAD79DA3D401CAD32D5F88A1025E8491D76AB3ADD3B" +
                "36500B8DC4D3FE6F5E5A1819AA767C5BE7344C985821D88CAFE9914F6388C0E1" +
                "E715F4A1B7BA67A4687697FF8632756101C70CFE6ECD4711F1DF8694D1A7DAB1" +
                "6998C6D96BEF5DEAD2FB97E36B39ECF794DCD057544878FA55135F71ADAFE8BB" +
                "A0249C3208E6B56E310767984F9E405389F54732CD6FCF9B2650E1ADA2B2AB1E" +
                "298C549D09050A2C004F1D4F491070FC9B115B06EE2AE30939A836E38CF1FA3C" +
                "6B39373E61AA64F030E772F78A49C6CE6762A2B3F1D771B1FE17669906E26042" +
                "54158DB5D4B5E8965D267B99008FCB36A9B2A3EA83E534DC255BF1B0C79EDBA9" +
                "86623FDC395A59A2D95EB0A7E1D67899ED84F7F832B2775E4DA9ECBA63253E65" +
                "7E509442E4A794E93E2A24D1672939D69EDCA784684781A5B40A98F5E6E61F0B" +
                "72918E04847070C5C1E46E9156A3E25849B2F701E8B11FAE6A9C949DFD70F106" +
                "9751D120A93B6BBF8C14A7419E470479C32FCC3824CB3F56A1F93F0EEAC3A402" +
                "26CDBA64FACCCBA3E09888B920ABCB9E18D8F1E673C3D3B6475C1E406221571E" +
                "C92FFBCCB2FE9F110410196FEBC905C6926D25E717ACA85EC10B456F02B11561" +
                "467B152A74BD263F26C636FEF093D03AA8FD880E68692B3808C6BA7152A16A50" +
                "E82000A04760C86756FF0EA382EDA8571AD0181F98EC110B278493512F052B2B" +
                "95E0CF3798A9A54C688661746DAE65603BD43F990CEB0D0CB53DA780494E59F1" +
                "608E5B0C150C044CDE74870225A2F932D69D27ECDDC5EC3818D043ED363F27E8" +
                "F55D9B9396E439B8A3154B4E7F2BE160615220FA327253EFDEC5F19B6D51E0B9" +
                "1E513D17A6277066C866E59123019025994ECA8BAE4CDDD4021E2497A8C62DBF" +
                "A22711832CCFE68824D0D410933DCEB1A15AE4ABEA523D18E8417A161A508796" +
                "CDB9B746A97EF90A1E0597C82B148DDE0572A1AFE212670509FB4CC3FB6ED3D4" +
                "A85372B85EAD17EBE704D97B976D4D157F58DD5D6E3EA8C307CEC87B936E4950" +
                "BB136131001E937184B5973D4038DCE98C15E68C0951286AA8D676913A67F3EB" +
                "1291F96326EC0B8A0B4DB8DBD9E11390B0AF1772B247A3ABE572367B8A4B6FC0" +
                "36AF30130DA7D148540BD294438509CD879AAC3F62357C52914F58AD2B1BCF3F" +
                "0DF373314423F64B3687D692BDFE595F478BC6D879F4DBA572063B8648EB59D6" +
                "DF4B1BB6BD01B1B077195CAA63DD2C927850A0AF54A8711A929EA899716B8928" +
                "CB9E559AA4C89B06D3C8A9FB2C64F9A61763A532FBA9CD03F6A750560893B3D0" +
                "F32605C0C9D260FA5DF10EA9498A19FD1BA8C35FF988959CB76CB178DB3AE544" +
                "2AA3DFCAAF79595A85D8EA3BDC860A06A1176D95EF72E66898E06D3357CC434C" +
                "A2F46E9B57C5FDEBDD23B23186D9DF280FC05B326013FD96F44BD56E54B90A59" +
                "AA21865A576FE43C14DCD8DB313E301706092A864886F70D010914310A1E0800" +
                "74006500730074302306092A864886F70D01091531160414C8FCFD94B774494D" +
                "0B5A9187550D9562468B86C8308203F306092A864886F70D010706A08203E430" +
                "8203E0020100308203D906092A864886F70D0107013028060A2A864886F70D01" +
                "0C0106301A041495480C3C3F897F0F2763D2A8A89AF4F546485E800202040080" +
                "8203A0C9D6ACAC66EEE46EA9B5A3188623408790361FFEE1BBD67D1CEC14A22E" +
                "9F90D6F65CEADE7B1A25313ADAB523FFFC47FB41A953C8E4414E834FB51CEB1C" +
                "853F6FEC9FDCA5CED07BB8DD695D728DE4CC2B2EE40F243D4CDB6A20E7EDDC2B" +
                "32B1A570A9EC50360D5864774F0C04501FCEB9BFBAA3B76CD3855DAE042D3FDC" +
                "884F6673F77AFBFC9FD1344854E3E26996E6493C8C39C86077FD998B066E56B7" +
                "3F3CE9175CA6369780D92EA4F8DE7BE7918670B6BA044DCF5C39BAF85FC305B8" +
                "16541B335A6D528945DE6047ED8A30AC1B1292CD3E2050F5665E9DBA4816E053" +
                "6E45EB3C2D271C1D4FE4C44502E298B9958351EC282DD218701A7B06E85FC7FC" +
                "58EA1C69FC89326D1936A5E485DEF9DB036C76CA1BFB1E9C2EF926E0076BADC6" +
                "6164905F62CAD6AD8042C269F91AFF5508E3DB16CFDD966D8A88A03F446AEBBC" +
                "A56A17E525770A51D2595E7605DB4864FEE301B0376C91B416327C635FBEE114" +
                "3F0D686FBF77758831F4FC1984E9274D2D4C463A4EDE3864DB1825DAF10C3FAB" +
                "BDAAC3A0127226356DF027C91D01210F9D1E2449A25C73A882BDAC43936EC224" +
                "4CA0CF84097BE8381014564B64F012739F97B91AAD04AFEA1C7BFCCAD998A693" +
                "8204DA560F510B1EA3E4B99908F25058DCFB0032C51E6DDD7D6BB4A459833B5A" +
                "5295F727B7A306A9DD43A537242F527BCEDC633BE50A5699F4AC39E1DBE54867" +
                "654BC85E5FD43594FE00F1F86651E43C4D9BE7F86ADFD862BAD7801D68D124DB" +
                "4A90CB71DF4AF601A05E65D5F4D48B21F1E5151AAAFE899434E5DE503D217867" +
                "73F7DC27AC46B35E02AF1102BD39285E8DBA07E425947D2100DAA3A6CB263F18" +
                "32AFE5BC659A28C67DF8572BE6360825BBC837BEDA8B87EE10A38C3682FC1C2C" +
                "2ABFDD0AA11C4DA433B9AB1927F9B1955814EE044BCC095BDE4BAB05C245E211" +
                "7FCD5FB503CBC918B48E6D7F2E4D12693B01D8969956E92AC49E54A63879F113" +
                "52D8EFA103CE559D071F430067B05FBF68B5DC54FD25BE68CA054170E777B575" +
                "E4355674DB504F9480AA7ADDAEB2295C964689561ED6A909F5C279E6F1EF1CAE" +
                "94202A9073B1CDC5746E7EC47D60B927C661E39E305051CC77CE54EDF4F199DF" +
                "7C67596A9264F6AA7D6AFE955116BC100117E41F361A2C9F1DA8073B8D184037" +
                "EC9E23DF5F346E16D2B7F713FCE053CD815E783D13EA8CDEF7ACD014BB4B1D3E" +
                "EC46C75DD09C005ADA5F8DBEAC2B47D5B77F31A967B419FA57A61C661CD10903" +
                "D8CEBD9820ABEA3089F4387471FBEC87596FD036CC448F37B775538AE72C5AF1" +
                "9502F3303D3021300906052B0E03021A0500041484BFCFD2BED79AAAAD26AFCC" +
                "CC8F76FB6FD9D40804143A666EE41B085688C62A85EC7964C0FACF0512DF0202" +
                "0400").HexToByteArray();
        }
    }
}
