/* SPDX-License-Identifier: GPL-2.0 */

#ifndef DRM_KUNIT_EDID_H_
#define DRM_KUNIT_EDID_H_

/*
 * edid-decode (hex):
 *
 * 00 ff ff ff ff ff ff 00 31 d8 2a 00 00 00 00 00
 * 00 21 01 03 81 a0 5a 78 0a 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 01 01 01 01 01 01 01 01 01 01
 * 01 01 01 01 01 01 02 3a 80 18 71 38 2d 40 58 2c
 * 45 00 40 84 63 00 00 1e 00 00 00 fc 00 54 65 73
 * 74 20 45 44 49 44 0a 20 20 20 00 00 00 fd 00 32
 * 46 1e 46 0f 00 0a 20 20 20 20 20 20 00 00 00 10
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 ab
 *
 * ----------------
 *
 * Block 0, Base EDID:
 *   EDID Structure Version & Revision: 1.3
 *   Vendor & Product Identification:
 *     Manufacturer: LNX
 *     Model: 42
 *     Made in: 2023
 *   Basic Display Parameters & Features:
 *     Digital display
 *     DFP 1.x compatible TMDS
 *     Maximum image size: 160 cm x 90 cm
 *     Gamma: 2.20
 *     RGB color display
 *     First detailed timing is the preferred timing
 *   Color Characteristics:
 *     Red  : 0.0000, 0.0000
 *     Green: 0.0000, 0.0000
 *     Blue : 0.0000, 0.0000
 *     White: 0.0000, 0.0000
 *   Established Timings I & II: none
 *   Standard Timings: none
 *   Detailed Timing Descriptors:
 *     DTD 1:  1920x1080   60.000000 Hz  16:9     67.500 kHz    148.500000 MHz (1600 mm x 900 mm)
 *                  Hfront   88 Hsync  44 Hback  148 Hpol P
 *                  Vfront    4 Vsync   5 Vback   36 Vpol P
 *     Display Product Name: 'Test EDID'
 *     Display Range Limits:
 *       Monitor ranges (GTF): 50-70 Hz V, 30-70 kHz H, max dotclock 150 MHz
 *     Dummy Descriptor:
 * Checksum: 0xab
 *
 * ----------------
 *
 * edid-decode 1.30.0-5367
 * edid-decode SHA: 41ebf7135691 2025-05-01 10:19:22
 *
 * EDID conformity: PASS
 */
static const unsigned char test_edid_dvi_1080p[] = {
	0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x31, 0xd8, 0x2a, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x21, 0x01, 0x03, 0x81, 0xa0, 0x5a, 0x78,
	0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
	0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x02, 0x3a, 0x80, 0x18, 0x71, 0x38,
	0x2d, 0x40, 0x58, 0x2c, 0x45, 0x00, 0x40, 0x84, 0x63, 0x00, 0x00, 0x1e,
	0x00, 0x00, 0x00, 0xfc, 0x00, 0x54, 0x65, 0x73, 0x74, 0x20, 0x45, 0x44,
	0x49, 0x44, 0x0a, 0x20, 0x20, 0x20, 0x00, 0x00, 0x00, 0xfd, 0x00, 0x32,
	0x46, 0x1e, 0x46, 0x0f, 0x00, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
	0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xab
};

/*
 *
 * This edid is intentionally broken with the 100MHz limit. It's meant
 * to be used only with tests in unusual situations.
 *
 * edid-decode (hex):
 *
 * 00 ff ff ff ff ff ff 00 31 d8 2a 00 00 00 00 00
 * 00 21 01 03 81 a0 5a 78 02 00 00 00 00 00 00 00
 * 00 00 00 20 00 00 01 01 01 01 01 01 01 01 01 01
 * 01 01 01 01 01 01 02 3a 80 18 71 38 2d 40 58 2c
 * 45 00 40 84 63 00 00 1e 00 00 00 fc 00 54 65 73
 * 74 20 45 44 49 44 0a 20 20 20 00 00 00 fd 00 32
 * 46 1e 46 0f 00 0a 20 20 20 20 20 20 00 00 00 10
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 01 92
 *
 * 02 03 15 81 e3 05 00 20 41 10 e2 00 4a 67 03 0c
 * 00 12 34 00 14 00 00 00 00 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 10
 *
 * ----------------
 *
 * Block 0, Base EDID:
 *   EDID Structure Version & Revision: 1.3
 *   Vendor & Product Identification:
 *     Manufacturer: LNX
 *     Model: 42
 *     Made in: 2023
 *   Basic Display Parameters & Features:
 *     Digital display
 *     DFP 1.x compatible TMDS
 *     Maximum image size: 160 cm x 90 cm
 *     Gamma: 2.20
 *     Monochrome or grayscale display
 *     First detailed timing is the preferred timing
 *   Color Characteristics:
 *     Red  : 0.0000, 0.0000
 *     Green: 0.0000, 0.0000
 *     Blue : 0.0000, 0.0000
 *     White: 0.0000, 0.0000
 *   Established Timings I & II:
 *     DMT 0x04:   640x480    59.940476 Hz   4:3     31.469 kHz     25.175000 MHz
 *   Standard Timings: none
 *   Detailed Timing Descriptors:
 *     DTD 1:  1920x1080   60.000000 Hz  16:9     67.500 kHz    148.500000 MHz (1600 mm x 900 mm)
 *                  Hfront   88 Hsync  44 Hback  148 Hpol P
 *                  Vfront    4 Vsync   5 Vback   36 Vpol P
 *     Display Product Name: 'Test EDID'
 *     Display Range Limits:
 *       Monitor ranges (GTF): 50-70 Hz V, 30-70 kHz H, max dotclock 150 MHz
 *     Dummy Descriptor:
 *   Extension blocks: 1
 * Checksum: 0x92
 *
 * ----------------
 *
 * Block 1, CTA-861 Extension Block:
 *   Revision: 3
 *   Underscans IT Video Formats by default
 *   Native detailed modes: 1
 *   Colorimetry Data Block:
 *     sRGB
 *   Video Data Block:
 *     VIC  16:  1920x1080   60.000000 Hz  16:9     67.500 kHz    148.500000 MHz
 *   Video Capability Data Block:
 *     YCbCr quantization: No Data
 *     RGB quantization: Selectable (via AVI Q)
 *     PT scan behavior: No Data
 *     IT scan behavior: Always Underscanned
 *     CE scan behavior: Always Underscanned
 *   Vendor-Specific Data Block (HDMI), OUI 00-0C-03:
 *     Source physical address: 1.2.3.4
 *     Maximum TMDS clock: 100 MHz
 * Checksum: 0x10  Unused space in Extension Block: 106 bytes
 *
 * ----------------
 *
 * edid-decode 1.30.0-5367
 * edid-decode SHA: 41ebf7135691 2025-05-01 10:19:22
 *
 * Failures:
 *
 * EDID:
 *   CTA-861: The maximum HDMI TMDS clock is 100000 kHz, but one or more video timings go up to 148500 kHz.
 *
 * EDID conformity: FAIL
 */
static const unsigned char test_edid_hdmi_1080p_rgb_max_100mhz[] = {
	0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x31, 0xd8, 0x2a, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x21, 0x01, 0x03, 0x81, 0xa0, 0x5a, 0x78,
	0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20,
	0x00, 0x00, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
	0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x02, 0x3a, 0x80, 0x18, 0x71, 0x38,
	0x2d, 0x40, 0x58, 0x2c, 0x45, 0x00, 0x40, 0x84, 0x63, 0x00, 0x00, 0x1e,
	0x00, 0x00, 0x00, 0xfc, 0x00, 0x54, 0x65, 0x73, 0x74, 0x20, 0x45, 0x44,
	0x49, 0x44, 0x0a, 0x20, 0x20, 0x20, 0x00, 0x00, 0x00, 0xfd, 0x00, 0x32,
	0x46, 0x1e, 0x46, 0x0f, 0x00, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
	0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x92, 0x02, 0x03, 0x15, 0x81,
	0xe3, 0x05, 0x00, 0x20, 0x41, 0x10, 0xe2, 0x00, 0x4a, 0x67, 0x03, 0x0c,
	0x00, 0x12, 0x34, 0x00, 0x14, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x10
};

/*
 * edid-decode (hex):
 *
 * 00 ff ff ff ff ff ff 00 31 d8 2a 00 00 00 00 00
 * 00 21 01 03 81 a0 5a 78 02 00 00 00 00 00 00 00
 * 00 00 00 20 00 00 01 01 01 01 01 01 01 01 01 01
 * 01 01 01 01 01 01 02 3a 80 18 71 38 2d 40 58 2c
 * 45 00 40 84 63 00 00 1e 00 00 00 fc 00 54 65 73
 * 74 20 45 44 49 44 0a 20 20 20 00 00 00 fd 00 32
 * 46 1e 46 0f 00 0a 20 20 20 20 20 20 00 00 00 10
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 01 92
 *
 * 02 03 15 81 e3 05 00 20 41 10 e2 00 4a 67 03 0c
 * 00 12 34 00 28 00 00 00 00 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 fc
 *
 * ----------------
 *
 * Block 0, Base EDID:
 *   EDID Structure Version & Revision: 1.3
 *   Vendor & Product Identification:
 *     Manufacturer: LNX
 *     Model: 42
 *     Made in: 2023
 *   Basic Display Parameters & Features:
 *     Digital display
 *     DFP 1.x compatible TMDS
 *     Maximum image size: 160 cm x 90 cm
 *     Gamma: 2.20
 *     Monochrome or grayscale display
 *     First detailed timing is the preferred timing
 *   Color Characteristics:
 *     Red  : 0.0000, 0.0000
 *     Green: 0.0000, 0.0000
 *     Blue : 0.0000, 0.0000
 *     White: 0.0000, 0.0000
 *   Established Timings I & II:
 *     DMT 0x04:   640x480    59.940476 Hz   4:3     31.469 kHz     25.175000 MHz
 *   Standard Timings: none
 *   Detailed Timing Descriptors:
 *     DTD 1:  1920x1080   60.000000 Hz  16:9     67.500 kHz    148.500000 MHz (1600 mm x 900 mm)
 *                  Hfront   88 Hsync  44 Hback  148 Hpol P
 *                  Vfront    4 Vsync   5 Vback   36 Vpol P
 *     Display Product Name: 'Test EDID'
 *     Display Range Limits:
 *       Monitor ranges (GTF): 50-70 Hz V, 30-70 kHz H, max dotclock 150 MHz
 *     Dummy Descriptor:
 *   Extension blocks: 1
 * Checksum: 0x92
 *
 * ----------------
 *
 * Block 1, CTA-861 Extension Block:
 *   Revision: 3
 *   Underscans IT Video Formats by default
 *   Native detailed modes: 1
 *   Colorimetry Data Block:
 *     sRGB
 *   Video Data Block:
 *     VIC  16:  1920x1080   60.000000 Hz  16:9     67.500 kHz    148.500000 MHz
 *   Video Capability Data Block:
 *     YCbCr quantization: No Data
 *     RGB quantization: Selectable (via AVI Q)
 *     PT scan behavior: No Data
 *     IT scan behavior: Always Underscanned
 *     CE scan behavior: Always Underscanned
 *   Vendor-Specific Data Block (HDMI), OUI 00-0C-03:
 *     Source physical address: 1.2.3.4
 *     Maximum TMDS clock: 200 MHz
 * Checksum: 0xfc  Unused space in Extension Block: 106 bytes
 *
 * ----------------
 *
 * edid-decode 1.30.0-5367
 * edid-decode SHA: 41ebf7135691 2025-05-01 10:19:22
 *
 * EDID conformity: PASS
 */
static const unsigned char test_edid_hdmi_1080p_rgb_max_200mhz[] = {
	0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x31, 0xd8, 0x2a, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x21, 0x01, 0x03, 0x81, 0xa0, 0x5a, 0x78,
	0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20,
	0x00, 0x00, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
	0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x02, 0x3a, 0x80, 0x18, 0x71, 0x38,
	0x2d, 0x40, 0x58, 0x2c, 0x45, 0x00, 0x40, 0x84, 0x63, 0x00, 0x00, 0x1e,
	0x00, 0x00, 0x00, 0xfc, 0x00, 0x54, 0x65, 0x73, 0x74, 0x20, 0x45, 0x44,
	0x49, 0x44, 0x0a, 0x20, 0x20, 0x20, 0x00, 0x00, 0x00, 0xfd, 0x00, 0x32,
	0x46, 0x1e, 0x46, 0x0f, 0x00, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
	0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x92, 0x02, 0x03, 0x15, 0x81,
	0xe3, 0x05, 0x00, 0x20, 0x41, 0x10, 0xe2, 0x00, 0x4a, 0x67, 0x03, 0x0c,
	0x00, 0x12, 0x34, 0x00, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0xfc
};

/*
 * edid-decode (hex):
 *
 * 00 ff ff ff ff ff ff 00 31 d8 2a 00 00 00 00 00
 * 00 21 01 03 81 a0 5a 78 02 00 00 00 00 00 00 00
 * 00 00 00 20 00 00 01 01 01 01 01 01 01 01 01 01
 * 01 01 01 01 01 01 02 3a 80 18 71 38 2d 40 58 2c
 * 45 00 40 84 63 00 00 1e 00 00 00 fc 00 54 65 73
 * 74 20 45 44 49 44 0a 20 20 20 00 00 00 fd 00 32
 * 46 1e 46 0f 00 0a 20 20 20 20 20 20 00 00 00 10
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 01 92
 *
 * 02 03 15 81 e3 05 00 20 41 10 e2 00 4a 67 03 0c
 * 00 12 34 00 44 00 00 00 00 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 e0
 *
 * ----------------
 *
 * Block 0, Base EDID:
 *   EDID Structure Version & Revision: 1.3
 *   Vendor & Product Identification:
 *     Manufacturer: LNX
 *     Model: 42
 *     Made in: 2023
 *   Basic Display Parameters & Features:
 *     Digital display
 *     DFP 1.x compatible TMDS
 *     Maximum image size: 160 cm x 90 cm
 *     Gamma: 2.20
 *     Monochrome or grayscale display
 *     First detailed timing is the preferred timing
 *   Color Characteristics:
 *     Red  : 0.0000, 0.0000
 *     Green: 0.0000, 0.0000
 *     Blue : 0.0000, 0.0000
 *     White: 0.0000, 0.0000
 *   Established Timings I & II:
 *     DMT 0x04:   640x480    59.940476 Hz   4:3     31.469 kHz     25.175000 MHz
 *   Standard Timings: none
 *   Detailed Timing Descriptors:
 *     DTD 1:  1920x1080   60.000000 Hz  16:9     67.500 kHz    148.500000 MHz (1600 mm x 900 mm)
 *                  Hfront   88 Hsync  44 Hback  148 Hpol P
 *                  Vfront    4 Vsync   5 Vback   36 Vpol P
 *     Display Product Name: 'Test EDID'
 *     Display Range Limits:
 *       Monitor ranges (GTF): 50-70 Hz V, 30-70 kHz H, max dotclock 150 MHz
 *     Dummy Descriptor:
 *   Extension blocks: 1
 * Checksum: 0x92
 *
 * ----------------
 *
 * Block 1, CTA-861 Extension Block:
 *   Revision: 3
 *   Underscans IT Video Formats by default
 *   Native detailed modes: 1
 *   Colorimetry Data Block:
 *     sRGB
 *   Video Data Block:
 *     VIC  16:  1920x1080   60.000000 Hz  16:9     67.500 kHz    148.500000 MHz
 *   Video Capability Data Block:
 *     YCbCr quantization: No Data
 *     RGB quantization: Selectable (via AVI Q)
 *     PT scan behavior: No Data
 *     IT scan behavior: Always Underscanned
 *     CE scan behavior: Always Underscanned
 *   Vendor-Specific Data Block (HDMI), OUI 00-0C-03:
 *     Source physical address: 1.2.3.4
 *     Maximum TMDS clock: 340 MHz
 * Checksum: 0xe0  Unused space in Extension Block: 106 bytes
 *
 * ----------------
 *
 * edid-decode 1.30.0-5367
 * edid-decode SHA: 41ebf7135691 2025-05-01 10:19:22
 *
 * EDID conformity: PASS
 */
static const unsigned char test_edid_hdmi_1080p_rgb_max_340mhz[] = {
	0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x31, 0xd8, 0x2a, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x21, 0x01, 0x03, 0x81, 0xa0, 0x5a, 0x78,
	0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20,
	0x00, 0x00, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
	0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x02, 0x3a, 0x80, 0x18, 0x71, 0x38,
	0x2d, 0x40, 0x58, 0x2c, 0x45, 0x00, 0x40, 0x84, 0x63, 0x00, 0x00, 0x1e,
	0x00, 0x00, 0x00, 0xfc, 0x00, 0x54, 0x65, 0x73, 0x74, 0x20, 0x45, 0x44,
	0x49, 0x44, 0x0a, 0x20, 0x20, 0x20, 0x00, 0x00, 0x00, 0xfd, 0x00, 0x32,
	0x46, 0x1e, 0x46, 0x0f, 0x00, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
	0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x92, 0x02, 0x03, 0x15, 0x81,
	0xe3, 0x05, 0x00, 0x20, 0x41, 0x10, 0xe2, 0x00, 0x4a, 0x67, 0x03, 0x0c,
	0x00, 0x12, 0x34, 0x00, 0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0xe0
};

/*
 * edid-decode (hex):
 *
 * 00 ff ff ff ff ff ff 00 31 d8 2a 00 00 00 00 00
 * 00 21 01 03 81 a0 5a 78 1a 00 00 00 00 00 00 00
 * 00 00 00 20 00 00 01 01 01 01 01 01 01 01 01 01
 * 01 01 01 01 01 01 02 3a 80 18 71 38 2d 40 58 2c
 * 45 00 40 84 63 00 00 1e 00 00 00 fc 00 54 65 73
 * 74 20 45 44 49 44 0a 20 20 20 00 00 00 fd 00 32
 * 46 1e 46 0f 00 0a 20 20 20 20 20 20 00 00 00 10
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 01 7a
 *
 * 02 03 15 b1 e3 05 00 20 41 10 e2 00 ca 67 03 0c
 * 00 12 34 78 28 00 00 00 00 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 d4
 *
 * ----------------
 *
 * Block 0, Base EDID:
 *   EDID Structure Version & Revision: 1.3
 *   Vendor & Product Identification:
 *     Manufacturer: LNX
 *     Model: 42
 *     Made in: 2023
 *   Basic Display Parameters & Features:
 *     Digital display
 *     DFP 1.x compatible TMDS
 *     Maximum image size: 160 cm x 90 cm
 *     Gamma: 2.20
 *     Undefined display color type
 *     First detailed timing is the preferred timing
 *   Color Characteristics:
 *     Red  : 0.0000, 0.0000
 *     Green: 0.0000, 0.0000
 *     Blue : 0.0000, 0.0000
 *     White: 0.0000, 0.0000
 *   Established Timings I & II:
 *     DMT 0x04:   640x480    59.940476 Hz   4:3     31.469 kHz     25.175000 MHz
 *   Standard Timings: none
 *   Detailed Timing Descriptors:
 *     DTD 1:  1920x1080   60.000000 Hz  16:9     67.500 kHz    148.500000 MHz (1600 mm x 900 mm)
 *                  Hfront   88 Hsync  44 Hback  148 Hpol P
 *                  Vfront    4 Vsync   5 Vback   36 Vpol P
 *     Display Product Name: 'Test EDID'
 *     Display Range Limits:
 *       Monitor ranges (GTF): 50-70 Hz V, 30-70 kHz H, max dotclock 150 MHz
 *     Dummy Descriptor:
 *   Extension blocks: 1
 * Checksum: 0x7a
 *
 * ----------------
 *
 * Block 1, CTA-861 Extension Block:
 *   Revision: 3
 *   Underscans IT Video Formats by default
 *   Supports YCbCr 4:4:4
 *   Supports YCbCr 4:2:2
 *   Native detailed modes: 1
 *   Colorimetry Data Block:
 *     sRGB
 *   Video Data Block:
 *     VIC  16:  1920x1080   60.000000 Hz  16:9     67.500 kHz    148.500000 MHz
 *   Video Capability Data Block:
 *     YCbCr quantization: Selectable (via AVI YQ)
 *     RGB quantization: Selectable (via AVI Q)
 *     PT scan behavior: No Data
 *     IT scan behavior: Always Underscanned
 *     CE scan behavior: Always Underscanned
 *   Vendor-Specific Data Block (HDMI), OUI 00-0C-03:
 *     Source physical address: 1.2.3.4
 *     DC_48bit
 *     DC_36bit
 *     DC_30bit
 *     DC_Y444
 *     Maximum TMDS clock: 200 MHz
 * Checksum: 0xd4  Unused space in Extension Block: 106 bytes
 *
 * ----------------
 *
 * edid-decode 1.30.0-5367
 * edid-decode SHA: 41ebf7135691 2025-05-01 10:19:22
 *
 * EDID conformity: PASS
 */
static const unsigned char test_edid_hdmi_1080p_rgb_yuv_dc_max_200mhz[] = {
	0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x31, 0xd8, 0x2a, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x21, 0x01, 0x03, 0x81, 0xa0, 0x5a, 0x78,
	0x1a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20,
	0x00, 0x00, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
	0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x02, 0x3a, 0x80, 0x18, 0x71, 0x38,
	0x2d, 0x40, 0x58, 0x2c, 0x45, 0x00, 0x40, 0x84, 0x63, 0x00, 0x00, 0x1e,
	0x00, 0x00, 0x00, 0xfc, 0x00, 0x54, 0x65, 0x73, 0x74, 0x20, 0x45, 0x44,
	0x49, 0x44, 0x0a, 0x20, 0x20, 0x20, 0x00, 0x00, 0x00, 0xfd, 0x00, 0x32,
	0x46, 0x1e, 0x46, 0x0f, 0x00, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
	0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x7a, 0x02, 0x03, 0x15, 0xb1,
	0xe3, 0x05, 0x00, 0x20, 0x41, 0x10, 0xe2, 0x00, 0xca, 0x67, 0x03, 0x0c,
	0x00, 0x12, 0x34, 0x78, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0xd4
};

/*
 * edid-decode (hex):
 *
 * 00 ff ff ff ff ff ff 00 31 d8 2a 00 00 00 00 00
 * 00 21 01 03 81 a0 5a 78 0a 00 00 00 00 00 00 00
 * 00 00 00 20 00 00 01 01 01 01 01 01 01 01 01 01
 * 01 01 01 01 01 01 02 3a 80 18 71 38 2d 40 58 2c
 * 45 00 40 84 63 00 00 1e 00 00 00 fc 00 54 65 73
 * 74 20 45 44 49 44 0a 20 20 20 00 00 00 fd 00 32
 * 46 1e 46 0f 00 0a 20 20 20 20 20 20 00 00 00 10
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 01 8a
 *
 * 02 03 15 b1 e3 05 00 20 41 10 e2 00 ca 67 03 0c
 * 00 12 34 78 44 00 00 00 00 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 b8
 *
 * ----------------
 *
 * Block 0, Base EDID:
 *   EDID Structure Version & Revision: 1.3
 *   Vendor & Product Identification:
 *     Manufacturer: LNX
 *     Model: 42
 *     Made in: 2023
 *   Basic Display Parameters & Features:
 *     Digital display
 *     DFP 1.x compatible TMDS
 *     Maximum image size: 160 cm x 90 cm
 *     Gamma: 2.20
 *     RGB color display
 *     First detailed timing is the preferred timing
 *   Color Characteristics:
 *     Red  : 0.0000, 0.0000
 *     Green: 0.0000, 0.0000
 *     Blue : 0.0000, 0.0000
 *     White: 0.0000, 0.0000
 *   Established Timings I & II:
 *     DMT 0x04:   640x480    59.940476 Hz   4:3     31.469 kHz     25.175000 MHz
 *   Standard Timings: none
 *   Detailed Timing Descriptors:
 *     DTD 1:  1920x1080   60.000000 Hz  16:9     67.500 kHz    148.500000 MHz (1600 mm x 900 mm)
 *                  Hfront   88 Hsync  44 Hback  148 Hpol P
 *                  Vfront    4 Vsync   5 Vback   36 Vpol P
 *     Display Product Name: 'Test EDID'
 *     Display Range Limits:
 *       Monitor ranges (GTF): 50-70 Hz V, 30-70 kHz H, max dotclock 150 MHz
 *     Dummy Descriptor:
 *   Extension blocks: 1
 * Checksum: 0x8a
 *
 * ----------------
 *
 * Block 1, CTA-861 Extension Block:
 *   Revision: 3
 *   Underscans IT Video Formats by default
 *   Supports YCbCr 4:4:4
 *   Supports YCbCr 4:2:2
 *   Native detailed modes: 1
 *   Colorimetry Data Block:
 *     sRGB
 *   Video Data Block:
 *     VIC  16:  1920x1080   60.000000 Hz  16:9     67.500 kHz    148.500000 MHz
 *   Video Capability Data Block:
 *     YCbCr quantization: Selectable (via AVI YQ)
 *     RGB quantization: Selectable (via AVI Q)
 *     PT scan behavior: No Data
 *     IT scan behavior: Always Underscanned
 *     CE scan behavior: Always Underscanned
 *   Vendor-Specific Data Block (HDMI), OUI 00-0C-03:
 *     Source physical address: 1.2.3.4
 *     DC_48bit
 *     DC_36bit
 *     DC_30bit
 *     DC_Y444
 *     Maximum TMDS clock: 340 MHz
 * Checksum: 0xb8  Unused space in Extension Block: 106 bytes
 *
 * ----------------
 *
 * edid-decode 1.30.0-5367
 * edid-decode SHA: 41ebf7135691 2025-05-01 10:19:22
 *
 * EDID conformity: PASS
 */
static const unsigned char test_edid_hdmi_1080p_rgb_yuv_dc_max_340mhz[] = {
	0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x31, 0xd8, 0x2a, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x21, 0x01, 0x03, 0x81, 0xa0, 0x5a, 0x78,
	0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x20,
	0x00, 0x00, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
	0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x02, 0x3a, 0x80, 0x18, 0x71, 0x38,
	0x2d, 0x40, 0x58, 0x2c, 0x45, 0x00, 0x40, 0x84, 0x63, 0x00, 0x00, 0x1e,
	0x00, 0x00, 0x00, 0xfc, 0x00, 0x54, 0x65, 0x73, 0x74, 0x20, 0x45, 0x44,
	0x49, 0x44, 0x0a, 0x20, 0x20, 0x20, 0x00, 0x00, 0x00, 0xfd, 0x00, 0x32,
	0x46, 0x1e, 0x46, 0x0f, 0x00, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
	0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x8a, 0x02, 0x03, 0x15, 0xb1,
	0xe3, 0x05, 0x00, 0x20, 0x41, 0x10, 0xe2, 0x00, 0xca, 0x67, 0x03, 0x0c,
	0x00, 0x12, 0x34, 0x78, 0x44, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0xb8
};

/*
 * Max resolution:
 * - 1920x1080@60Hz with RGB, YUV444, YUV422
 * - 3840x2160@30Hz with YUV420 only
 * Max BPC:         16 for all modes
 * Max TMDS clock:  200 MHz
 *
 * edid-decode (hex):
 *
 * 00 ff ff ff ff ff ff 00 31 d8 34 00 00 00 00 00
 * ff 23 01 03 80 60 36 78 0f ee 91 a3 54 4c 99 26
 * 0f 50 54 20 00 00 01 01 01 01 01 01 01 01 01 01
 * 01 01 01 01 01 01 02 3a 80 18 71 38 2d 40 58 2c
 * 45 00 c0 1c 32 00 00 1e 00 00 00 fc 00 54 65 73
 * 74 20 45 44 49 44 0a 20 20 20 00 00 00 fd 00 18
 * 55 18 5e 11 00 0a 20 20 20 20 20 20 00 00 00 10
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 01 bb
 *
 * 02 03 29 31 42 90 5f 6c 03 0c 00 10 00 78 28 20
 * 00 00 01 03 6d d8 5d c4 01 28 80 07 00 00 00 00
 * 00 00 e3 0f 00 00 e2 0e 5f 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 ca
 *
 * ----------------
 *
 * Block 0, Base EDID:
 *   EDID Structure Version & Revision: 1.3
 *   Vendor & Product Identification:
 *     Manufacturer: LNX
 *     Model: 52
 *     Model year: 2025
 *   Basic Display Parameters & Features:
 *     Digital display
 *     Maximum image size: 96 cm x 54 cm
 *     Gamma: 2.20
 *     RGB color display
 *     Default (sRGB) color space is primary color space
 *     First detailed timing is the preferred timing
 *     Supports GTF timings within operating range
 *   Color Characteristics:
 *     Red  : 0.6396, 0.3300
 *     Green: 0.2998, 0.5996
 *     Blue : 0.1503, 0.0595
 *     White: 0.3125, 0.3291
 *   Established Timings I & II:
 *     DMT 0x04:   640x480    59.940476 Hz   4:3     31.469 kHz     25.175000 MHz
 *   Standard Timings: none
 *   Detailed Timing Descriptors:
 *     DTD 1:  1920x1080   60.000000 Hz  16:9     67.500 kHz    148.500000 MHz (960 mm x 540 mm)
 *                  Hfront   88 Hsync  44 Hback  148 Hpol P
 *                  Vfront    4 Vsync   5 Vback   36 Vpol P
 *     Display Product Name: 'Test EDID'
 *     Display Range Limits:
 *       Monitor ranges (GTF): 24-85 Hz V, 24-94 kHz H, max dotclock 170 MHz
 *     Dummy Descriptor:
 *   Extension blocks: 1
 * Checksum: 0xbb
 *
 * ----------------
 *
 * Block 1, CTA-861 Extension Block:
 *   Revision: 3
 *   Supports YCbCr 4:4:4
 *   Supports YCbCr 4:2:2
 *   Native detailed modes: 1
 *   Video Data Block:
 *     VIC  16:  1920x1080   60.000000 Hz  16:9     67.500 kHz    148.500000 MHz (native)
 *     VIC  95:  3840x2160   30.000000 Hz  16:9     67.500 kHz    297.000000 MHz
 *   Vendor-Specific Data Block (HDMI), OUI 00-0C-03:
 *     Source physical address: 1.0.0.0
 *     DC_48bit
 *     DC_36bit
 *     DC_30bit
 *     DC_Y444
 *     Maximum TMDS clock: 200 MHz
 *     Extended HDMI video details:
 *   Vendor-Specific Data Block (HDMI Forum), OUI C4-5D-D8:
 *     Version: 1
 *     Maximum TMDS Character Rate: 200 MHz
 *     SCDC Present
 *     Supports 16-bits/component Deep Color 4:2:0 Pixel Encoding
 *     Supports 12-bits/component Deep Color 4:2:0 Pixel Encoding
 *     Supports 10-bits/component Deep Color 4:2:0 Pixel Encoding
 *   YCbCr 4:2:0 Capability Map Data Block:
 *     Empty Capability Map
 *   YCbCr 4:2:0 Video Data Block:
 *     VIC  95:  3840x2160   30.000000 Hz  16:9     67.500 kHz    297.000000 MHz
 * Checksum: 0xca
 */
static const unsigned char test_edid_hdmi_1080p_rgb_yuv_4k_yuv420_dc_max_200mhz[] = {
	0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x31, 0xd8, 0x34, 0x00,
	0x00, 0x00, 0x00, 0x00, 0xff, 0x23, 0x01, 0x03, 0x80, 0x60, 0x36, 0x78,
	0x0f, 0xee, 0x91, 0xa3, 0x54, 0x4c, 0x99, 0x26, 0x0f, 0x50, 0x54, 0x20,
	0x00, 0x00, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
	0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x02, 0x3a, 0x80, 0x18, 0x71, 0x38,
	0x2d, 0x40, 0x58, 0x2c, 0x45, 0x00, 0xc0, 0x1c, 0x32, 0x00, 0x00, 0x1e,
	0x00, 0x00, 0x00, 0xfc, 0x00, 0x54, 0x65, 0x73, 0x74, 0x20, 0x45, 0x44,
	0x49, 0x44, 0x0a, 0x20, 0x20, 0x20, 0x00, 0x00, 0x00, 0xfd, 0x00, 0x18,
	0x55, 0x18, 0x5e, 0x11, 0x00, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
	0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xbb, 0x02, 0x03, 0x29, 0x31,
	0x42, 0x90, 0x5f, 0x6c, 0x03, 0x0c, 0x00, 0x10, 0x00, 0x78, 0x28, 0x20,
	0x00, 0x00, 0x01, 0x03, 0x6d, 0xd8, 0x5d, 0xc4, 0x01, 0x28, 0x80, 0x07,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xe3, 0x0f, 0x00, 0x00, 0xe2, 0x0e,
	0x5f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0xca
};

/*
 * Max resolution: 3840x2160@30Hz with RGB, YUV444, YUV422, YUV420
 * Max BPC:        16 for all modes
 * Max TMDS clock: 340 MHz
 *
 * edid-decode (hex):
 *
 * 00 ff ff ff ff ff ff 00 31 d8 34 00 00 00 00 00
 * ff 23 01 03 80 60 36 78 0f ee 91 a3 54 4c 99 26
 * 0f 50 54 20 00 00 01 01 01 01 01 01 01 01 01 01
 * 01 01 01 01 01 01 04 74 00 30 f2 70 5a 80 b0 58
 * 8a 00 40 84 63 00 00 1e 00 00 00 fc 00 54 65 73
 * 74 20 45 44 49 44 0a 20 20 20 00 00 00 fd 00 18
 * 55 18 5e 22 00 0a 20 20 20 20 20 20 00 00 00 10
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 01 ce
 *
 * 02 03 27 31 41 5f 6c 03 0c 00 10 00 78 44 20 00
 * 00 01 03 6d d8 5d c4 01 44 80 07 00 00 00 00 00
 * 00 e3 0f 01 00 e1 0e 00 00 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00
 * 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 84
 *
 * ----------------
 *
 * Block 0, Base EDID:
 *   EDID Structure Version & Revision: 1.3
 *   Vendor & Product Identification:
 *     Manufacturer: LNX
 *     Model: 52
 *     Model year: 2025
 *   Basic Display Parameters & Features:
 *     Digital display
 *     Maximum image size: 96 cm x 54 cm
 *     Gamma: 2.20
 *     RGB color display
 *     Default (sRGB) color space is primary color space
 *     First detailed timing is the preferred timing
 *     Supports GTF timings within operating range
 *   Color Characteristics:
 *     Red  : 0.6396, 0.3300
 *     Green: 0.2998, 0.5996
 *     Blue : 0.1503, 0.0595
 *     White: 0.3125, 0.3291
 *   Established Timings I & II:
 *     DMT 0x04:   640x480    59.940476 Hz   4:3     31.469 kHz     25.175000 MHz
 *   Standard Timings: none
 *   Detailed Timing Descriptors:
 *     DTD 1:  3840x2160   30.000000 Hz  16:9     67.500 kHz    297.000000 MHz (1600 mm x 900 mm)
 *                  Hfront  176 Hsync  88 Hback  296 Hpol P
 *                  Vfront    8 Vsync  10 Vback   72 Vpol P
 *     Display Product Name: 'Test EDID'
 *     Display Range Limits:
 *       Monitor ranges (GTF): 24-85 Hz V, 24-94 kHz H, max dotclock 340 MHz
 *     Dummy Descriptor:
 *   Extension blocks: 1
 * Checksum: 0xce
 *
 * ----------------
 *
 * Block 1, CTA-861 Extension Block:
 *   Revision: 3
 *   Supports YCbCr 4:4:4
 *   Supports YCbCr 4:2:2
 *   Native detailed modes: 1
 *   Video Data Block:
 *     VIC  95:  3840x2160   30.000000 Hz  16:9     67.500 kHz    297.000000 MHz
 *   Vendor-Specific Data Block (HDMI), OUI 00-0C-03:
 *     Source physical address: 1.0.0.0
 *     DC_48bit
 *     DC_36bit
 *     DC_30bit
 *     DC_Y444
 *     Maximum TMDS clock: 340 MHz
 *     Extended HDMI video details:
 *   Vendor-Specific Data Block (HDMI Forum), OUI C4-5D-D8:
 *     Version: 1
 *     Maximum TMDS Character Rate: 340 MHz
 *     SCDC Present
 *     Supports 16-bits/component Deep Color 4:2:0 Pixel Encoding
 *     Supports 12-bits/component Deep Color 4:2:0 Pixel Encoding
 *     Supports 10-bits/component Deep Color 4:2:0 Pixel Encoding
 *   YCbCr 4:2:0 Capability Map Data Block:
 *     VIC  95:  3840x2160   30.000000 Hz  16:9     67.500 kHz    297.000000 MHz
 *   YCbCr 4:2:0 Video Data Block:
 * Checksum: 0x84
 */
static const unsigned char test_edid_hdmi_4k_rgb_yuv420_dc_max_340mhz[] = {
	0x00, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0x00, 0x31, 0xd8, 0x34, 0x00,
	0x00, 0x00, 0x00, 0x00, 0xff, 0x23, 0x01, 0x03, 0x80, 0x60, 0x36, 0x78,
	0x0f, 0xee, 0x91, 0xa3, 0x54, 0x4c, 0x99, 0x26, 0x0f, 0x50, 0x54, 0x20,
	0x00, 0x00, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x01,
	0x01, 0x01, 0x01, 0x01, 0x01, 0x01, 0x04, 0x74, 0x00, 0x30, 0xf2, 0x70,
	0x5a, 0x80, 0xb0, 0x58, 0x8a, 0x00, 0x40, 0x84, 0x63, 0x00, 0x00, 0x1e,
	0x00, 0x00, 0x00, 0xfc, 0x00, 0x54, 0x65, 0x73, 0x74, 0x20, 0x45, 0x44,
	0x49, 0x44, 0x0a, 0x20, 0x20, 0x20, 0x00, 0x00, 0x00, 0xfd, 0x00, 0x18,
	0x55, 0x18, 0x5e, 0x22, 0x00, 0x0a, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
	0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0xce, 0x02, 0x03, 0x27, 0x31,
	0x41, 0x5f, 0x6c, 0x03, 0x0c, 0x00, 0x10, 0x00, 0x78, 0x44, 0x20, 0x00,
	0x00, 0x01, 0x03, 0x6d, 0xd8, 0x5d, 0xc4, 0x01, 0x44, 0x80, 0x07, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xe3, 0x0f, 0x01, 0x00, 0xe1, 0x0e, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x84
};

#endif // DRM_KUNIT_EDID_H_
